<?php
/**
 * Template variables:
 *
 * @var $controllers        array settings as array
 * @var $is_builder_mode    bool Editor Mode status
 * @var $order
 * @var $guest_user_verify boolean
 * @var $needs_payment boolean
 */

do_action( 'before_woocommerce_pay' );

$order_id = absint( $order->get_id() );


try {
	// Logged out customer does not have permission to pay for this order.
	if ( ! current_user_can( 'pay_for_order', $order_id ) && ! is_user_logged_in() ) {
		wc_print_notice( esc_html__( 'Please log in to your account below to continue to the payment form.', 'shopbuilder-pro' ), 'notice' );
		woocommerce_login_form(
			[
				'redirect' => $order->get_checkout_payment_url(),
			]
		);
		return;
	}

	// Add notice if logged in customer is trying to pay for guest order.
	if ( ! $order->get_user_id() && is_user_logged_in() ) {
		// If order has does not have same billing email then current logged in user then show warning.
		if ( $order->get_billing_email() !== wp_get_current_user()->user_email ) {
			wc_print_notice( __( 'You are paying for a guest order. Please continue with payment only if you recognize this order.', 'shopbuilder-pro' ), 'error' );
		}
	}

	// Logged in customer trying to pay for someone else's order.
	if ( ! current_user_can( 'pay_for_order', $order_id ) ) {
		throw new Exception( __( 'This order cannot be paid for. Please contact us if you need assistance.', 'shopbuilder-pro' ) );
	}

	// Does not need payment.
	if ( $needs_payment ) {
		// * translators: %s: order status */
		throw new Exception( sprintf( __( 'This order&rsquo;s status is &ldquo;%s&rdquo;&mdash;it cannot be paid for. Please contact us if you need assistance.', 'shopbuilder-pro' ), wc_get_order_status_name( $order->get_status() ) ) );
	}

	// Ensure order items are still stocked if paying for a failed order. Pending orders do not need this check because stock is held.
	if ( ! $order->has_status( wc_get_is_pending_statuses() ) ) {
		$quantities = [];

		foreach ( $order->get_items() as $item_key => $item ) {
			if ( $item && is_callable( [ $item, 'get_product' ] ) ) {
				$product = $item->get_product();

				if ( ! $product ) {
					continue;
				}

				$quantities[ $product->get_stock_managed_by_id() ] = isset( $quantities[ $product->get_stock_managed_by_id() ] ) ? $quantities[ $product->get_stock_managed_by_id() ] + $item->get_quantity() : $item->get_quantity();
			}
		}

		// Stock levels may already have been adjusted for this order (in which case we don't need to worry about checking for low stock).
		if ( ! $order->get_data_store()->get_stock_reduced( $order->get_id() ) ) {
			foreach ( $order->get_items() as $item_key => $item ) {
				if ( $item && is_callable( [ $item, 'get_product' ] ) ) {
					$product = $item->get_product();
					if ( ! $product ) {
						continue;
					}

					if ( ! apply_filters( 'woocommerce_pay_order_product_in_stock', $product->is_in_stock(), $product, $order ) ) {
						/* translators: %s: product name */
						throw new Exception( sprintf( __( 'Sorry, "%s" is no longer in stock so this order cannot be paid for. We apologize for any inconvenience caused.', 'shopbuilder-pro' ), $product->get_name() ) );
					}

					// We only need to check products managing stock, with a limited stock qty.
					if ( ! $product->managing_stock() || $product->backorders_allowed() ) {
						continue;
					}

					// Check stock based on all items in the cart and consider any held stock within pending orders.
					$held_stock     = wc_get_held_stock_quantity( $product, $order->get_id() );
					$required_stock = $quantities[ $product->get_stock_managed_by_id() ];

					if ( ! apply_filters( 'woocommerce_pay_order_product_has_enough_stock', ( $product->get_stock_quantity() >= ( $held_stock + $required_stock ) ), $product, $order ) ) {
						/* translators: 1: product name 2: quantity in stock */
						throw new Exception( sprintf( __( 'Sorry, we do not have enough "%1$s" in stock to fulfill your order (%2$s available). We apologize for any inconvenience caused.', 'shopbuilder-pro' ), $product->get_name(), wc_format_stock_quantity_for_display( $product->get_stock_quantity() - $held_stock, $product ) ) );
					}
				}
			}
		}
	}

	// If we cannot match the order with the current user, ask that they verify their email address.
	if ( $guest_user_verify ) {
		wc_get_template(
			'checkout/form-verify-email.php',
			[
				'failed_submission' => ! empty( $_POST['email'] ), // phpcs:ignore WordPress.Security.NonceVerification.Missing
				'verify_url'        => $order->get_checkout_payment_url(),
			]
		);
		return;
	}
	if ( WC()->customer ) {
		WC()->customer->set_props(
			[
				'billing_country'  => $order->get_billing_country() ? $order->get_billing_country() : null,
				'billing_state'    => $order->get_billing_state() ? $order->get_billing_state() : null,
				'billing_postcode' => $order->get_billing_postcode() ? $order->get_billing_postcode() : null,
			]
		);
		WC()->customer->save();
	}


	$available_gateways = WC()->payment_gateways->get_available_payment_gateways();

	if ( count( $available_gateways ) ) {
		current( $available_gateways )->set_current();
	}

	/**
	 * Allows the text of the submit button on the Pay for Order page to be changed.
	 *
	 * @param string $text The text of the button.
	 *
	 * @since 3.0.2
	 */
	$order_button_text = apply_filters( 'woocommerce_pay_order_button_text', __( 'Pay for orders', 'shopbuilder-pro' ) );

	/**
	 * Triggered right before the Pay for Order form, after validation of the order and customer.
	 *
	 * @param WC_Order $order              The order that is being paid for.
	 * @param string   $order_button_text  The text for the submit button.
	 * @param array    $available_gateways All available gateways.
	 *
	 * @since 6.6
	 */
	do_action( 'before_woocommerce_pay_form', $order, $order_button_text, $available_gateways );

	wc_get_template(
		'checkout/form-pay.php',
		[
			'order'              => $order,
			'available_gateways' => $available_gateways,
			'order_button_text'  => $order_button_text,
		]
	);

} catch ( Exception $e ) {
	wc_print_notice( $e->getMessage(), 'error' );
}


do_action( 'after_woocommerce_pay' );
