<?php
/**
 * Field renderer for Variation Swatches.
 *
 * @package RadiusTheme\SBPRO
 */

namespace RadiusTheme\SBPRO\Modules\VariationSwatches;

/**
 * Main class.
 */
class Field {

	/**
	 * Render a list of meta fields.
	 *
	 * @param array $fields Array of field configuration.
	 * @return void
	 */
	public static function generate_fields( $fields ) {
		$fields = apply_filters( 'rtsb_vs_meta_fields', $fields );

		if ( empty( $fields ) ) {
			return;
		}

		foreach ( $fields as $field ) {
			$field = wp_parse_args(
				$field,
				[
					'id'    => '',
					'class' => '',
					'value' => '',
				]
			);
			$field = apply_filters( 'rtsb_vs_meta_field', $field );

			$field['id'] = esc_html( $field['id'] );

			$field['size']        = isset( $field['size'] ) ? $field['size'] : '40';
			$field['required']    = ( isset( $field['required'] ) && true === $field['required'] ) ? ' aria-required="true"' : '';
			$field['placeholder'] = ( isset( $field['placeholder'] ) ) ? ' placeholder="' . $field['placeholder'] . '" data-placeholder="' . $field['placeholder'] . '"' : '';
			$field['desc']        = ( isset( $field['desc'] ) ) ? $field['desc'] : '';
			$field['name']        = ( isset( $field['name'] ) ) ? $field['name'] : $field['id'];

			$field['dependency'] = ( isset( $field['dependency'] ) ) ? $field['dependency'] : [];

			self::field_start( $field );
			switch ( $field['type'] ) {
				case 'text':
				case 'url':
					?>
					<input name="<?php echo esc_attr( $field['name'] ); ?>" id="<?php echo esc_attr( $field['id'] ); ?>"
						   type="<?php echo esc_attr( $field['type'] ); ?>"
						   value="<?php echo esc_attr( $field['value'] ); ?>"
						   size="<?php echo esc_attr( $field['size'] ); ?>" <?php echo esc_html( $field['required'] ) . esc_html( $field['placeholder'] ); ?>>
					<?php
					break;
				case 'checkbox':
					$label = isset( $field['trigger_label'] ) ? $field['trigger_label'] : $field['label'];
					?>
					<label for="<?php echo esc_attr( $field['id'] ); ?>">
						<input name="<?php echo esc_attr( $field['name'] ); ?>" id="<?php echo esc_attr( $field['id'] ); ?>"
							<?php checked( $field['value'], 'yes' ); ?>
							   type="<?php echo esc_attr( $field['type'] ); ?>"
							   value="yes" <?php echo esc_html( $field['required'] ) . esc_html( $field['placeholder'] ); ?>>
						<?php echo esc_html( $label ); ?></label>
					<?php
					break;
				case 'color':
					?>
					<input name="<?php echo esc_attr( $field['name'] ); ?>" id="<?php echo esc_attr( $field['id'] ); ?>" type="text"
						   class="rtsb-vs-color-picker" value="<?php echo esc_attr( $field['value'] ); ?>"
						   data-default-color="<?php echo esc_attr( $field['value'] ); ?>"
						   size="<?php echo esc_attr( $field['size'] ); ?>" <?php echo esc_html( $field['required'] ) . esc_html( $field['placeholder'] ); ?>>
					<?php
					break;
				case 'textarea':
					?>
					<textarea name="<?php echo esc_attr( $field['name'] ); ?>" id="<?php echo esc_attr( $field['id'] ); ?>" rows="5"
							  cols="<?php echo esc_attr( $field['size'] ); ?>" <?php echo esc_attr( $field['required'] ) . esc_attr( $field['placeholder'] ); ?>><?php echo esc_html( $field['value'] ); ?></textarea>
					<?php
					break;
				case 'editor':
					$field['settings'] = $field['settings'] ?? [
						'textarea_rows' => 8,
						'quicktags'     => false,
						'media_buttons' => false,
					];
					wp_editor( $field['value'], $field['id'], $field['settings'] );
					break;
				case 'select':
				case 'select2':
					$field['options']  = isset( $field['options'] ) ? $field['options'] : [];
					$field['multiple'] = isset( $field['multiple'] ) ? ' multiple="multiple"' : '';
					$css_class         = ( 'select2' === $field['type'] ) ? 'rtsb-vs-selectwoo' : '';
					?>
					<select name="<?php echo esc_attr( $field['name'] ); ?>" id="<?php echo esc_attr( $field['id'] ); ?>"
							class="<?php echo esc_attr( $css_class ); ?>" <?php echo esc_html( $field['multiple'] ); ?>>
						<?php
						foreach ( $field['options'] as $key => $option ) {
							echo '<option' . selected( $field['value'], $key, false ) . ' value="' . esc_attr( $key ) . '">' . esc_html( $option ) . '</option>';
						}
						?>
					</select>
					<?php
					break;
				case 'image':
					?>
					<div class="rtsb-image-wrapper">
						<div class="image-preview">
							<img data-placeholder="<?php echo esc_url( self::placeholder_img_src() ); ?>"
								 src="<?php echo esc_url( self::get_img_src( $field['value'] ) ); ?>" width="60px"
								 height="60px"/>
						</div>
						<div class="button-wrapper">
							<input type="hidden" id="<?php echo esc_attr( $field['id'] ); ?>" name="<?php echo esc_attr( $field['name'] ); ?>"
								   value="<?php echo esc_attr( $field['value'] ); ?>"/>
							<button type="button"
									class="rtsb-vs-upload-image button button-primary button-small"><?php esc_html_e( 'Upload / Add image', 'shopbuilder-pro' ); ?></button>
							<button type="button"
									style="<?php echo( empty( $field['value'] ) ? 'display:none' : '' ); ?>"
									class="rtsb-vs-remove-image button button-danger button-small"><?php esc_html_e( 'Remove image', 'shopbuilder-pro' ); ?></button>
						</div>
					</div>
					<?php
					break;
				default:
					do_action( 'rtsb_vs_meta_field', $field );
					break;

			}
			self::field_end( $field );

		}
	}

	/**
	 * Start field wrapper markup.
	 *
	 * @param array $field Field config array.
	 * @return void
	 */
	private static function field_start( $field ) {
		$depends = empty( $field['dependency'] ) ? '' : "data-rtsb-vs-depends='" . wp_json_encode( $field['dependency'] ) . "'";
		?>
		<div <?php echo esc_attr( $depends ); ?> class="form-field rtsb-vs-field-wrapper <?php echo esc_attr( $field['class'] ); ?> <?php echo empty( $field['required'] ) ? '' : 'form-required'; ?>">
		<label for="<?php echo esc_attr( $field['id'] ); ?>"><?php echo esc_html( $field['label'] ); ?></label>
		<div class="field">
		<?php
	}
	/**
	 * Get image source URL from attachment ID or fallback to placeholder.
	 *
	 * @param int|false $thumbnail_id Attachment ID.
	 * @return string Image URL.
	 */
	private static function get_img_src( $thumbnail_id = false ) {
		if ( ! empty( $thumbnail_id ) ) {
			$image = wp_get_attachment_thumb_url( $thumbnail_id );
		} else {
			$image = self::placeholder_img_src();
		}

		return $image;
	}
	/**
	 * Get fallback placeholder image URL.
	 *
	 * @return string Placeholder image URL.
	 */
	private static function placeholder_img_src() {
		return 'https://placehold.co/400';
	}
	/**
	 * End field wrapper markup.
	 *
	 * @param array $field Field config array.
	 * @return void
	 */
	private static function field_end( $field ) {
		if ( ! empty( $field['desc'] ) ) :
			?>
			<p><?php echo wp_kses_post( $field['desc'] ); ?></p>
		<?php endif; ?>
		</div>
		</div>
		<?php
	}
}