<?php
/**
 * Main FilterHooks class.
 *
 * @package RadiusTheme\SB
 */

namespace RadiusTheme\SBPRO\Modules\VariationSwatches;

use RadiusTheme\SB\Helpers\Fns;
use RadiusTheme\SBPRO\Traits\SingletonTrait;
/**
 * Class AdminFilter
 *
 * Handles the addition of a custom Product Swatches tab
 * in the WooCommerce product data panel for variable products.
 *
 * @package RadiusTheme\SBPRO\Modules\VariationSwatches
 */
class AdminHooks {
	/**
	 * Singleton Trait.
	 */
	use SingletonTrait;

	/**
	 * AdminFilter constructor.
	 *
	 * Hooks into WooCommerce to insert a custom tab and its content panel.
	 */
	private function __construct() {
		add_filter( 'woocommerce_get_script_data', [ $this, 'wc_get_script_data' ], 10, 2 );
		add_filter( 'rtsb_taxonomy_common_fields', [ $this, 'taxonomy_custom_meta' ] );
		add_filter( 'rtsb_get_taxonomy_meta_color', [ $this, 'get_taxonomy_meta_color' ] );
		// Add custom tab to product data tabs.
		add_action( 'woocommerce_product_data_tabs', [ $this, 'product_swatches_tab' ], 99, 1 );
		// Add content for the custom tab.
		add_action( 'woocommerce_product_data_panels', [ $this, 'product_swatches_tab_content' ] );
		add_action( 'wp_ajax_rtsb_vs_save_attributes', [ $this, 'save_product_attributes' ] );
		add_action( 'wp_ajax_rtsb_vs_reset_attributes', [ $this, 'reset_product_attributes' ] );
		add_action( 'wp_ajax_rtsb_vs_render_attributes', [ $this, 'render_attributes' ] );
	}
	/**
	 * Reset product attributes.
	 */
	public function render_attributes() {
		if ( ! isset( $_POST['post_id'] ) || ! wp_verify_nonce( Fns::get_nonce(), rtsb()->nonceText ) ) {
			wp_send_json_error( esc_html__( 'Something went wrong.', 'shopbuilder-pro' ) );
		}

		if ( ! current_user_can( 'edit_products' ) ) { // phpcs:ignore WordPress.WP.Capabilities.Unknown
			wp_send_json_error( esc_html__( 'Permission denied.', 'shopbuilder-pro' ) );
		}

		$product_id = absint( $_POST['post_id'] );
		do_action( 'rtsb_vs_render_attributes', $product_id );
		ob_start();
		$this->product_swatches_tab_inner_content( $product_id );
		$html = ob_get_clean();
		wp_send_json_success(
			[
				'html' => $html,
			]
		);
	}

	/**
	 * Reset product attributes.
	 */
	public function reset_product_attributes() {
		if ( ! isset( $_POST['post_id'] ) || ! wp_verify_nonce( Fns::get_nonce(), rtsb()->nonceText ) ) {
			wp_send_json_error( esc_html__( 'Something went wrong.', 'shopbuilder-pro' ) );
		}

		if ( ! current_user_can( 'edit_products' ) ) { // phpcs:ignore WordPress.WP.Capabilities.Unknown
			wp_send_json_error( esc_html__( 'Permission denied.', 'shopbuilder-pro' ) );
		}

		$product_id = absint( $_POST['post_id'] );

		delete_post_meta( $product_id, '_rtsb_vs' );

		do_action( 'rtsb_vs_reset_attributes', $product_id );

		ob_start();
		$this->product_swatches_tab_inner_content( $product_id );
		$html = ob_get_clean();
		wp_send_json_success(
			[
				'html' => $html,
			]
		);
	}

	/**
	 * Save product attributes.
	 */
	public function save_product_attributes() {
		if ( ! isset( $_POST['post_id'], $_REQUEST['data'] ) || ! wp_verify_nonce( Fns::get_nonce(), rtsb()->nonceText ) ) {
			wp_send_json_error( esc_html__( 'Something went wrong.', 'shopbuilder-pro' ) );
		}

		if ( ! current_user_can( 'edit_products' ) ) { // phpcs:ignore WordPress.WP.Capabilities.Unknown
			wp_send_json_error( esc_html__( 'Permission denied.', 'shopbuilder-pro' ) );
		}

		$product_id = absint( $_POST['post_id'] );
		parse_str( $_REQUEST['data'], $parsed ); // phpcs:ignore WordPress.Security.ValidatedSanitizedInput.InputNotSanitized, WordPress.Security.ValidatedSanitizedInput.MissingUnslash

		$data = $parsed['rtsb_vs'] ?? [];

		update_post_meta( $product_id, '_rtsb_vs', $data );

		do_action( 'rtsb_vs_save_product_attributes', $product_id, $data );

		wp_send_json_success();
	}
	/**
	 * @param array $default meta fields.
	 * @return mixed|null
	 */
	public function get_taxonomy_meta_color( $default ) {
		$new      = [
			[
				'label'         => esc_html__( 'Dual Color', 'shopbuilder-pro' ),
				'trigger_label' => esc_html__( 'Enable', 'shopbuilder-pro' ),
				'id'            => 'rtsb_vs_is_dual_color',
				'type'          => 'checkbox',
			],
			[
				'label' => esc_html__( 'Secondary Color', 'shopbuilder-pro' ),
				'desc'  => esc_html__( 'Add term secondary color', 'shopbuilder-pro' ),
				'id'    => 'rtsb_vs_secondary_color',
				'type'  => 'color',
			],
		];
		$extended = array_merge( array_slice( $default, 0, 1 ), $new, array_slice( $default, 1 ) );
		return apply_filters( 'rtsb_vs_get_taxonomy_meta_color', $extended );
	}

	/**
	 * @return array
	 */
	public function taxonomy_custom_meta() {
		return [
			[
				'label'   => esc_html__( 'Show Tooltip', 'shopbuilder-pro' ),
				'desc'    => esc_html__( 'Individually show hide tooltip.', 'shopbuilder-pro' ),
				'id'      => 'rtsb_vs_attribute_tooltip',
				'type'    => 'select',
				'options' => apply_filters(
					'rtsb_vs_tooltip_option',
					[
						'text'  => esc_html__( 'Text', 'shopbuilder-pro' ),
						'image' => esc_html__( 'Image', 'shopbuilder-pro' ),
						'no'    => esc_html__( 'No', 'shopbuilder-pro' ),
					]
				),
			],
			[
				'label' => esc_html__( 'Tooltip Text', 'shopbuilder-pro' ),
				'desc'  => esc_html__( 'By default tooltip text will be the term name.', 'shopbuilder-pro' ),
				'id'    => 'rtsb_vs_attribute_tooltip_text',
				'type'  => 'text',
				'class' => 'rtsb-hidden',
			],
			[
				'label' => esc_html__( 'Tooltip Image', 'shopbuilder-pro' ),
				'desc'  => esc_html__( 'Choose an image for tooltip.', 'shopbuilder-pro' ),
				'id'    => 'rtsb_vs_attribute_tooltip_image',
				'type'  => 'image',
				'class' => 'rtsb-hidden',
			],
		];
	}
	/**
	 * @param array  $params params.
	 * @param string $handle handle.
	 * @return array
	 */
	public function wc_get_script_data( $params, $handle ) {
		if ( 'wc-add-to-cart-variation' == $handle ) {
			$params = array_merge(
				$params,
				[
					'ajax_url'                => WC()->ajax_url(),
					'i18n_view_cart'          => apply_filters( 'rtsb_vs_view_cart_text', esc_attr__( 'View cart', 'shopbuilder-pro' ) ),
					'i18n_add_to_cart'        => apply_filters( 'rtsb_vs_add_to_cart_text', esc_attr__( 'Add to cart', 'shopbuilder-pro' ) ),
					'i18n_select_options'     => apply_filters( 'rtsb_vs_select_options_text', esc_attr__( 'Select options', 'shopbuilder-pro' ) ),
					'cart_url'                => apply_filters( 'woocommerce_add_to_cart_redirect', wc_get_cart_url(), null ),
					'is_cart'                 => is_cart(),
					'cart_redirect_after_add' => get_option( 'woocommerce_cart_redirect_after_add' ),
					'enable_ajax_add_to_cart' => get_option( 'woocommerce_enable_ajax_add_to_cart' ),
				]
			);
			Fns::load_template( 'variation-swatches/variation-template', [], false, '', rtsbpro()->get_plugin_template_path() );
		}

		return $params;
	}
	/**
	 * Adds a new tab to the WooCommerce product data panel.
	 *
	 * @param array $original_tabs Existing WooCommerce product data tabs.
	 *
	 * @return array Modified product data tabs with our custom tab inserted.
	 */
	public function product_swatches_tab( $original_tabs ) {
		$new_tab['rtsb-vs-tab'] = [
			'label'  => __( 'Variation Swatches', 'shopbuilder-pro' ),
			'target' => 'rtsb_vs_tab_options',
			'class'  => [ 'show_if_variable' ],
		];

		$insert_at_position = 6;

		// Split and reassemble the tabs array to insert our custom tab.
		$tabs = array_slice( $original_tabs, 0, $insert_at_position, true );
		$tabs = array_merge( $tabs, $new_tab );
		$tabs = array_merge( $tabs, array_slice( $original_tabs, $insert_at_position, null, true ) );

		return $tabs;
	}

	/**
	 * Outputs the content of the custom product swatches tab.
	 *
	 * Includes a PHP view file containing the custom tab fields.
	 *
	 * @param int|null $product_id product id.
	 * @return void
	 */
	private function product_swatches_tab_inner_content( $product_id = null ) {
		if ( $product_id ) {
			$product = wc_get_product( $product_id );
		} else {
			global $post;
			$product = wc_get_product( $post->ID );
		}

		require_once RTSBPRO_PATH . '/app/Modules/VariationSwatches/view/product-switches-tab-content.php';
	}
	/**
	 * Outputs the content of the custom product swatches tab.
	 *
	 * Includes a PHP view file containing the custom tab fields.
	 *
	 * @return void
	 */
	public function product_swatches_tab_content() {
		wp_enqueue_style( 'rtsb-variation-admin' );
		wp_enqueue_script( 'rtsb-variation-swatch-admin' );
		?>
		<div id="rtsb_vs_tab_options" class="panel wc-metaboxes-wrapper rtsb-vs-tab-options hidden"></div>
		<?php
	}
}
