<?php
/**
 * Sticky add-to-cart Frontend Class.
 *
 * @package RadiusTheme\SB
 */

namespace RadiusTheme\SBPRO\Modules\StickyAddToCart;

use RadiusTheme\SBPRO\Modules\AddOns\AddOnsFns;
use RadiusTheme\SBPRO\Modules\PreOrder\PreOrderFns;
use WC_Product;
use RadiusTheme\SB\Helpers\Fns;
use RadiusTheme\SBPRO\Traits\SingletonTrait;

defined( 'ABSPATH' ) || exit();

/**
 * Sticky add-to-cart Frontend Class.
 */
class StickyAddToCartFrontEnd {
	/**
	 * Singleton Trait.
	 */
	use SingletonTrait;

	/**
	 * Options.
	 *
	 * @var array|mixed
	 */
	private array $options;

	/**
	 * Settings data.
	 *
	 * @var array|mixed
	 */
	private array $data;

	/**
	 * Asset Handle
	 *
	 * @var string
	 */
	private $handle = 'rtsb-sticky-add-to-cart';

	/**
	 * Class Constructor.
	 */
	private function __construct() {
		/**
		 * Get the options.
		 */
		$this->options = Fns::get_options( 'modules', 'sticky_add_to_cart' );
		$this->data    = StickyAddToCartFns::get_settings_data();

		/**
		 * Render sticky bar.
		 */
		add_action( 'wp_footer', [ $this, 'render_sticky_bar' ], 99 );

		/**
		 * Dynamic styles.
		 */
		add_action( 'wp_enqueue_scripts', [ $this, 'module_styles' ], 99 );
	}

	/**
	 * Render sticky bar.
	 *
	 * @return void
	 */
	public function render_sticky_bar() {
		if ( ! is_product() ) {
			return;
		}

		$product_id = get_the_ID();
		$product    = wc_get_product( $product_id );

		if ( ! $this->should_show_sticky_bar( $product ) ) {
			return;
		}

		// Init hooks.
		$this->init_hooks();

		$template_args = [
			'product_id'    => $product_id,
			'product'       => $product,
			'product_title' => $product->get_name(),
			'product_price' => $product->get_price_html(),
			'product_image' => Fns::get_product_image_html( 'product', $product_id, 'thumbnail' ),
			'classes'       => 'rtsb-sticky-cart-container rtsb-elementor-container',
		];
		$template_args = wp_parse_args( $template_args, $this->data );

		$template_args['classes'] .= $product->is_type( 'simple' ) ? ' is-simple' : ' is-variable';
		$template_args['classes'] .= $template_args['show_qty'] ? ' has-qty' : ' no-qty';
		$template_args['classes'] .= $template_args['show_fullwidth'] ? ' is-fullwidth' : ' no-fullwidth';
		$template_args['classes'] .= ' has-animation-' . $template_args['animation'];
		$template_args['classes'] .= 'top' === $template_args['position'] ? ' position-top' : ' position-bottom';
		$template_args['classes'] .= ' color-scheme-' . $template_args['color_scheme'];

		Fns::load_template( 'sticky-add-to-cart/sticky-bar', $template_args, false, '', rtsbpro()->get_plugin_template_path() );

		// Reset hooks.
		 $this->reset_hooks();
	}

	/**
	 * Initialize hooks.
	 *
	 * @return void
	 */
	private function init_hooks() {
		if ( $this->data['show_qty'] ) {
			remove_all_actions( 'woocommerce_before_quantity_input_field' );
			remove_all_actions( 'woocommerce_after_quantity_input_field' );

			add_action( 'woocommerce_before_quantity_input_field', [ $this, 'add_minus_button' ], 99 );
			add_action( 'woocommerce_after_quantity_input_field', [ $this, 'add_plus_button' ], 99 );
		}

		if ( ! empty( $this->data['button_label'] ) ) {
			add_filter( 'woocommerce_product_single_add_to_cart_text', [ $this, 'sticky_button_label' ], 100 );
		}
	}

	/**
	 * Reset hooks.
	 *
	 * @return void
	 */
	private function reset_hooks() {
		if ( $this->data['show_qty'] ) {
			remove_action( 'woocommerce_before_quantity_input_field', [ $this, 'add_minus_button' ], 99 );
			remove_action( 'woocommerce_after_quantity_input_field', [ $this, 'add_plus_button' ], 99 );
		}

		if ( ! empty( $this->data['button_label'] ) ) {
			remove_filter( 'woocommerce_product_single_add_to_cart_text', [ $this, 'sticky_button_label' ], 100 );
		}
	}

	/**
	 * Add minus button.
	 *
	 * @return void
	 */
	public function add_minus_button() {
		?>
		<div class="rtsb-quantity-box-group">
			<button type="button" class="rtsb-quantity-btn rtsb-quantity-minus">
				<i aria-hidden="true" class="rtsb-icon rtsb-icon-minus"></i>
			</button>
		<?php
	}

	/**
	 * Add plus button.
	 *
	 * @return void
	 */
	public static function add_plus_button() {
		?>
			<button type="button" class="rtsb-quantity-btn rtsb-quantity-plus">
				<i aria-hidden="true" class="rtsb-icon rtsb-icon-plus"></i>
			</button>
		</div><!-- .rtsb-quantity-box-group -->
		<?php
	}

	/**
	 * Modify the button text for.
	 *
	 * @param string $button_label The default button label.
	 *
	 * @return string
	 */
	public function sticky_button_label( $button_label ) {
		global $product;

		if ( PreOrderFns::is_on_pre_order( $product ) ) {
			return $button_label;
		}

		return ! empty( $this->data['button_label'] ) ? esc_html( $this->data['button_label'] ) : $button_label;
	}

	/**
	 * Dynamic styles.
	 *
	 * @return void
	 */
	public function module_styles() {
		if ( ! is_product() ) {
			return;
		}

		// Enqueue assets.
		$this->handle = Fns::enqueue_module_assets(
			$this->handle,
			'sticky-add-to-cart',
			[
				'context' => rtsbpro(),
				'version' => RTSBPRO_VERSION,
			]
		);

		$cache_key       = 'rtsb_sticky_cart_dynamic_styles_' . md5( wp_json_encode( $this->options ) );
		$parent          = '.rtsb-sticky-cart-container';
		$width_container = '.rtsb-sticky-cart-container';

		if ( $this->options['show_fullwidth'] ?? '' ) {
			$width_container = $parent . '.is-fullwidth > .rtsb-sticky-cart-wrapper';
		}

		$price  = $parent . ' .rtsb-row .rtsb-sticky-content .product-price del,' . $parent . ' .rtsb-row .rtsb-sticky-content .product-price ins,' . $parent . ' .rtsb-row .product-col .product-price > .woocommerce-Price-amount,' . $parent . ' .rtsb-row .product-col .product-price,' . $parent . ' .button-col form .variations tbody tr > .label,' . $parent . '.is-variable .button-col form .single_variation_wrap .woocommerce-variation del,' . $parent . '.is-variable .button-col form .single_variation_wrap .woocommerce-variation ins';
		$input  = $parent . '.rtsb-elementor-container .rtsb-row .rtsb-quantity-box-group .qty, ' . $parent . '.rtsb-elementor-container.color-scheme-dark .rtsb-row .button-col form .variations select';
		$qty    = $parent . '.rtsb-elementor-container .rtsb-row .rtsb-quantity-box-group .rtsb-quantity-btn';
		$button = $parent . '.rtsb-elementor-container .rtsb-row .single_add_to_cart_button';

		// phpcs:disable
		// Disabled phpcs for one-liner display.
		$css_properties = [
			'sticky_bar_width'                   => [ 'selector' => $width_container, 'property' => 'width', 'unit' => 'px' ],
			'sticky_bar_primary_color'           => [ 'selector' => $parent, 'property' => '--rtsb-color-primary' ],
			'sticky_bar_bg_color'                => [ 'selector' => $parent, 'property' => 'background-color' ],
			'sticky_bar_title_color'             => [ 'selector' => $parent . ' .product-title', 'property' => 'color' ],
			'sticky_bar_title_font_size'         => [ 'selector' => $parent . ' .product-title', 'property' => 'font-size' ],
			'sticky_bar_content_color'           => [ 'selector' => $price, 'property' => 'color' ],
			'sticky_bar_content_font_size'       => [ 'selector' => $price, 'property' => 'font-size' ],
			'sticky_bar_border_color'            => [ 'selector' => $parent, 'property' => 'border-color' ],
			'sticky_bar_shadow'                  => [ 'selector' => $parent, 'property' => 'box-shadow' ],
			'sticky_bar_radius'                  => [ 'selector' => $parent, 'property' => 'border-radius' ],
			'sticky_bar_padding'                 => [ 'selector' => $parent, 'property' => 'padding' ],
			'input_text_color'                   => [ 'selector' => $input, 'property' => 'color' ],
			'input_bg_color'                     => [ 'selector' => $input . ', ' . $qty, 'property' => 'background-color' ],
			'input_border_color'                 => [ 'selector' => $input . ', ' . $qty, 'property' => 'border-color' ],
			'qty_color'                          => [ 'selector' => $qty, 'property' => 'color' ],
			'sticky_bar_button_text_color'       => [ 'selector' => $button, 'property' => 'color' ],
			'sticky_bar_button_hover_text_color' => [ 'selector' => $button . ':hover', 'property' => 'color' ],
			'sticky_bar_button_bg_color'         => [ 'selector' => $button, 'property' => 'background-color' ],
			'sticky_bar_button_hover_bg_color'   => [ 'selector' => $button . ':hover', 'property' => 'background-color' ],
		];
		// phpcs:enable

		// Generate dynamic CSS.
		Fns::dynamic_styles( $this->options, $cache_key, $css_properties );
	}

	/**
	 * Check if the product type is supported.
	 *
	 * @param object $product The product object.
	 *
	 * @return bool
	 */
	private function is_supported_product_type( $product ) {
		$supported_types = [ 'simple' ];

		if ( $this->data['show_on_variation'] ) {
			$supported_types[] = 'variable';
		}

		return in_array( $product->get_type(), $supported_types, true );
	}

	/**
	 * Check if the sticky add-to-cart bar should be displayed.
	 *
	 * @param object $product The product object.
	 *
	 * @return bool
	 */
	private function should_show_sticky_bar( $product ) {
		if ( ! $product instanceof WC_Product ) {
			return false;
		}

		if ( ! $product->is_purchasable() || ! $product->is_in_stock() ) {
			return false;
		}

		if ( wp_is_mobile() && 'desktop' === $this->data['visibility'] ) {
			return false;
		}

		if ( ! wp_is_mobile() && 'mobile' === $this->data['visibility'] ) {
			return false;
		}

		if ( ! $this->is_supported_product_type( $product )
			|| ( Fns::is_module_active( 'product_add_ons' ) && AddOnsFns::product_has_addons( $product ) ) ) {
			return false;
		}

		return true;
	}
}
