<?php
/**
 * Volume Discount Coupon Handler.
 *
 * @package RadiusTheme\SB
 */

namespace RadiusTheme\SBPRO\Modules\SmartCoupons\Frontend;

use RadiusTheme\SBPRO\Helpers\FnsPro;
use WC_Cart;
use WC_Coupon;
use RadiusTheme\SBPRO\Traits\SingletonTrait;
use RadiusTheme\SBPRO\Modules\SmartCoupons\RTSB_Coupon;
use RadiusTheme\SBPRO\Modules\SmartCoupons\SmartCouponsFns;

defined( 'ABSPATH' ) || exit();

/**
 * Volume Discount Coupon Handler.
 */
class VolumeDiscount {
	/**
	 * Singleton Trait.
	 */
	use SingletonTrait;

	/**
	 * Initialize the class.
	 */
	private function __construct() {
		/**
		 * Actions
		 */
		add_action( 'woocommerce_before_calculate_totals', [ $this, 'apply_per_product_volume_discount' ], 20, 1 );
		add_action( 'woocommerce_cart_calculate_fees', [ $this, 'apply_cart_quantity_volume_discount' ], 10, 1 );

		/**
		 * Filters
		 */
		add_filter( 'rtsb/module/smart_coupons/adjust_coupon_discount', [ $this, 'adjust_coupon_discount' ], 10, 5 );
	}

	/**
	 * Apply the volume discount to individual items in the cart.
	 *
	 * @param WC_Cart $cart WooCommerce cart object.
	 *
	 * @return void
	 */
	public function apply_per_product_volume_discount( $cart ) {
		$this->process_volume_discount( $cart );
	}

	/**
	 * Apply the volume discount to the total cart amount.
	 *
	 * @param WC_Cart $cart The WooCommerce cart object.
	 *
	 * @return void
	 */
	public function apply_cart_quantity_volume_discount( $cart ) {
		$this->process_volume_discount( $cart, true );
	}

	/**
	 * Adjusts the coupon discount to zero for volume discount items.
	 *
	 * @param float  $discount    Original coupon discount.
	 * @param float  $discounting_amount Discount amount.
	 * @param array  $cart_item   Cart item data.
	 * @param bool   $single      Single flag.
	 * @param object $coupon      Coupon object.
	 *
	 * @return float
	 */
	public function adjust_coupon_discount( $discount, $discounting_amount, $cart_item, $single, $coupon ) {
		return SmartCouponsFns::coupon_includes_volume_discount( $coupon ) ? 0 : $discount;
	}

	/**
	 * Main method to display the volume discount message.
	 *
	 * @param array $applied_coupons List of applied coupon codes.
	 *
	 * @return void
	 */
	public static function display_volume_discount_message( $applied_coupons ) {
		if ( empty( $applied_coupons ) ) {
			return;
		}

		foreach ( $applied_coupons as $coupon_code ) {
			$coupon                = new RTSB_Coupon( $coupon_code );
			$coupon_type           = get_post_meta( $coupon->get_id(), 'discount_type', true );
			$volume_discount_rules = SmartCouponsFns::get_volume_discount_rules( $coupon );

			if ( ( 'rtsb_volume_discount' !== $coupon_type ) || empty( $volume_discount_rules ) ) {
				continue;
			}

			$discount_type = get_post_meta( $coupon->get_id(), 'rtsb_volume_coupon_type', true );
			$message       = 'total_cart' === $discount_type
				? self::get_cart_discount_message( $volume_discount_rules )
				: self::get_per_item_discount_message( $volume_discount_rules, $coupon );

			if ( ! empty( $message ) ) {
                FnsPro::custom_notice( $message, 'rtsb-volume-message', self::get_icon_svg() );
			}
		}
	}

	/**
	 * Add the volume discount metadata to order line items.
	 *
	 * @param object $item Line item.
	 * @param string $cart_item_key Cart item key.
	 * @param array  $values Cart item values.
	 *
	 * @return void
	 */
	public static function add_volume_discount_to_order_items( $item, $cart_item_key, $values ) {
		if ( isset( $values['rtsb_volume_discount'] ) ) {
			$product         = $values['data'];
			$applied_coupons = WC()->cart->get_applied_coupons();

			if ( ! empty( $applied_coupons ) ) {
				foreach ( $applied_coupons as $coupon_code ) {
					$coupon              = new RTSB_Coupon( $coupon_code );
					$rules               = SmartCouponsFns::get_volume_discount_rules( $coupon );
					$applicable_discount = SmartCouponsFns::get_applicable_discount( absint( $item->get_quantity() ), $rules );

					if ( $applicable_discount > 0 ) {
						$regular_price   = $product->get_regular_price();
						$sale_price      = $product->get_sale_price();
						$product_price   = ! empty( $sale_price ) ? $sale_price : $regular_price;
						$discount_amount = ( $product_price * $applicable_discount ) / 100;

						$item->add_meta_data( '_rtsb_volume_discount', $discount_amount, true );
					}
				}
			}
		}
	}

	/**
	 * Add the volume discount metadata to order line items.
	 *
	 * @param array  $meta_data Array of meta-data.
	 * @param string $meta_value Meta-data value.
	 * @param object $item Line item.
	 *
	 * @return array
	 */
	public static function format_volume_discount_meta( $meta_data, $meta_value, $item ) {
		$quantity = $item->get_quantity();
		$product  = $item->get_product();

		if ( $product ) {
			$product_price       = $product->get_price();
			$discount_percentage = ( $product_price > 0 ) ? ( $meta_value / $product_price ) * 100 : 0;
			$total_discount      = $meta_value * $quantity;

			$display_value = sprintf(
				'%s%s&nbsp;(%d%%), %s %s',
				esc_html__( 'Per Product Discount: ', 'shopbuilder-pro' ),
				wc_price( $meta_value ),
				round( $discount_percentage ),
				esc_html__( 'Total Discount:', 'shopbuilder-pro' ),
				wc_price( $total_discount )
			);

			$meta_data[] = (object) [
				'key'           => esc_html__( 'Volume Discount Applied', 'shopbuilder-pro' ),
				'value'         => wp_strip_all_tags( $display_value ),
				'display_key'   => esc_html__( 'Volume Discount Applied', 'shopbuilder-pro' ),
				'display_value' => wp_strip_all_tags( $display_value ),
			];
		}

		return $meta_data;
	}

	/**
	 * Get the message for per-item volume discount.
	 *
	 * @param array     $volume_discount_rules Volume discount rules.
	 * @param WC_Coupon $coupon Coupon object.
	 *
	 * @return string The formatted display message.
	 */
	private static function get_per_item_discount_message( $volume_discount_rules, $coupon ) {
		$applied_products       = [];
		$associated_product_ids = $coupon->get_product_ids();

		foreach ( WC()->cart->get_cart() as $cart_item ) {
			if ( ! in_array( $cart_item['product_id'], $associated_product_ids, true ) ) {
				continue;
			}

			$quantity = (int) $cart_item['quantity'];

			foreach ( $volume_discount_rules as $min_qty => $discount_percentage ) {
				if ( $quantity >= $min_qty ) {
					$product_title      = get_the_title( $cart_item['product_id'] );
					$applied_products[] = $product_title;

					break;
				}
			}
		}

		if ( ! empty( $applied_products ) ) {
			$message = esc_html__( 'Congratulations! A volume discount has been successfully applied to: {{products}}', 'shopbuilder-pro' );

			return str_replace( '{{products}}', implode( ', ', $applied_products ), $message );
		}

		return '';
	}

	/**
	 * Get the message for cart-level volume discount.
	 *
	 * @param array $volume_discount_rules Volume discount rules.
	 *
	 * @return string The formatted display message.
	 */
	private static function get_cart_discount_message( $volume_discount_rules ) {
		$cart_quantity     = self::get_cart_quantity( WC()->cart );
		$matching_discount = null;
		$message           = '';

		foreach ( $volume_discount_rules as $min_qty => $discount_percentage ) {
			if ( $cart_quantity >= $min_qty ) {
				$matching_discount = [
					'discount_percentage' => $discount_percentage,
					'quantity'            => $min_qty,
				];
			}
		}

		if ( ! empty( $matching_discount['discount_percentage'] ) && ! empty( $matching_discount['quantity'] ) ) {
			$message = esc_html__( 'Congratulations! Cart Quantity Discount ({{discount_percentage}}% off for {{quantity}} and more items) has been successfully applied.', 'shopbuilder-pro' );
			$message = str_replace( '{{discount_percentage}}', $matching_discount['discount_percentage'], $message );
			$message = str_replace( '{{quantity}}', $matching_discount['quantity'], $message );
		}

		return $message;
	}

	/**
	 * Process volume discounts.
	 *
	 * @param WC_Cart $cart           The WooCommerce cart object.
	 * @param bool    $is_cart_total  Whether the discount should apply to the total cart or individual items.
	 *
	 * @return void
	 */
	private function process_volume_discount( $cart, $is_cart_total = false ) {
		if ( is_admin() && ! wp_doing_ajax() ) {
			return;
		}

		foreach ( $cart->get_coupons() as $coupon ) {
			$volume_discount_rules = SmartCouponsFns::get_volume_discount_rules( $coupon );
			$discount_type         = get_post_meta( $coupon->get_id(), 'discount_type', true );

			if ( ( 'rtsb_volume_discount' !== $discount_type ) || empty( $volume_discount_rules ) ) {
				continue;
			}

			$coupon_type   = get_post_meta( $coupon->get_id(), 'rtsb_volume_coupon_type', true );
			$cart_quantity = self::get_cart_quantity( $cart );
			$discount      = $this->calculate_discount( $cart_quantity, $volume_discount_rules );

			if ( $is_cart_total && 'total_cart' === $coupon_type ) {
				$this->apply_cart_total_discount( $cart, $discount, $volume_discount_rules );
			} elseif ( ! $is_cart_total && 'total_cart' !== $coupon_type ) {
				$this->apply_discount( $cart, $coupon, $volume_discount_rules );
			}
		}
	}

	/**
	 * Get total cart quantity for the coupon's eligible products.
	 *
	 * @param WC_Cart $cart   WooCommerce cart object.
	 *
	 * @return int
	 */
	private static function get_cart_quantity( $cart ) {
		$quantity = 0;

		foreach ( $cart->get_cart() as $cart_item ) {
			if ( FreeGift::is_free_gift( $cart_item ) ) {
				continue;
			}

			$quantity += $cart_item['quantity'];
		}

		return $quantity;
	}

	/**
	 * Calculate discount based on cart quantity and volume discount rules.
	 *
	 * @param int   $cart_quantity Total cart quantity.
	 * @param array $rules         Volume discount rules.
	 *
	 * @return float
	 */
	private function calculate_discount( $cart_quantity, $rules ) {
		$discount = 0;

		foreach ( $rules as $quantity => $discount_percent ) {
			if ( $cart_quantity >= $quantity ) {
				$discount = $discount_percent;
			}
		}

		return $discount;
	}

	/**
	 * Apply the volume discount to the entire cart total.
	 *
	 * @param WC_Cart $cart WooCommerce cart object.
	 * @param float   $discount Discount percentage.
	 * @param array   $rules    Volume discount rules.
	 *
	 * @return void
	 */
	private function apply_cart_total_discount( $cart, $discount, $rules = [] ) {
		$cart_subtotal        = 0;
		$cart_quantity        = self::get_cart_quantity( $cart );
		$applicable_discount  = 0;
		$applicable_threshold = 0;

		foreach ( WC()->cart->get_cart() as $cart_item ) {
			if ( FreeGift::is_free_gift( $cart_item ) ) {
				continue;
			}

			$cart_subtotal += $cart_item['quantity'] * $cart_item['data']->get_price();
		}

		foreach ( $rules as $quantity => $percent ) {
			if ( $cart_quantity >= $quantity ) {
				$applicable_discount  = $percent;
				$applicable_threshold = $quantity;
			}
		}

		if ( $applicable_discount > 0 ) {
			$discount_amount = ( $cart_subtotal * $discount ) / 100;
			$label           = sprintf(
				/* translators: %1$s: applicable discount, %2$s: applicable threshold */
				esc_html__( 'Cart Quantity Discount (%1$s%% off for %2$s+ items)', 'shopbuilder-pro' ),
				$applicable_discount,
				$applicable_threshold
			);

			$cart->add_fee( $label, -$discount_amount, true );
		}
	}

	/**
	 * Apply the discount to the cart.
	 *
	 * @param WC_Cart   $cart     WooCommerce cart object.
	 * @param WC_Coupon $coupon   WooCommerce coupon object.
	 * @param array     $rules    Volume discount rules.
	 *
	 * @return void
	 */
	private function apply_discount( $cart, $coupon, $rules = [] ) {
		$associated_product_ids = $coupon->get_product_ids();

		foreach ( $cart->get_cart() as $cart_item_key => $cart_item ) {
			if ( ! in_array( $cart_item['product_id'], $associated_product_ids, true ) ) {
				continue;
			}

			$applicable_discount = SmartCouponsFns::get_applicable_discount( $cart_item['quantity'], $rules );

			if ( $applicable_discount > 0 ) {
				$regular_price   = $cart_item['data']->get_regular_price();
				$sale_price      = $cart_item['data']->get_sale_price();
				$product_price   = ! empty( $sale_price ) ? $sale_price : $regular_price;
				$discount_amount = ( $product_price * $applicable_discount ) / 100;

				$cart_item['data']->set_price( $product_price - $discount_amount );
				$cart->cart_contents[ $cart_item_key ]['rtsb_volume_discount'] = $discount_amount;
			}
		}
	}

	/**
	 * Get SVG icon for volume discount message.
	 *
	 * @return string
	 */
	private static function get_icon_svg() {
		return '<svg fill="currentColor" height="20" width="20" xmlns="http://www.w3.org/2000/svg" viewBox="0 0 511.999 511.999" xml:space="preserve"><path d="M399.647 227.207c-53.203 0-93.887 22.533-93.887 84.499v88.88c0 61.966 40.684 84.499 93.887 84.499 52.576 0 93.887-22.533 93.887-84.499v-88.88c.001-61.966-41.311-84.499-93.887-84.499zm31.296 173.38c0 20.654-11.892 30.043-31.295 30.043-19.403 0-30.67-9.389-30.67-30.043v-88.88c0-20.656 11.267-30.044 30.67-30.044 19.403 0 31.295 9.389 31.295 30.044v88.88zM363.345 0c-11.267 0-21.282 5.007-26.289 15.648L117.359 466.933c-1.878 3.756-3.13 8.137-3.13 11.892 0 15.648 13.77 33.174 35.052 33.174 11.892 0 23.785-6.259 28.166-15.648L397.77 45.066c1.877-3.756 2.502-8.137 2.502-11.893C400.272 13.144 380.869 0 363.345 0zM112.351 25.662c-53.203 0-93.887 22.533-93.887 84.499v88.88c0 61.966 40.685 84.499 93.887 84.499 52.577 0 93.887-22.534 93.887-84.499v-88.88c.001-61.966-41.309-84.499-93.887-84.499zm31.297 173.38c0 20.656-11.892 30.043-31.295 30.043-19.403 0-30.67-9.389-30.67-30.043v-88.88c0-20.656 11.267-30.044 30.67-30.044 19.403 0 31.295 9.389 31.295 30.044v88.88z"/></svg>';
	}
}
