<?php
/**
 * URL Coupon Handler.
 *
 * @package RadiusTheme\SB
 */

namespace RadiusTheme\SBPRO\Modules\SmartCoupons\Frontend;

use RadiusTheme\SB\Helpers\Fns;
use RadiusTheme\SBPRO\Traits\SingletonTrait;
use RadiusTheme\SBPRO\Modules\SmartCoupons\SmartCouponsFns;

defined( 'ABSPATH' ) || exit();

/**
 * URL Coupon Handler.
 */
class URLCoupon {
	/**
	 * Singleton Trait.
	 */
	use SingletonTrait;

	/**
	 * Initialize the class.
	 */
	private function __construct() {
		/**
		 * Actions
		 */
		add_action( 'template_redirect', [ $this, 'apply_coupon_via_url' ] );
		add_action( 'woocommerce_add_to_cart', [ $this, 'apply_pending_coupon' ] );

		/**
		 * Filters
		 */
		add_filter( 'wc_empty_cart_message', [ $this, 'empty_cart_message' ] );
	}

	/**
	 * Apply coupon if "apply_coupon" exists in the URL.
	 *
	 * @return void
	 */
	public function apply_coupon_via_url() {
		$coupon_code = SmartCouponsFns::get_url_coupon();

		if ( empty( $coupon_code ) ) {
			return;
		}

		if ( ! WC()->cart || WC()->cart->is_empty() ) {
			Fns::setSession( 'rtsb_pending_url_coupon', $coupon_code );
		} else {
			$this->apply_coupon( $coupon_code );
		}

		wp_safe_redirect( wc_get_page_permalink( 'cart' ) );
		exit;
	}

	/**
	 * Apply pending coupon when an item is added to the cart.
	 */
	public function apply_pending_coupon() {
		$coupon_code = Fns::getSession( 'rtsb_pending_url_coupon' );

		if ( ! empty( $coupon_code ) ) {
			$this->apply_coupon( $coupon_code );

			Fns::removeSession( 'rtsb_pending_url_coupon' );
		}
	}

	/**
	 * Apply the coupon if it's not already applied.
	 *
	 * @param string $coupon_code Coupon code to apply.
	 */
	private function apply_coupon( $coupon_code ) {
		if ( ! in_array( strtolower( $coupon_code ), WC()->cart->get_applied_coupons(), true ) ) {
			WC()->cart->add_discount( $coupon_code );
			SmartCouponsFns::trigger_notice(
				sprintf(
					/* translators: %s: Coupon code. */
					esc_html__( 'The coupon "%s" has been successfully applied!', 'shopbuilder-pro' ),
					esc_html( $coupon_code )
				)
			);
		}
	}

	/**
	 * Modify empty cart message if a coupon was applied via URL.
	 *
	 * @param string $message Default empty cart message.
	 * @return string Modified empty cart message.
	 */
	public function empty_cart_message( $message ) {
		$coupon_code = Fns::getSession( 'rtsb_pending_url_coupon' );

		if ( empty( $coupon_code ) ) {
			return $message;
		}

		return sprintf(
			/* translators: %s: Coupon code. */
			esc_html__( 'The coupon "%s" has been successfully applied via URL! Please add some items to your cart.', 'shopbuilder-pro' ),
			esc_html( $coupon_code )
		);
	}
}
