<?php
/**
 * Smart Coupons Module Coupon Export Ajax Class.
 *
 * @package RadiusTheme\SB
 */

namespace RadiusTheme\SBPRO\Modules\SmartCoupons\Admin;

use WP_Query;
use WC_Coupon;
use RadiusTheme\SB\Helpers\Fns;
use RadiusTheme\SBPRO\Traits\SingletonTrait;
use RadiusTheme\SBPRO\Modules\SmartCoupons\SmartCouponsFns;

defined( 'ABSPATH' ) || exit();

/**
 * Smart Coupons Module Coupon Filters Class.
 */
class ExportCoupons {
	/**
	 * Singleton Trait.
	 */
	use SingletonTrait;

	/**
	 * Class Constructor.
	 */
	private function __construct() {
		if ( empty( SmartCouponsFns::get_settings_data()['enable_exporter'] ) ) {
			return;
		}

		/**
		 * Actions.
		 */
		add_action( 'wp_ajax_rtsb_export_coupons', [ $this, 'export_coupons' ] );
		add_action( 'wp_loaded', [ $this, 'clean_if_needed' ] );
	}

	/**
	 * Export coupons.
	 *
	 * @return void
	 */
	public function export_coupons() {
		// phpcs:ignore WordPress.Security.NonceVerification.Recommended, WordPress.WP.Capabilities.Unknown
		if ( ! Fns::verify_nonce() && ! is_admin() && ! current_user_can( 'manage_woocommerce' ) ) {
			wp_send_json_error( esc_html__( 'Invalid request', 'shopbuilder-pro' ) );
		}

		$coupon_data = [];

		// phpcs:ignore WordPress.Security.NonceVerification.Recommended
		if ( ! empty( $_REQUEST['form_data'] ) ) {
			// phpcs:ignore WordPress.Security.NonceVerification.Recommended
			parse_str( sanitize_text_field( wp_unslash( $_REQUEST['form_data'] ) ), $coupon_data );
		}

		// Get coupon IDs based on filters.
		$coupon_ids = $this->get_coupon_ids( $coupon_data );

		if ( empty( $coupon_ids ) ) {
			wp_send_json_error( [ 'message' => esc_html__( 'No coupons found to export.', 'shopbuilder-pro' ) ] );
		}

		// Generate the CSV file and get its URL.
		$download_url = $this->generate_csv( $coupon_ids );

		if ( ! $download_url ) {
			wp_send_json_error( [ 'message' => esc_html__( 'Failed to generate CSV file.', 'shopbuilder-pro' ) ] );
		}

		update_option( 'rtsb/module/coupons/last_export_time', time() );

		wp_send_json_success(
			[
				'message'      => esc_html__( 'Coupons exported successfully!', 'shopbuilder-pro' ),
				'download_url' => esc_url( $download_url['file_url'] ),
			]
		);
	}

	/**
	 * Get coupon ids.
	 *
	 * @param array $data Coupon data.
	 *
	 * @return int[]
	 */
	private function get_coupon_ids( $data ) {
		$args = [
			'post_type'      => 'shop_coupon',
			'posts_per_page' => -1,
			'fields'         => 'ids',
		];

		$get_coupon_type     = ( ! empty( $data['coupon_type'] ) ) ? sanitize_text_field( wp_unslash( $data['coupon_type'] ) ) : '';
		$post_status         = ( ! empty( $data['post_status '] ) ) ? sanitize_text_field( wp_unslash( $data['post_status '] ) ) : 'all';
		$group               = ( ! empty( $data['rtsb_coupon_group'] ) ) ? sanitize_text_field( wp_unslash( $data['rtsb_coupon_group'] ) ) : '';
		$args['post_status'] = 'all' === $post_status ? [ 'publish', 'draft', 'pending', 'private', 'future' ] : $post_status;

		if ( ! empty( $get_coupon_type ) ) {
			// phpcs:ignore WordPress.DB.SlowDBQuery.slow_db_query_meta_query
			$args['meta_query'] = [
				[
					'key'   => 'discount_type',
					'value' => $get_coupon_type,
				],
			];
		}

		if ( ! empty( $group ) ) {
			// phpcs:ignore WordPress.DB.SlowDBQuery.slow_db_query_tax_query
			$args['tax_query'] = [
				[
					'taxonomy' => 'rtsb_coupon_group',
					'field'    => 'slug',
					'terms'    => $group,
				],
			];
		}

		$query = new WP_Query( $args );

		return $query->posts;
	}

	/**
	 * Generate and send CSV file.
	 *
	 * @param int[] $coupon_ids Array of coupon IDs.
	 *
	 * @return array
	 */
	private function generate_csv( $coupon_ids ) {
		global $wp_filesystem;

		if ( ! function_exists( 'WP_Filesystem' ) ) {
			require_once ABSPATH . 'wp-admin/includes/file.php';
		}
		WP_Filesystem();

		$upload_dir = wp_upload_dir();
		$csv_dir    = trailingslashit( $upload_dir['basedir'] ) . 'shopbuilder-exports/';
		$csv_url    = trailingslashit( $upload_dir['baseurl'] ) . 'shopbuilder-exports/';

		if ( ! $wp_filesystem->is_dir( $csv_dir ) ) {
			$wp_filesystem->mkdir( $csv_dir );
		}

		$temp_csv_path = $csv_dir . 'shopbuilder_coupons_export_' . gmdate( 'Y_m_d_H_i_s' ) . '.csv';

		if ( ! $wp_filesystem->put_contents( $temp_csv_path, '', FS_CHMOD_FILE ) ) {
			wp_send_json( [ 'message' => __( 'Failed to create CSV file.', 'shopbuilder-pro' ) ] );
		}

		$csv_content = '';
		$counter     = 1;

		$header_row   = [
			esc_html__( 'No.', 'shopbuilder-pro' ),
			esc_html__( 'Coupon Code', 'shopbuilder-pro' ),
			esc_html__( 'Description', 'shopbuilder-pro' ),
			esc_html__( 'Discount Type', 'shopbuilder-pro' ),
			esc_html__( 'Amount', 'shopbuilder-pro' ),
			esc_html__( 'Date Created', 'shopbuilder-pro' ),
			esc_html__( 'Coupon Group', 'shopbuilder-pro' ),
			esc_html__( 'Start Date', 'shopbuilder-pro' ),
			esc_html__( 'Expiry Date', 'shopbuilder-pro' ),
		];
		$header_row   = apply_filters( 'rtsb/module/coupons/csv_headers', $header_row, $coupon_ids );
		$csv_content .= $this->array_to_csv_line( $header_row );

		foreach ( $coupon_ids as $coupon_id ) {
			$coupon = new WC_Coupon( $coupon_id );
			$post   = get_post( $coupon_id );

			if ( ! $coupon->get_id() ) {
				continue;
			}

			$row = [
				$counter++,
				$coupon->get_code(),
				$post ? $post->post_excerpt : '',
				$coupon->get_discount_type(),
				$coupon->get_amount(),
				$coupon->get_date_created() ? $coupon->get_date_created()->date( 'F j, Y' ) : '',
				$this->get_coupon_group( $coupon_id ),
				$this->get_coupon_start_date( $coupon_id ),
				$coupon->get_date_expires() ? $coupon->get_date_expires()->date( 'F j, Y' ) : '',
			];
			$row = apply_filters( 'rtsb/module/coupons/csv_row', $row, $coupon_id );

			$csv_content .= $this->array_to_csv_line( $row );
		}

		if ( ! $wp_filesystem->put_contents( $temp_csv_path, $csv_content ) ) {
			wp_send_json( [ 'message' => esc_html__( 'Failed to write CSV file content.', 'shopbuilder-pro' ) ] );
		}

		return [
			'file_url'  => $csv_url . basename( $temp_csv_path ),
			'file_path' => $temp_csv_path,
		];
	}

	/**
	 * Cleanup exported files if needed.
	 *
	 * @return void
	 */
	public function clean_if_needed() {
		$last_call_time = get_option( 'rtsb/module/coupons/last_export_time', false );

		if ( false === $last_call_time ) {
			return;
		}

		// Cleanup after 6 hours.
		if ( time() - absint( $last_call_time ) > 21600 ) {
			$this->clean();

			delete_option( 'rtsb/module/coupons/last_export_time' );
		}
	}

	/**
	 * Clean exported files.
	 *
	 * @return void
	 */
	public function clean() {
		$upload_dir = wp_upload_dir();
		$csv_dir    = trailingslashit( $upload_dir['basedir'] ) . 'shopbuilder-exports/';

		if ( ! is_dir( $csv_dir ) ) {
			return;
		}

		// Get all files in the directory.
		$files = glob( $csv_dir . '*' );

		if ( ! empty( $files ) ) {
			foreach ( $files as $file ) {
				wp_delete_file( $file );
			}
		}
	}

	/**
	 * Convert an array to a single CSV-formatted line.
	 *
	 * @param array $data The data to convert.
	 *
	 * @return string
	 */
	private function array_to_csv_line( $data ) {
		ob_start();

		$output = fopen( 'php://output', 'w' );

		if ( false !== $output ) {
			fputcsv( $output, $data );
			fclose( $output ); // phpcs:ignore WordPress.WP.AlternativeFunctions.file_system_operations_fclose
		}

		return ob_get_clean();
	}

	/**
	 * Get the coupon group taxonomy terms for a coupon.
	 *
	 * @param int $coupon_id The coupon post ID.
	 *
	 * @return string
	 */
	private function get_coupon_group( $coupon_id ) {
		$terms = wp_get_post_terms( $coupon_id, 'rtsb_coupon_group', [ 'fields' => 'names' ] );

		return $terms ? implode( ', ', $terms ) : '';
	}

	/**
	 * Retrieve the start date for a given coupon and format it.
	 *
	 * @param int $coupon_id The ID of the coupon.
	 *
	 * @return string
	 */
	private function get_coupon_start_date( $coupon_id ) {
		$start_date = get_post_meta( $coupon_id, 'rtsb_coupon_start', true );

		return $start_date ? date_i18n( 'F j, Y', strtotime( $start_date ) ) : '';
	}
}
