<?php
/**
 * Smart Coupons Module Admin Pages Class.
 *
 * @package RadiusTheme\SB
 */

namespace RadiusTheme\SBPRO\Modules\SmartCoupons\Admin;

use WP_Post;
use WP_Query;
use WC_Coupon;
use WC_Meta_Box_Coupon_Data;
use RadiusTheme\SB\Helpers\Fns;
use Automattic\Jetpack\Constants;
use RadiusTheme\SBPRO\Traits\SingletonTrait;
use RadiusTheme\SBPRO\Helpers\AdminInputsFns;
use RadiusTheme\SBPRO\Modules\SmartCoupons\SmartCouponsFns;

defined( 'ABSPATH' ) || exit();

/**
 * Smart Coupons Module Admin Pages Class.
 */
class AdminPages {
	/**
	 * Singleton Trait.
	 */
	use SingletonTrait;

	/**
	 * Settings data.
	 *
	 * @var array|mixed
	 */
	private array $data;

	/**
	 * Active Tab.
	 *
	 * @var string
	 */
	private $active_tab = 'coupons';

	/**
	 * Class Constructor.
	 */
	private function __construct() {
		// Get settings data.
		$this->data = SmartCouponsFns::get_settings_data();

		/**
		 * Actions.
		 */
		add_action( 'in_admin_header', [ $this, 'remove_all_notices' ] );
		add_action( 'admin_menu', [ $this, 'register_admin_page' ] );
		add_action( 'admin_head', [ $this, 'remove_submenu' ] );
		add_action( 'rtsb/admin/shop_coupons/views', [ $this, 'add_coupons_tabs' ] );
		add_action( 'admin_post_rtsb_generate_coupons', [ $this, 'process_bulk_coupon_generation' ] );

		/**
		 * Filters
		 */
		add_filter( 'views_edit-shop_coupon', [ $this, 'add_custom_hook' ] );
		add_filter( 'woocommerce_navigation_is_connected_page', [ $this, 'connect_custom_page' ], 10, 2 );
		add_filter( 'woocommerce_navigation_get_breadcrumbs', [ $this, 'custom_breadcrumbs' ] );
		add_filter( 'woocommerce_screen_ids', [ $this, 'add_custom_screen_id' ] );
	}

	/**
	 * Registers the Coupons page.
	 *
	 * @return void
	 */
	public function register_admin_page() {
		if ( empty( $this->data['enable_bulk_coupon'] ) ) {
			return;
		}

		add_submenu_page(
			'woocommerce-marketing',
			esc_html__( 'ShopBuilder Smart Coupons', 'shopbuilder-pro' ),
			esc_html__( 'ShopBuilder Smart Coupons', 'shopbuilder-pro' ),
			'manage_woocommerce', // phpcs:ignore WordPress.WP.Capabilities.Unknown
			'rtsb-smart-coupons',
			[ $this, 'render_coupon_generator_page' ]
		);
	}

	/**
	 * Render content on the Coupons page.
	 *
	 * @return void
	 */
	public function render_coupon_generator_page() {
		?>
		<div class="wrap">
			<h1 class="wp-heading-inline rtsb-coupon-generator-title"><?php echo esc_html__( 'Bulk Coupon Generator', 'shopbuilder-pro' ); ?></h1>
			<hr class="wp-header-end">
			<?php
			$this->render_notice();
			$this->set_active_tab( 'bulk_generator' );
			$this->add_coupons_tabs();
			$this->render_coupon_generator_form();
			?>
		</div>
		<?php
	}

	/**
	 * Remove the submenu item from the WooCommerce Marketing menu.
	 *
	 * @return void
	 */
	public function remove_submenu() {
		remove_submenu_page( 'woocommerce-marketing', 'rtsb-smart-coupons' );
	}

	/**
	 * Add custom views to the coupon list table.
	 *
	 * @param array $views An array of existing views.
	 *
	 * @return array
	 */
	public function add_custom_hook( $views ) {
		global $typenow;

		if ( 'shop_coupon' === $typenow ) {
			$this->set_active_tab( 'coupons' );

			do_action( 'rtsb/admin/shop_coupons/views' );
		}

		return $views;
	}

	/**
	 * Adds custom navigation tabs for the Coupons and Bulk Coupon Generator views in the admin area.
	 *
	 * @return void
	 */
	public function add_coupons_tabs() {
		if ( empty( $this->data['enable_bulk_coupon'] ?? '' ) ) {
			return;
		}
		?>
		<div id="rtsb_coupons_tabs">
			<h2 class="nav-tab-wrapper">
				<?php
				$coupons_url        = esc_url( add_query_arg( [ 'post_type' => 'shop_coupon' ], admin_url( 'edit.php' ) ) );
				$bulk_generator_url = esc_url( add_query_arg( [ 'page' => 'rtsb-smart-coupons' ], admin_url( 'admin.php' ) ) );

				$coupons_class = 'coupons' === $this->active_tab ? 'nav-tab nav-tab-active' : 'nav-tab';
				$bulk_class    = 'bulk_generator' === $this->active_tab ? 'nav-tab nav-tab-active' : 'nav-tab';

				echo '<a href="' . esc_url( $coupons_url ) . '" class="' . esc_attr( $coupons_class ) . '">' . esc_html__( 'Coupons', 'shopbuilder-pro' ) . '</a>';
				echo '<a href="' . esc_url( $bulk_generator_url ) . '" class="' . esc_attr( $bulk_class ) . '">' . esc_html__( 'Bulk Coupon Generator', 'shopbuilder-pro' ) . '</a>';
				?>
			</h2>
		</div>
		<?php
	}

	/**
	 * Renders the coupon generator form on the Bulk Coupon Generator page.
	 *
	 * @return void
	 */
	private function render_coupon_generator_form() {
		$coupon_code = $this->generate_random_coupon();
		$coupon      = new WC_Coupon( $coupon_code );
		$post_data   = [
			'ID'           => 0,
			'post_type'    => 'shop_coupon',
			'post_title'   => $coupon_code,
			'post_excerpt' => $coupon->get_description(),
			'post_status'  => 'auto-draft',
		];

		$post = new WP_Post( (object) $post_data );
		?>
		<form method="post" action="<?php echo esc_url( admin_url( 'admin-post.php' ) ); ?>">
			<input type="hidden" name="action" value="rtsb_generate_coupons">
			<?php wp_nonce_field( 'rtsb_generate_coupons_nonce' ); ?>
			<div id="poststuff">
				<div id="post-body" class="metabox-holder columns-2 postbox rtsb-bulk-generator-metabox">

					<!-- Main Column for WooCommerce Coupon Fields -->
					<div id="post-body-content" class="rtsb-bulk-generator">
						<div id="rtsb-bulk-generator-settings">
							<div class="rtsb-group-heading">
								<h2><?php esc_html_e( 'Bulk Generation Settings', 'shopbuilder-pro' ); ?></h2>
							</div>
							<div class="rtsb-coupon-settings woocommerce_options_panel">
								<?php
								$fields = AdminInputsFns::generate_number_input(
									[
										'id'            => 'rtsb_number_of_coupons',
										'label'         => esc_html__( 'Number of Coupons', 'shopbuilder-pro' ),
										'desc_tip'      => true,
										'aria_label'    => esc_html__( 'Number of Coupons', 'shopbuilder-pro' ),
										'description'   => __( 'Enter the number of coupons to generate.', 'shopbuilder-pro' ),
										'placeholder'   => esc_html__( '10', 'shopbuilder-pro' ),
										'value'         => 10,
										'min'           => '0',
										'step'          => '1',
										'input_class'   => 'rtsb-number-input',
										'wrapper_class' => 'form-field rtsb-number-field',
									]
								);

								$fields .= AdminInputsFns::generate_text_input(
									[
										'id'            => 'rtsb_coupons_prefix',
										'label'         => esc_html__( 'Coupon Prefix', 'shopbuilder-pro' ),
										'desc_tip'      => true,
										'aria_label'    => esc_html__( 'Coupon Prefix', 'shopbuilder-pro' ),
										'description'   => __( 'Enter the coupon prefix. This prefix will be appended before the coupon code. Recommended prefix is up to three characters.', 'shopbuilder-pro' ),
										'placeholder'   => esc_html__( 'Ex: PR', 'shopbuilder-pro' ),
										'value'         => '',
										'input_class'   => 'rtsb-number-input',
										'wrapper_class' => 'form-field rtsb-number-field',
									]
								);

								$fields .= AdminInputsFns::generate_text_input(
									[
										'id'            => 'rtsb_coupons_suffix',
										'label'         => esc_html__( 'Coupon Suffix', 'shopbuilder-pro' ),
										'desc_tip'      => true,
										'aria_label'    => esc_html__( 'Coupon Suffix', 'shopbuilder-pro' ),
										'description'   => __( 'Enter the coupon suffix. This suffix will be appended after the coupon code. Recommended suffix is up to three characters.', 'shopbuilder-pro' ),
										'placeholder'   => esc_html__( 'Ex: SF', 'shopbuilder-pro' ),
										'value'         => '',
										'input_class'   => 'rtsb-number-input',
										'wrapper_class' => 'form-field rtsb-number-field',
									]
								);

								$fields .= AdminInputsFns::generate_textarea_input(
									[
										'id'            => 'rtsb_bulk_coupon_description',
										'label'         => esc_html__( 'Coupon Description', 'shopbuilder-pro' ),
										'desc_tip'      => true,
										'aria_label'    => esc_html__( 'Bulk Coupon Description', 'shopbuilder-pro' ),
										'description'   => esc_html__( 'Enter the coupon description. This description will be applied to all the coupons generated.', 'shopbuilder-pro' ),
										'placeholder'   => esc_html__( 'Coupon Description (Optional)', 'shopbuilder-pro' ),
										'input_class'   => 'short rtsb-bulk-coupon-description',
										'wrapper_class' => 'form-field coupon-description-field',
									]
								);

								Fns::print_html( $fields, true );
								?>
							</div>
						</div>
					</div>

					<div id="postbox-container-1" class="postbox-container">
						<div id="side-sortables" class="meta-box-sortables">
							<div id="submitdiv" class="postbox">
								<h2><?php esc_html_e( 'Action', 'shopbuilder-pro' ); ?></h2>
								<div class="inside">
									<div class="submitbox">
										<div id="major-publishing-actions">
											<div id="publishing-action">
												<input type="submit" name="rtsb_generate_coupons" id="publish" class="button button-primary button-large" value="<?php esc_attr_e( 'Generate Coupons', 'shopbuilder-pro' ); ?>">
											</div>
											<div class="clear"></div>
										</div>
									</div>
								</div>
							</div>

							<?php
							// Display the coupon groups meta-box.
							$this->render_custom_taxonomy_metabox( $post );
							?>
						</div>
					</div>
					<div id="postbox-container-2" class="postbox-container">
						<div id="woocommerce-coupon-data" class="postbox">
							<div class="postbox-header">
								<h2><?php esc_html_e( 'Coupon data', 'shopbuilder-pro' ); ?></h2>
							</div>
							<div class="inside">
								<?php
								// Load WooCommerce coupon fields.
								$this->render_coupon_fields( $post );
								?>
							</div>
						</div>
					</div>
				</div>
			</div>
		</form>
		<?php
	}

	/**
	 * Renders WooCommerce coupon fields.
	 *
	 * @param WP_Post|null $post The post-object.
	 *
	 * @return void
	 */
	public function render_coupon_fields( $post = null ) {
		$this->load_bulk_coupon_assets();

		include_once WC()->plugin_path() . '/includes/admin/meta-boxes/class-wc-meta-box-coupon-data.php';

		global $post;

		$temp_post = $post;

		if ( ! $post ) {
			$post = new WP_Post( (object) [ 'ID' => 0 ] ); // phpcs:ignore WordPress.WP.GlobalVariablesOverride.Prohibited
		}

		WC_Meta_Box_Coupon_Data::output( $post );

		$post = $temp_post; // phpcs:ignore WordPress.WP.GlobalVariablesOverride.Prohibited
	}

	/**
	 * Renders the custom taxonomy meta-box for coupon groups.
	 *
	 * @param WP_Post $post The post-object.
	 *
	 * @return void
	 */
	public function render_custom_taxonomy_metabox( $post ) {
		wp_enqueue_script( 'tags-box' );
		?>
		<div id="rtsb_coupon_groupdiv" class="postbox">
			<div class="postbox-header">
				<h2><span><?php esc_html_e( 'Coupon Groups', 'shopbuilder-pro' ); ?></span></h2>
			</div>
			<div class="inside">
				<?php
				$args     = [ 'taxonomy' => 'rtsb_coupon_group' ];
				$tax_name = esc_attr( $args['taxonomy'] );
				$taxonomy = get_taxonomy( $args['taxonomy'] );
				?>

				<div id="taxonomy-<?php echo esc_attr( $tax_name ); ?>" class="categorydiv">
					<div class="rtsb-view-category">
						<a target="_blank" title="" href="<?php echo esc_url( admin_url( 'edit-tags.php?taxonomy=rtsb_coupon_group&post_type=shop_coupon' ) ); ?>">
							<?php echo esc_html__( 'Manage Coupon Groups', 'shopbuilder-pro' ); ?>
						</a>
					</div>
					<ul id="<?php echo esc_attr( $tax_name ); ?>-tabs" class="category-tabs">
						<li class="tabs"><a href="#<?php echo esc_attr( $tax_name ); ?>-all"><?php echo esc_html( $taxonomy->labels->all_items ); ?></a></li>
						<li class="hide-if-no-js"><a href="#<?php echo esc_attr( $tax_name ); ?>-pop"><?php echo esc_html( $taxonomy->labels->most_used ); ?></a></li>
					</ul>

					<div id="<?php echo esc_attr( $tax_name ); ?>-pop" class="tabs-panel" style="display: none;">
						<ul id="<?php echo esc_attr( $tax_name ); ?>checklist-pop" class="categorychecklist form-no-clear">
							<?php wp_popular_terms_checklist( $tax_name ); ?>
						</ul>
					</div>

					<div id="<?php echo esc_attr( $tax_name ); ?>-all" class="tabs-panel">
						<ul id="<?php echo esc_attr( $tax_name ); ?>checklist" data-wp-lists="list:<?php echo esc_attr( $tax_name ); ?>" class="categorychecklist form-no-clear">
							<?php
							wp_terms_checklist(
								$post->ID,
								[
									'taxonomy' => $tax_name,
								]
							);
							?>
						</ul>
					</div>
				</div>
				<script>
					jQuery(document).ready(function($) {
						// Handler for tab clicks.
						$('#rtsb_coupon_group-tabs a').on('click', function(e) {
							e.preventDefault();

							$('#rtsb_coupon_group-tabs .tabs').removeClass('tabs');
							$(this).parent().addClass('tabs');

							var panelToShow = $(this).attr('href');
							$('.tabs-panel').hide();
							$(panelToShow).show();
						});

						$('#rtsb_coupon_group-all').show();
						$('#rtsb_coupon_group-pop').hide();

						// Synchronize category checkboxes for the rtsb_coupon_group taxonomy.
						$('#taxonomy-rtsb_coupon_group .categorychecklist :checkbox').on('change', function() {
							var checkbox = $(this),
								isChecked = checkbox.is(':checked'),
								termId = checkbox.val().toString();

							$('#in-rtsb_coupon_group-' + termId + '-2, #in-popular-rtsb_coupon_group-' + termId).prop('checked', isChecked);
						}).filter(':checked').trigger('change');
					});
				</script>
			</div>
		</div>
		<?php
	}

	/**
	 * Enqueues the necessary scripts and styles for the bulk coupon generation page.
	 *
	 * @return void
	 */
	private function load_bulk_coupon_assets() {
		wp_enqueue_style( 'rtsb-product-admin' );
		wp_enqueue_script( 'wc-admin-meta-boxes' );

		wp_enqueue_script( 'wc-admin-coupon-meta-boxes', WC()->plugin_url() . '/assets/js/admin/meta-boxes-coupon' . ( Constants::is_true( 'SCRIPT_DEBUG' ) ? '' : '.min' ) . '.js', [ 'wc-admin-meta-boxes' ], Constants::get_constant( 'WC_VERSION' ), true );
		wp_localize_script(
			'wc-admin-coupon-meta-boxes',
			'woocommerce_admin_meta_boxes_coupon',
			[
				'generate_button_text' => esc_html__( 'Generate coupon code', 'shopbuilder-pro' ),
				'characters'           => apply_filters( 'woocommerce_coupon_code_generator_characters', 'ABCDEFGHJKMNPQRSTUVWXYZ23456789' ),
				'char_length'          => apply_filters( 'woocommerce_coupon_code_generator_character_length', 8 ),
				'prefix'               => apply_filters( 'woocommerce_coupon_code_generator_prefix', '' ),
				'suffix'               => apply_filters( 'woocommerce_coupon_code_generator_suffix', '' ),
			]
		);
	}

	/**
	 * Processes the bulk coupon generation based on form input.
	 *
	 * @return void
	 */
	public function process_bulk_coupon_generation() {
		if ( empty( $this->data['enable_bulk_coupon'] ?? '' ) ) {
			return;
		}

		if ( ! isset( $_POST['rtsb_generate_coupons'] ) || ! check_admin_referer( 'rtsb_generate_coupons_nonce' ) ) {
			return;
		}

		$coupon_count = isset( $_POST['rtsb_number_of_coupons'] ) ? absint( $_POST['rtsb_number_of_coupons'] ) : 0;
		$prefix       = isset( $_POST['rtsb_coupons_prefix'] ) ? sanitize_text_field( wp_unslash( $_POST['rtsb_coupons_prefix'] ) ) : '';
		$suffix       = isset( $_POST['rtsb_coupons_suffix'] ) ? sanitize_text_field( wp_unslash( $_POST['rtsb_coupons_suffix'] ) ) : '';
		$description  = isset( $_POST['rtsb_bulk_coupon_description'] ) ? sanitize_textarea_field( wp_unslash( $_POST['rtsb_bulk_coupon_description'] ) ) : '';

		if ( $coupon_count <= 0 ) {
			wp_safe_redirect( add_query_arg( 'rtsb_coupons_generated', 'error', wp_get_referer() ) );
			exit;
		}

		for ( $i = 0; $i < $coupon_count; $i++ ) {
			do {
				$coupon_code = $prefix . $this->generate_random_coupon() . $suffix;
			} while ( wc_get_coupon_id_by_code( $coupon_code ) );

			$coupon = new WC_Coupon();

			// phpcs:disable
			$coupon->set_props(
				[
					'code'                        => $coupon_code,
					'discount_type'               => wc_clean( $_POST['discount_type'] ),
					'amount'                      => wc_format_decimal( $_POST['coupon_amount'] ),
					'date_expires'                => wc_clean( $_POST['expiry_date'] ),
					'individual_use'              => isset( $_POST['individual_use'] ),
					'product_ids'                 => isset( $_POST['product_ids'] ) ? array_filter( array_map( 'intval', (array) $_POST['product_ids'] ) ) : [],
					'excluded_product_ids'        => isset( $_POST['exclude_product_ids'] ) ? array_filter( array_map( 'intval', (array) $_POST['exclude_product_ids'] ) ) : [],
					'usage_limit'                 => absint( $_POST['usage_limit'] ),
					'usage_limit_per_user'        => absint( $_POST['usage_limit_per_user'] ),
					'limit_usage_to_x_items'      => absint( $_POST['limit_usage_to_x_items'] ),
					'free_shipping'               => isset( $_POST['free_shipping'] ),
					'product_categories'          => isset( $_POST['product_categories'] ) ? array_filter( array_map( 'intval', (array) $_POST['product_categories'] ) ) : [],
					'excluded_product_categories' => isset( $_POST['exclude_product_categories'] ) ? array_filter( array_map( 'intval', (array) $_POST['exclude_product_categories'] ) ) : [],
					'exclude_sale_items'          => isset( $_POST['exclude_sale_items'] ),
					'minimum_amount'              => wc_format_decimal( $_POST['minimum_amount'] ),
					'maximum_amount'              => wc_format_decimal( $_POST['maximum_amount'] ),
					'email_restrictions'          => array_filter( array_map( 'trim', explode( ',', wc_clean( $_POST['customer_email'] ) ) ) ),
					'description'                 => $description,
				]
			);
			// phpcs:enable

			$coupon->save();

			// Save taxonomy data.
			if ( isset( $_POST['tax_input']['rtsb_coupon_group'] ) ) {
				$terms = array_map( 'intval', $_POST['tax_input']['rtsb_coupon_group'] );

				wp_set_post_terms( $coupon->get_id(), $terms, 'rtsb_coupon_group' );
			}

			do_action( 'woocommerce_coupon_options_save', $coupon->get_id(), $coupon );
		}

		wp_safe_redirect( add_query_arg( 'rtsb_coupons_generated', 'success', wp_get_referer() ) );
		exit;
	}

	/**
	 * Generate a unique coupon code.
	 *
	 * @param int $length Length of the coupon code.
	 * @return string
	 */
	public function generate_random_coupon( $length = 8 ) {
		do {
			$chars       = array_diff( array_merge( range( 'a', 'z' ), range( '0', '9' ) ), [ 'a', 'e', 'i', 'o', 'u' ] );
			$chars       = array_values( $chars );
			$unique_code = '';

			for ( $i = 0; $i < $length; $i++ ) {
				$random_index = wp_rand( 0, count( $chars ) - 1 );
				$unique_code .= $chars[ $random_index ];
			}

			$query = new WP_Query(
				[
					'post_type'      => 'shop_coupon',
					'title'          => $unique_code,
					'post_status'    => 'any',
					'fields'         => 'ids',
					'posts_per_page' => 1,
				]
			);

		} while ( $query->have_posts() );

		return $unique_code;
	}

	/**
	 * Sets the active tab for the coupons view.
	 *
	 * @param string $tab The slug of the tab to set as active.
	 *
	 * @return void
	 */
	public function set_active_tab( $tab ) {
		$this->active_tab = $tab;
	}

	/**
	 * Remove admin notices
	 *
	 * @return void
	 */
	public function remove_all_notices() {
		$screen = get_current_screen();

		if ( 'marketing_page_rtsb-smart-coupons' === $screen->base || 'shop_coupon' === $screen->post_type ) {
			remove_all_actions( 'admin_notices' );
			remove_all_actions( 'all_admin_notices' );
		}
	}

	/**
	 * Connects the Coupons page to the WooCommerce navigation.
	 *
	 * @param  boolean $is_connected_page Is connected page.
	 * @return boolean
	 */
	public function connect_custom_page( $is_connected_page ) {
		$get_page = ( ! empty( $_GET['page'] ) ) ? sanitize_text_field( wp_unslash( $_GET['page'] ) ) : ''; // phpcs:ignore WordPress.Security.NonceVerification.Recommended

		if ( empty( $is_connected_page ) && 'rtsb-smart-coupons' === $get_page ) {
			return true;
		}

		return $is_connected_page;
	}

	/**
	 * WooCommerce Navigation Is Connected Page
	 *
	 * @param  boolean $breadcrumbs  The breadcrumbs.
	 *
	 * @return boolean
	 */
	public function custom_breadcrumbs( $breadcrumbs ) {
		if ( empty( $this->data['enable_bulk_coupon'] ?? '' ) ) {
			return $breadcrumbs;
		}

		$get_page = ( ! empty( $_GET['page'] ) ) ? sanitize_text_field( wp_unslash( $_GET['page'] ) ) : ''; // phpcs:ignore WordPress.Security.NonceVerification.Recommended

		if ( 'rtsb-smart-coupons' === $get_page ) {
			$breadcrumbs[] = esc_html__( 'Bulk Coupon Generator', 'shopbuilder-pro' );
		}
		return $breadcrumbs;
	}

	/**
	 * Renders a notice based on the result of the coupon generation process.
	 *
	 * @return void
	 */
	private function render_notice() {
		$message = '';
		$type    = 'success updated';
		$success = isset( $_GET['rtsb_coupons_generated'] ) && 'success' === $_GET['rtsb_coupons_generated']; // phpcs:ignore WordPress.Security.NonceVerification.Recommended
		$error   = isset( $_GET['rtsb_coupons_generated'] ) && 'error' === $_GET['rtsb_coupons_generated']; // phpcs:ignore WordPress.Security.NonceVerification.Recommended

		if ( $success ) {
			$message = esc_html__( 'Coupon codes generated successfully.', 'shopbuilder-pro' );
		} elseif ( $error ) {
			$message = esc_html__( 'The number of coupons cannot be zero (0). Please try again.', 'shopbuilder-pro' );
			$type    = 'error';
		}

		if ( ! empty( $message ) ) {
			?>
			<div id="message" class="notice notice-<?php echo esc_attr( $type ); ?>">
				<p><?php echo esc_html( $message ); ?></p>

				<?php
				if ( $success ) {
					?>
					<p><a href="<?php echo esc_url( admin_url( 'edit.php?post_type=shop_coupon' ) ); ?>">&larr; <?php esc_html_e( 'Back to Coupons', 'shopbuilder-pro' ); ?></a></p>
					<?php
				}
				?>
			</div>
			<?php
		}
	}

	/**
	 * Adds custom screen ID for the Coupons page.
	 *
	 * @param array $screen_ids The existing screen IDs.
	 *
	 * @return array
	 */
	public function add_custom_screen_id( $screen_ids ) {
		$screen_ids[] = 'marketing_page_rtsb-smart-coupons';

		return $screen_ids;
	}
}
