<?php
/**
 * Main FilterHooks class.
 *
 * @package RadiusTheme\SB
 */

namespace RadiusTheme\SBPRO\Modules\SalesNotification;

use RadiusTheme\SB\Helpers\Fns;
use RadiusTheme\SBPRO\Traits\SingletonTrait;

defined( 'ABSPATH' ) || exit();

/**
 * Main FilterHooks class.
 */
class Notification {
	/**
	 * Singleton Trait.
	 */
	use SingletonTrait;

	/**
	 * @var array|mixed
	 */
	private array $options;

	/**
	 * Asset Handle
	 *
	 * @var string
	 */
	private $handle = 'rtsb-sales-notification';


	/**
	 * Notifications hooks.
	 */
	private function __construct() {
		$this->options = Fns::get_options( 'modules', 'sales_notification' );

		add_action( 'wp_footer', [ $this, 'toster_script' ], 99 );
		/**
		 * Public scripts.
		 */
		add_action( 'wp_enqueue_scripts', [ $this, 'enqueue_public_scripts' ], 99 );
	}

	/**
	 * @return void
	 */
	public function enqueue_public_scripts() {
		// Enqueue assets.
		$this->handle = Fns::enqueue_module_assets(
			$this->handle,
			'sales-notification',
			[
				'context' => rtsbpro(),
				'version' => RTSBPRO_VERSION,
			]
		);

		$options     = $this->options;
		$dynamic_css = '';
		$pre         = '.rtsb-toast-container .rtsb-notifications ';

		if ( ! empty( $options['notification_background_color'] ) ) {
			$dynamic_css .= $pre . '.rtsb-notifications-item{';
			$dynamic_css .= 'background-color:' . $options['notification_background_color'];
			$dynamic_css .= '}';
		}

		if ( ! empty( $options['notification_link_color'] ) ) {
			$dynamic_css .= $pre . ' a{';
			$dynamic_css .= 'color:' . $options['notification_link_color'];
			$dynamic_css .= '}';
		}

		if ( ! empty( $options['notification_link_hover_color'] ) ) {
			$dynamic_css .= $pre . ' a:hover{';
			$dynamic_css .= 'color:' . $options['notification_link_hover_color'];
			$dynamic_css .= '}';
		}

		if ( ! empty( $options['notification_text_color'] ) ) {
			$dynamic_css .= $pre . ' .rtsb-notifications-item{';
			$dynamic_css .= 'color:' . $options['notification_text_color'];
			$dynamic_css .= '}';
		}

		if ( ! empty( $options['notification_name_address_color'] ) ) {
			$dynamic_css .= $pre . '.rtsb-notifications-item-address{';
			$dynamic_css .= 'color:' . $options['notification_name_address_color'];
			$dynamic_css .= '}';
		}

		if ( ! empty( $options['notification_purchased_time_color'] ) ) {
			$dynamic_css .= $pre . '.rtsb-notifications-item-message{';
			$dynamic_css .= 'color:' . $options['notification_purchased_time_color'];
			$dynamic_css .= '}';
		}

		if ( ! empty( $options['notification_cross_button_color'] ) ) {
			$dynamic_css .= $pre . '.rtsb-notifications-button{';
			$dynamic_css .= 'color:' . $options['notification_cross_button_color'];
			$dynamic_css .= '}';
		}

		if ( ! empty( $dynamic_css ) ) {
			wp_add_inline_style( $this->handle, $dynamic_css );
		}
	}
	/**
	 * Get notifications data
	 *
	 * @param array $args Args.
	 * @param array $setings Settings.
	 *
	 * @return array
	 */
	public function get_notifications_data( $args = [], $setings = [] ) {
		$args = wp_parse_args(
			$args,
			[
				'limit'   => 5,
				'orderby' => 'date',
				'order'   => 'DESC',
				'status'  => [ 'wc-completed' ],
			]
		);

		$expiration = ! empty( $setings['notification_cache'] ) ? absint( $setings['notification_cache'] ) : HOUR_IN_SECONDS;
		$cache_id   = 'rtsb_get_notifications_data';

		$notifications = get_transient( $cache_id );

		if ( empty( $notifications ) ) {
			$notifications = [];
			$query         = new \WC_Order_Query( $args );
			$orders        = $query->get_orders();
			foreach ( $orders as $order ) {
				$user = $order->get_user();
				if ( ! $user ) {
					continue;
				}
				$user_id    = $user->get_id();
				$order_time = $order->get_date_created();
				$_id        = $order->get_id();

				$billingAddress = $order->get_address( 'billing' );

				$notifications[ $_id ]['user_nicename'] = $user->user_nicename;
				$notifications[ $_id ]['display_name']  = $user->display_name;
				$notifications[ $_id ]['user_image']    = get_avatar( $user_id );

				$notifications[ $_id ]['user_address'] = $billingAddress['city'] . ', ' . $billingAddress['country'];

				$notifications[ $_id ]['order_time'] = sprintf(
				/* translators: time */
					esc_html__( '%s ago', 'shopbuilder-pro' ),
					human_time_diff( strtotime( $order_time ), current_time( 'timestamp' ) ) // phpcs:ignore WordPress.DateTime.CurrentTimeTimestamp.Requested
				);

				$items = $order->get_items();
				foreach ( $items as $item ) {
					$product_id = $item->get_product_id();
					$product    = wc_get_product( $product_id );
					if ( ! $product ) {
						continue;
					}
					$product_title = $product->get_title();
					$product_url   = $product->get_permalink();

					$product_image = wp_get_attachment_image( $product->get_image_id(), 'thumbnail' );

					$notifications[ $_id ]['products'][ $product_id ] = [
						'product_title' => $product_title,
						'product_url'   => $product_url,
						'product_image' => '',
					];
					if ( $product_image ) {
						$notifications[ $_id ]['products'][ $product_id ]['product_image'] = $product_image;
					}
				}
			}
			set_transient( $cache_id, $notifications, $expiration );
		}

		return is_array( $notifications ) ? $notifications : [];
	}

	/**
	 * @return void
	 */
	public function toster_script() {
		$options = $this->options;

		$includes_pages = Fns::multiselect_settings_field_value( $options['notification_include_pages'] ?? '' );

		// phpcs:ignore WordPress.PHP.StrictInArray.MissingTrueStrict
		if ( is_array( $includes_pages ) && count( $includes_pages ) && ! in_array( get_the_ID(), $includes_pages ) ) {
			return;
		}

		$exclude_pages = Fns::multiselect_settings_field_value( $options['notification_exclude_pages'] ?? '' );

		// phpcs:ignore WordPress.PHP.StrictInArray.MissingTrueStrict
		if ( is_array( $exclude_pages ) && count( $exclude_pages ) && in_array( get_the_ID(), $exclude_pages ) ) {
			return;
		}

		$include_archive = Fns::multiselect_settings_field_value( $options['notification_include_archive'] ?? '' );
		if ( is_array( $include_archive ) && count( $include_archive ) ) {
			// phpcs:ignore WordPress.PHP.StrictInArray.MissingTrueStrict
			if ( is_home() && ! in_array( 'post', $include_archive ) ) {
				return;
				// phpcs:ignore WordPress.PHP.StrictInArray.MissingTrueStrict
			} elseif ( ( is_archive() && ! in_array( get_queried_object()->name, $include_archive ) ) || ( is_single() && ! is_singular( $include_archive ) ) ) {
				return;
			}
		}

		$exclude_archive = Fns::multiselect_settings_field_value( $options['notification_exclude_archive'] ?? '' );

		if ( is_array( $exclude_archive ) && count( $exclude_archive ) ) {
			// phpcs:ignore WordPress.PHP.StrictInArray.MissingTrueStrict
			if ( is_home() && in_array( 'post', $exclude_archive ) ) {
				return;
				// phpcs:ignore WordPress.PHP.StrictInArray.MissingTrueStrict
			} elseif ( ( is_archive() && in_array( get_queried_object()->name, $exclude_archive ) ) || ( is_single() && is_singular( $exclude_archive ) ) ) {
				return;
			}
		}

		wp_enqueue_style( 'dashicons' );

		$keep_onscreen      = $options['notification_keep_onscreen'] ?? 5000;
		$notification_delay = $options['notification_delay'] ?? 3000;
		$position           = $options['notification_position'] ?? 'bottom_left';
		$animate            = $options[ 'notification_animate_for_' . $position ] ?? 'fadein';
		$close_button       = isset( $options['notification_show_close_button'] ) ? $options['notification_show_close_button'] : true;
		$show_on_mobile     = $options['notification_show_on_mobile'] ?? false;

		$product_link = $options['notification_enable_product_link'] ?? true;

		$is_processing_order = isset( $options['notification_show_processing_order'] ) ? $options['notification_show_processing_order'] : true;
		$notification_image  = $options['notification_image'] ?? 'products-image';
		$number_of_popups    = $options['notification_number_of_popups'] ?? 10;

		$title_max_words    = $options['notification_product_title_max_words'] ?? 10;
		$notification_cache = $options['notification_cache'] ?? 10;

		$notification_style = $options['notification_layout'] ?? 'style-1';

		$display_user_name = $options['notification_display_user_name'] ?? '';

		$notification_data = [
			'delay'        => absint( $notification_delay ),
			'stayOnScreen' => absint( $keep_onscreen ),
			'showOnMobile' => boolval( $show_on_mobile ),
		];

		$settings = [
			'notification_cache' => $notification_cache,
		];

		?>
		<div class="rtsb-toast-container" data-settings="<?php echo esc_attr( wp_json_encode( $notification_data ) ); ?>">
			<?php

			$data = [
				'template'           => 'sales-notification/notification',
				'options'            => $options,
				'close_button'       => $close_button,
				'display_user_name'  => $display_user_name,
				'notification_style' => $notification_style,
				'is_display_image'   => 'no-image' !== $notification_image,
				'cross_button_color' => $options['notification_cross_button_color'] ?? '#000',
			];
			$args = [
				'limit' => $number_of_popups,
			];
			if ( $is_processing_order ) {
				$args['status'] = [ 'wc-completed', 'wc-processing' ];
			}

			$thenotice = $this->get_notifications_data( $args, $settings );

			foreach ( $thenotice as $orderid => $item ) {
				if ( empty( $item['products'] ) ) {
					continue;
				}
				$data['item']   = $item;
				$productsRandom = $item['products'];
				shuffle( $productsRandom );

				$data['display_product']                  = reset( $productsRandom );
				$data['display_image']                    = 'products-image' === $notification_image ? $data['display_product']['product_image'] : $item['user_image'];
				$data['product_link']                     = $product_link;
				$title                                    = wp_trim_words( $data['display_product']['product_title'], $title_max_words, '...' );
				$data['display_product']['product_title'] = $title;
				$data                                     = apply_filters( 'rtsb_get_notifications_data', $data );
				?>
				<div class="rtsb-notifications rtsb-slider-<?php echo esc_attr( $position . '_' . $animate ) . ' ' . esc_attr( 'rtsb-notification-' . $notification_style ); ?>">
					<?php Fns::load_template( $data['template'], $data, false, '', rtsbpro()->get_plugin_template_path() ); ?>
				</div>
			<?php } ?>
		</div>
		<?php
	}
}
