<?php
/**
 * Main FilterHooks class.
 *
 * @package RadiusTheme\SB
 */

namespace RadiusTheme\SBPRO\Modules\QuickCheckout;

use RadiusTheme\SBPRO\Helpers\FnsPro;
use WC_Product;
use RadiusTheme\SB\Helpers\Fns;
use RadiusTheme\SBPRO\Traits\SingletonTrait;
use RadiusTheme\SBPRO\Modules\AddOns\AddOnsFns;

defined( 'ABSPATH' ) || exit();

/**
 * Main FilterHooks class.
 */
class CheckoutFrontEnd {
	/**
	 * Singleton Trait.
	 */
	use SingletonTrait;

	/**
	 * @var array
	 */
	private $cache = [];

	/**
	 * @var array|mixed
	 */
	private array $options;

	/**
	 * Asset Handle
	 *
	 * @var string
	 */
	private $handle = 'rtsb-quick-checkout';

	/**
	 * Constructor
	 */
	private function __construct() {
		$this->options = Fns::get_options( 'modules', 'quick_checkout' );

		add_action( 'rtsb/modules/quick_checkout/frontend/display', [ $this, 'frontend_display' ] );
		// Add do_action( 'rtsb/modules/quick_checkout/frontend/display' ); for display anywhere inside product query.
		add_action( 'rtsb/modules/quick_checkout/print_button', [ $this, 'print_button' ] );
		// ShortCode.
		add_shortcode( 'rtsb_quick_checkout_button', [ $this, 'quick_checkout_shortcode' ] );

		add_action( 'wp_enqueue_scripts', [ $this, 'enqueue_public_scripts' ], 99 );

		$this->attach_button();
	}

	/**
	 * Enqueue public scripts.
	 *
	 * @return void
	 */
	public function enqueue_public_scripts() {
		// Enqueue assets.
		$this->handle = Fns::enqueue_module_assets(
			$this->handle,
			'quick-checkout',
			[
				'context' => rtsbpro(),
				'version' => RTSBPRO_VERSION,
			]
		);

		if ( ! apply_filters( 'rtsb/module/quick_checkout/show_button', true ) ) {
			return;
		}

		$pre         = '.rtsb_theme_' . rtsb()->current_theme;
		$dynamic_css = '';

		$dynamic_css .= $pre . ' .rtsb-quick-checkout-btn{';
		$dynamic_css .= 'display: inline-flex !important;margin: 0 !important;align-items: center !important;text-decoration: none !important;';
		$dynamic_css .= '}';

		if ( ! empty( $this->options['loop_quick_checkout_button_height'] ) && absint( $this->options['loop_quick_checkout_button_height'] ) ) {
			$dynamic_css .= $pre . ' .rtsb-checkout-button-products{';
			$dynamic_css .= 'height:' . absint( $this->options['loop_quick_checkout_button_height'] ) . 'px !important;';
			$dynamic_css .= '}';
		}

		if ( ! empty( $this->options['loop_quick_checkout_button_padding'] ) ) {
			$dynamic_css .= $pre . ' .rtsb-checkout-button-products{';
			$dynamic_css .= 'padding:' . $this->options['loop_quick_checkout_button_padding'] . ' !important;';
			$dynamic_css .= '}';
		}
		if ( ! empty( $this->options['loop_quick_checkout_button_margin'] ) ) {
			$dynamic_css .= $pre . ' .rtsb-checkout-button-products{';
			$dynamic_css .= 'margin:' . $this->options['loop_quick_checkout_button_margin'] . ' !important;';
			$dynamic_css .= '}';
		}

		if ( ! empty( $this->options['product_quick_checkout_button_height'] ) && absint( $this->options['product_quick_checkout_button_height'] ) ) {
			$dynamic_css .= $pre . ' .rtsb-checkout-button-product{';
			$dynamic_css .= 'height:' . absint( $this->options['product_quick_checkout_button_height'] ) . 'px !important;';
			$dynamic_css .= '}';
		}

		if ( ! empty( $this->options['product_quick_checkout_button_padding'] ) ) {
			$dynamic_css .= $pre . ' .rtsb-checkout-button-product{';
			$dynamic_css .= 'padding:' . $this->options['product_quick_checkout_button_padding'] . ' !important;';
			$dynamic_css .= '}';
		}
		if ( ! empty( $this->options['product_quick_checkout_button_margin'] ) ) {
			$dynamic_css .= $pre . ' .rtsb-checkout-button-product{';
			$dynamic_css .= 'margin:' . $this->options['product_quick_checkout_button_margin'] . ' !important;';
			$dynamic_css .= '}';
		}

		if ( ! empty( $this->options['quick_checkout_text_color'] ) ) {
			$dynamic_css .= $pre . ' .rtsb-quick-checkout-btn{';
			$dynamic_css .= 'color:' . $this->options['quick_checkout_text_color'] . ' !important;';
			$dynamic_css .= '}';
		}

		if ( ! empty( $this->options['quick_checkout_text_hover_color'] ) ) {
			$dynamic_css .= $pre . ' .rtsb-quick-checkout-btn:hover{';
			$dynamic_css .= 'color:' . $this->options['quick_checkout_text_hover_color'] . ' !important;';
			$dynamic_css .= '}';
		}

		if ( ! empty( $this->options['quick_checkout_background_color'] ) ) {
			$dynamic_css .= $pre . ' .rtsb-quick-checkout-btn{';
			$dynamic_css .= 'background-color:' . $this->options['quick_checkout_background_color'] . ' !important;';
			$dynamic_css .= '}';
		}

		if ( ! empty( $this->options['quick_checkout_background_hover_color'] ) ) {
			$dynamic_css .= $pre . ' .rtsb-quick-checkout-btn:hover{';
			$dynamic_css .= 'background-color:' . $this->options['quick_checkout_background_hover_color'] . ' !important;';
			$dynamic_css .= '}';
		}

		if ( ! empty( $dynamic_css ) ) {
			wp_add_inline_style( $this->handle, $dynamic_css );
		}
	}

	/**
	 * Add the "Add to Wishlist" button. Needed to use in wp_head hook.
	 *
	 * @return void
	 * @since 1.0.0
	 */
	public function attach_button() {

		$product_hook_priority = $this->options['product_hook_priority'] ?? null;
		$product_hook_priority = is_int( $product_hook_priority ) ? $product_hook_priority : null;

		$positions = apply_filters(
			'rtsb/module/quick_checkout/product_page_button_positions',
			[
				'before_add_to_cart' => [
					'hook'     => 'woocommerce_before_add_to_cart_button',
					'priority' => $product_hook_priority ?? 20,
				],
				'after_add_to_cart'  => [
					'hook'     => 'woocommerce_single_product_summary',
					'priority' => $product_hook_priority ?? 31,
				],
				'after_thumbnail'    => [
					'hook'     => 'woocommerce_before_single_product_summary',
					'priority' => $product_hook_priority ?? 21,
				],
				'after_summary'      => [
					'hook'     => 'woocommerce_after_single_product_summary',
					'priority' => $product_hook_priority ?? 11,
				],
				'after_short_desc'   => [
					'hook'     => 'woocommerce_single_product_summary',
					'priority' => $product_hook_priority ?? 21,
				],

				'custom'             => [
					'hook'     => $this->options['product_custom_hook_name'] ?? '',
					'priority' => $product_hook_priority ?? 10,
				],
			]
		);

		// Add the link "Add to wishlist".

		$product_button_enable   = boolval( $this->options['show_quick_checkout_product_page'] ?? null );
		$product_button_position = $this->options['product_quick_checkout_position'] ?? 'after_add_to_cart';

		if ( $product_button_enable && 'shortcode' !== $product_button_position && isset( $positions[ $product_button_position ]['hook'] ) ) {
			add_action( $positions[ $product_button_position ]['hook'], [ $this, 'frontend_display' ], isset( $positions[ $product_button_position ]['priority'] ) ? absint( $positions[ $product_button_position ]['priority'] ) : '' );
		}

		// check if Add to wishlist button is enabled for loop.
		$loop_btn_enable = boolval( $this->options['show_quick_checkout_on_loop'] ?? null );
		if ( $product_button_enable || $loop_btn_enable ) {
			wp_enqueue_script( 'wc-checkout' );
		}
		if ( ! $loop_btn_enable ) {
			return;
		}

		$loop_hook_priority = $this->options['loop_hook_priority'] ?? null;
		$loop_hook_priority = is_int( $loop_hook_priority ) ? $loop_hook_priority : null;

		$positions = apply_filters(
			'rtsb/module/quick_checkout/product_loop_button_positions',
			[
				'custom' => [
					'hooks'    => $this->options['loop_custom_hook_name'] ?? '',
					'priority' => $loop_hook_priority ?? 10,
				],
			]
		);

		// Add the link "Add to wishlist" in the loop.
		$loop_button_position = $this->options['loop_quick_checkout_position'] ?? '';
		if ( 'shortcode' == $loop_button_position || empty( $loop_button_position ) ) {
			return;
		}
		if ( 'custom' == $loop_button_position ) {
			if ( empty( $positions[ $loop_button_position ]['hooks'] ) ) {
				return;
			}
			add_action( trim( $positions[ $loop_button_position ]['hooks'] ), [ $this, 'frontend_display' ], $positions[ $loop_button_position ]['priority'] ?? '' );
		} else {
			add_filter( 'woocommerce_loop_add_to_cart_link', [ $this, 'loop_add_to_cart_link' ], $loop_hook_priority ?? 11, 2 );
		}
	}

	/**
	 * Print "Add to compare" button
	 *
	 * @param string $cart_link Cart link.
	 * @param object $product Product object.
	 *
	 * @return string
	 */
	public function loop_add_to_cart_link( $cart_link, $product ) {

		// Add the link "Add to wishlist" in the loop.
		$loop_button_position = $this->options['loop_quick_checkout_position'] ?? 'after_add_to_cart';

		$button = shortcode_exists( 'rtsb_quick_checkout_button' ) ? do_shortcode( '[rtsb_quick_checkout_button product_id="' . $product->get_id() . '"/]' ) : '';

		switch ( $loop_button_position ) {
			case 'before_add_to_cart':
				$cart_link = $button . $cart_link;
				break;
			case 'after_add_to_cart':
				$cart_link = $cart_link . $button;
				break;
			default:
				$cart_link = $cart_link . $button;
		}

		return '<div class="rtsb-checkout-and-add-to-cart">' . $cart_link . '</div>';
	}

	/**
	 * Print "Add to compare" button
	 *
	 * @return void
	 */
	public function frontend_display() {

		if ( ! apply_filters( 'rtsb/module/quick_checkout/show_button', true ) ) {
			return;
		}
		global $product;
		if ( $product instanceof WC_Product ) {
			do_action( 'rtsb/modules/quick_checkout/print_button', $product->get_id() );
		}
	}


	/**
	 * Wishlist button Shortcode callable function
	 *
	 * @param array $atts Shortcode attributes.
	 *
	 * @return string [HTML]
	 */
	public function quick_checkout_shortcode( $atts ) {
		$default_atts = [
			'product_id' => null,
		];
		$atts         = shortcode_atts( $default_atts, $atts );

		global $product;
		$cache_key = 'qc_product_';
		if ( ! empty( $atts['product_id'] ) ) {
			$cache_key = 'qc_product_' . absint( $atts['product_id'] );
			if ( isset( $this->cache[ $cache_key ] ) ) {
				return $this->cache[ $cache_key ];
			}
		}
		$need_reset_product = false;
		if ( $product instanceof WC_Product ) {
			$cache_key = 'qc_product_' . $product->get_id();
			if ( isset( $this->cache[ $cache_key ] ) ) {
				return $this->cache[ $cache_key ];
			}
		}

		if ( ! $product instanceof WC_Product && ! empty( $atts['product_id'] ) ) {
			$cache_key = 'qc_product_' . $atts['product_id'];
			if ( isset( $this->cache[ $cache_key ] ) ) {
				return $this->cache[ $cache_key ];
			}

			$product            = wc_get_product( $atts['product_id'] );
			$GLOBALS['product'] = $product;
			$need_reset_product = true;
		}

		ob_start();

		do_action( 'rtsb/modules/quick_checkout/frontend/display' );

		$button = ob_get_clean();

		$this->cache[ $cache_key ] = $button;

		if ( $need_reset_product ) {
			unset( $GLOBALS['product'] );
		}
		return $button;
	}

	/**
	 * Print "Add to compare" button
	 *
	 * @param int $product_id Product ID.
	 *
	 * @return void
	 */
	public function print_button( $product_id = 0 ) {
		global $product, $woocommerce_loop;

		if ( ! $product instanceof WC_Product && $product_id ) {
			$product = wc_get_product( $product_id );
		}
		if ( $product->is_type( 'grouped' ) || $product->is_type( 'external' ) ) {
			return;
		}

		$product_id = $product->get_id();
		// Product Object.
		if ( ! is_product() && $product->is_type( 'variable' ) && ! ( function_exists( 'rtwpvsp' ) && rtwpvs()->get_option( 'archive_swatches' ) ) ) {
			return;
		}

		// Need Support For Product Addons.
		if ( FnsPro::is_module_active( 'product_add_ons' ) && AddOnsFns::product_has_addons( $product ) ) {
			 return;
		}

		$classes = [ 'button', 'rtsb-quick-checkout-product-id-' . $product_id ];
		if ( is_shop() || in_array( ( $woocommerce_loop['name'] ?? '' ), [ 'related', 'up-sells', 'cross-sells' ] ) ) { // phpcs:ignore WordPress.PHP.StrictInArray.MissingTrueStrict

			if ( is_product() && $product->is_type( 'variable' ) && ! ( function_exists( 'rtwpvsp' ) && rtwpvs()->get_option( 'archive_swatches' ) ) ) {
				 return;
			}

			$classes[] = ! empty( $this->options['show_quick_checkout_on_loop'] ) ? 'rtsb-checkout-button-products' : '';
		} else {
			$classes[] = ! empty( $this->options['show_quick_checkout_product_page'] ) ? 'rtsb-checkout-button-product' : '';
		}
		$args = [
			'quick_checkout' => $product_id,
			'quantity'       => 1,
		];
		if ( $product->is_type( 'variable' ) ) {
			$classes[] = 'rtsb-has-variation';
			$classes[] = 'rtsb-disabled';
		}

		$checkout_url = add_query_arg( $args, wc_get_cart_url() );

		$classes = apply_filters( 'rtsb/quick_checkout/parent/class', $classes, $product, $woocommerce_loop );

		$button_text_option = $this->options['quick_checkout_button_text'] ?? '';
		$button_text        = apply_filters( 'rtsb/quick_checkout/button/text', $button_text_option );
		$icon_html          = apply_filters( 'rtsb/quick_checkout/button/icon', '' );
		$tooltip_attr       = '';

		$button = apply_filters(
			'rtsb/quick_checkout/button_data',
			[
				'wrapper_class' => $classes,
				'button_text'   => $button_text,
				'button_icon'   => $icon_html,
			],
			$this->options,
			$classes,
			$product,
			$woocommerce_loop
		);

		if ( empty( $button_text ) ) {
			$tooltip_attr = ' title="' . esc_attr( $button_text_option ) . '"';
		}
		$modalClass = '';
		if ( 'popups' === ( $this->options['quick_checkout_button_action'] ?? 'redirect' ) ) {
			$modalClass = 'rtsb-checkout-button-popups';
		}

		?>
		<a class="rtsb-quick-checkout-btn rtsb-action-btn <?php echo esc_attr( $modalClass ); ?> <?php echo ! empty( $button['wrapper_class'] ) ? esc_attr( implode( ' ', $button['wrapper_class'] ) ) : ''; ?>" rel="nofollow"
		   href="<?php echo esc_url( $checkout_url ); ?>"
			<?php Fns::print_html( $tooltip_attr ); ?>
		   data-title="<?php echo esc_attr( ! empty( $button['button_text'] ) ? $button['button_text'] : __( 'Buy Now', 'shopbuilder-pro' ) ); ?>" href="<?php echo esc_url( get_permalink( $product_id ) ); ?>" aria-label="<?php echo esc_attr( $button_text_option ); ?>">
			<?php if ( ! empty( $button['button_icon'] ) ) { ?>
				<span class="icon">
					<?php Fns::print_html( $button['button_icon'] ); ?>
				</span>
			<?php } ?>
			<?php if ( ! empty( $button['button_text'] ) ) { ?>
				<span class="text"><?php echo esc_html( $button['button_text'] ); ?></span>
			<?php } ?>
		</a>

		<?php
	}
}
