<?php
/**
 * Main FilterHooks class.
 *
 * @package RadiusTheme\SB
 */

namespace RadiusTheme\SBPRO\Modules\ProductSizeChart;

use RadiusTheme\SB\Helpers\Cache;
use RadiusTheme\SB\Helpers\Fns;
use WC_Product;

defined( 'ABSPATH' ) || exit();

/**
 * Main FilterHooks class.
 */
class SizeChartFns {

	/**
	 * @return array
	 */
	public static function settings_field() {
		return apply_filters(
			'rtsb/module/product_size_chart/fields',
			[
				// Single Product Page.
				'product_size_chart_position'              => [
					'id'      => 'product_size_chart_position',
					'label'   => esc_html__( 'Product Size Chart Position', 'shopbuilder-pro' ),
					'help'    => esc_html__( 'You can manage Size Chart Button position in single product page.', 'shopbuilder-pro' ),
					'type'    => 'select',
					'value'   => 'custom',
					'options' => [
						'before_add_to_cart' => esc_html__( 'Before Add To Cart', 'shopbuilder-pro' ),
						'after_add_to_cart'  => esc_html__( 'After Add To Cart', 'shopbuilder-pro' ),
						'after_thumbnail'    => esc_html__( 'After Image', 'shopbuilder-pro' ),
						'after_summary'      => esc_html__( 'After Summary', 'shopbuilder-pro' ),
						'after_short_desc'   => esc_html__( 'After Short Description', 'shopbuilder-pro' ),
						'product_tab'        => esc_html__( 'Product Tab', 'shopbuilder-pro' ),
						'shortcode'          => esc_html__( 'Use Shortcode', 'shopbuilder-pro' ),
						'custom'             => esc_html__( 'Custom Position', 'shopbuilder-pro' ),
					],
					'tab'     => 'general',
				],

				'product_size_chart_shortcode'             => [
					'type'       => 'raw',
					'label'      => ' ',
					'html'       => sprintf(
					/* translators: 1: The shortcode.*/
						esc_html__( 'Choose where to show button on the product page. Copy this shortcode %1$s and paste it where you want to show the button.', 'shopbuilder-pro' ),
						'<code>[rtsb_product_size_chart]</code>'
					),
					'dependency' => [
						'rules' => [
							[
								'item'     => 'modules.product_size_chart.product_size_chart_position',
								'value'    => 'shortcode',
								'operator' => '==',
							],
						],
					],
					'tab'        => 'general',
				],
				'product_custom_hook_name'                 => [
					'id'         => 'product_custom_hook_name',
					'type'       => 'text',
					'label'      => esc_html__( 'Product Hook Name', 'shopbuilder-pro' ),
					'tab'        => 'general',
					'help'       => sprintf(
					/* translators: 1: The shortcode.*/
						esc_html__( 'Or Copy the php code %1$s and paste it in your product query where you want to show the button.', 'shopbuilder-pro' ),
						"<code>&lt;?php do_action( 'rtsb/modules/product_size_chart/frontend/display' ); ?&gt;</code>"
					),
					'dependency' => [
						'rules' => [
							[
								'item'     => 'modules.product_size_chart.product_size_chart_position',
								'value'    => 'custom',
								'operator' => '==',
							],
						],
					],
				],
				'product_hook_priority'                    => [
					'id'         => 'product_hook_priority',
					'type'       => 'number',
					'size'       => 'small',
					'min'        => 0,
					'max'        => 999,
					'label'      => esc_html__( 'Hook Priority', 'shopbuilder-pro' ),
					'tab'        => 'general',
					'dependency' => [
						'rules' => [
							[
								'item'     => 'modules.product_size_chart.product_size_chart_position',
								'value'    => 'custom',
								'operator' => '==',
							],
						],
					],
				],
				'size_chart_button_action'                 => [
					'id'         => 'size_chart_button_action',
					'type'       => 'select',
					'value'      => 'popups',
					'label'      => esc_html__( 'Size Chart Action', 'shopbuilder-pro' ),
					'options'    => [
						'general' => esc_html__( 'General View', 'shopbuilder-pro' ),
						'popups'  => esc_html__( 'Open In a modal', 'shopbuilder-pro' ),
					],
					'help'       => esc_html__( 'size-chart Button Behavior After Click.', 'shopbuilder-pro' ),
					'dependency' => [
						'rules' => [
							[
								'item'     => 'modules.product_size_chart.product_size_chart_position',
								'value'    => 'product_tab',
								'operator' => '!=',
							],
						],
					],
					'tab'        => 'general',
				],
				'size_chart_button_text'                   => [
					'id'    => 'size_chart_button_text',
					'type'  => 'text',
					'label' => esc_html__( 'Size Chart Button Text', 'shopbuilder-pro' ),
					'tab'   => 'general',
					'value' => esc_html__( 'Size Chart', 'shopbuilder-pro' ),
					'help'  => esc_html__( 'Enter the size-chart button text.', 'shopbuilder-pro' ),
				],
				// End Single Product Page.
				'chart_field_intro'                        => [
					'id'   => 'chart_field_intro',
					'type' => 'description',
					'text' => esc_html__( 'To add new size charts, simply click on the \'Add New\' button below.', 'shopbuilder-pro' ),
					'tab'  => 'chart',
				],
				'chart_field'                              => [
					'id'     => 'chart_field',
					'type'   => 'repeaters',
					'label'  => '',
					'tab'    => 'chart',
					'repeat' => [
						'title'                 => [
							'id'          => 'title',
							'label'       => esc_html__( 'Title', 'shopbuilder-pro' ),
							'type'        => 'text',
							'placeholder' => 'Chart 1',
							'value'       => 'Chart 1',
						],
						'apply_for'             => [
							'id'      => 'apply_for',
							'type'    => 'select',
							'value'   => 'product',
							'label'   => esc_html__( 'Apply For', 'shopbuilder-pro' ),
							'options' => [
								'product'     => esc_html__( 'Product', 'shopbuilder-pro' ),
								'product_cat' => esc_html__( 'Product Categories', 'shopbuilder-pro' ),
							],
						],

						'applicable_products'   => [
							'id'              => 'applicable_products',
							'type'            => 'search_and_multi_select',
							'label'           => esc_html__( 'Applicable Products', 'shopbuilder-pro' ),
							'help'            => esc_html__( 'Please search and add the product.', 'shopbuilder-pro' ),
							'placeholder'     => esc_html__( 'Search Products', 'shopbuilder-pro' ),
							// Function will receive one params for search string.
							'func_with_param' => [ Fns::class, 'get_post_types', [ 'post_type' => 'product' ] ],
							'options'         => Fns::get_post_types( null, [ 'post_type' => 'product' ] ),
							'dependency'      => [
								'rules' => [
									[
										'item'     => 'modules.product_size_chart.chart_field.apply_for',
										'value'    => 'product',
										'operator' => '==',
									],
								],
							],
						],

						'applicable_categories' => [
							'id'              => 'applicable_categories',
							'type'            => 'search_and_multi_select',
							'label'           => esc_html__( 'Applicable Categories', 'shopbuilder-pro' ),
							'help'            => esc_html__( 'Please enter the product category.', 'shopbuilder-pro' ),
							'placeholder'     => esc_html__( 'Search Category', 'shopbuilder-pro' ),
							// Function will receive one params for search string.
							'func_with_param' => [
								Fns::class,
								'products_category_query',
							],
							'options'         => Fns::products_category_query(),
							'dependency'      => [
								'rules' => [
									[
										'item'     => 'modules.product_size_chart.chart_field.apply_for',
										'value'    => 'product_cat',
										'operator' => '==',
									],
								],
							],
						],

						'exclude_product'       => [
							'id'              => 'exclude_product',
							'type'            => 'search_and_multi_select',
							'label'           => esc_html__( 'Exclude Products', 'shopbuilder-pro' ),
							'help'            => esc_html__( 'Please add the product.', 'shopbuilder-pro' ),
							'placeholder'     => esc_html__( 'Search Products', 'shopbuilder-pro' ),
							// Function will receive one params for search string.
							'func_with_param' => [ Fns::class, 'get_post_types', [ 'post_type' => 'product' ] ],
							'options'         => Fns::get_post_types( null, [ 'post_type' => 'product' ] ),
						],

						'size_chart_image'      => [
							'id'    => 'size_chart_image',
							'type'  => 'fileupload',
							'label' => esc_html__( 'Size Chart Image', 'shopbuilder-pro' ),
							'help'  => esc_html__( 'Upload size chart image.', 'shopbuilder-pro' ),
							'tab'   => 'general',
						],
					],
				],

				// Style Tab.
				'size_chart_button_height'                 => [
					'id'         => 'size_chart_button_height',
					'type'       => 'number',
					'size'       => 'small',
					'min'        => 0,
					'max'        => 999,
					'label'      => esc_html__( 'Product Page Button Height (px)', 'shopbuilder-pro' ),
					'help'       => esc_html__( 'Button Height (px).', 'shopbuilder-pro' ),
					'dependency' => [
						'rules' => [
							[
								'item'     => 'modules.product_size_chart.size_chart_button_action',
								'value'    => 'popups',
								'operator' => '==',
							],
							[
								'item'     => 'modules.product_size_chart.product_size_chart_position',
								'value'    => 'product_tab',
								'operator' => '!=',
							],
						],
					],
					'tab'        => 'style',
				],

				'size_chart_wrapper_padding'               => [
					'id'    => 'size_chart_wrapper_padding',
					'type'  => 'text',
					'size'  => 'small',
					'min'   => 0,
					'max'   => 999,
					'label' => esc_html__( 'Size Chart Padding', 'shopbuilder-pro' ),
					'help'  => esc_html__( 'Size Chart Padding. Example: 0 15px 15px 15px', 'shopbuilder-pro' ),
					'tab'   => 'style',
				],

				'size_chart_wrapper_margin'                => [
					'id'    => 'size_chart_wrapper_margin',
					'type'  => 'text',
					'size'  => 'small',
					'min'   => 0,
					'max'   => 999,
					'label' => esc_html__( 'Size Chart Margin', 'shopbuilder-pro' ),
					'help'  => esc_html__( 'Size Chart Margin. Example: 0 15px 15px 15px', 'shopbuilder-pro' ),
					'tab'   => 'style',
				],

				'size_chart_button_background_color'       => [
					'id'         => 'size_chart_button_background_color',
					'label'      => esc_html__( 'Button Background Color', 'shopbuilder-pro' ),
					'help'       => esc_html__( 'Change Notification background color', 'shopbuilder-pro' ),
					'type'       => 'color',
					'tab'        => 'style',
					'dependency' => [
						'rules' => [
							[
								'item'     => 'modules.product_size_chart.size_chart_button_action',
								'value'    => 'popups',
								'operator' => '==',
							],
						],
					],
				],

				'size_chart_button_text_color'             => [
					'id'         => 'size_chart_button_text_color',
					'label'      => esc_html__( 'Button Text Color', 'shopbuilder-pro' ),
					'help'       => esc_html__( 'Text Color', 'shopbuilder-pro' ),
					'type'       => 'color',
					'tab'        => 'style',
					'dependency' => [
						'rules' => [
							[
								'item'     => 'modules.product_size_chart.size_chart_button_action',
								'value'    => 'popups',
								'operator' => '==',
							],
						],
					],
				],

				'size_chart_button_background_hover_color' => [
					'id'         => 'size_chart_button_background_hover_color',
					'label'      => esc_html__( 'Button Hover Background Color', 'shopbuilder-pro' ),
					'help'       => esc_html__( 'Change Notification background color', 'shopbuilder-pro' ),
					'type'       => 'color',
					'tab'        => 'style',
					'dependency' => [
						'rules' => [
							[
								'item'     => 'modules.product_size_chart.size_chart_button_action',
								'value'    => 'popups',
								'operator' => '==',
							],
						],
					],
				],

				'size_chart_button_text_hover_color'       => [
					'id'         => 'size_chart_button_text_hover_color',
					'label'      => esc_html__( 'Button Hover Text Color', 'shopbuilder-pro' ),
					'help'       => esc_html__( 'Text Color', 'shopbuilder-pro' ),
					'type'       => 'color',
					'tab'        => 'style',
					'dependency' => [
						'rules' => [
							[
								'item'     => 'modules.product_size_chart.size_chart_button_action',
								'value'    => 'popups',
								'operator' => '==',
							],
						],
					],
				],

			]
		);
	}

	/**
	 * Get product min/max price.
	 *
	 * @return array|false
	 */
	public static function get_size_chart_list() {
		// Check if the cached result exists.
		$cache_key     = 'product_size_chart_list';
		$cached_result = wp_cache_get( $cache_key, 'shopbuilder' );

		if ( false !== $cached_result ) {
			// Return the cached result.
			return $cached_result;
		}
		// If the cached result doesn't exist, fetch it from the database.
		$options         = Fns::get_options( 'modules', 'product_size_chart' );
		$size_chart_list = json_decode( $options['chart_field'] ?? '', true );

		// Store the result in the cache.
		wp_cache_set( $cache_key, $size_chart_list, 'shopbuilder', 3600 ); // Cache for 1 hour.
		Cache::set_data_cache_key( $cache_key );
		return $size_chart_list;
	}

	/**
	 * @param $product_id
	 *
	 * @return array|mixed
	 */
	public static function get_size_chart_for_current_product( $product_id = 0 ) {
		$available_size_chart = [];
		if ( ! $product_id ) {
			global $product;
		} else {
			$product = wc_get_product( $product_id );
		}
		if ( ! $product instanceof WC_Product ) {
			return $available_size_chart;
		}
		$product_id = $product->get_id();

		$active_size_chart = self::get_size_chart_list();
		if ( ! is_array( $active_size_chart ) || ! count( $active_size_chart ) ) {
			return $available_size_chart;
		}

		foreach ( $active_size_chart as $index => $chart ) {

			$size_chart_exclude_for = $chart['exclude_product'] ?? '';
			$exclude_for            = Fns::multiselect_settings_field_value( $size_chart_exclude_for );
			$exclude_for            = array_map( 'absint', $exclude_for );

			if ( is_array( $exclude_for ) && count( $exclude_for ) && in_array( $product_id, $exclude_for ) ) {
				continue;
			}

			$size_chart_apply_for = $chart['apply_for'] ?? '';
			if ( 'product_cat' === $size_chart_apply_for ) {
				$term_ids = [];
				$terms    = get_the_terms( $product_id, 'product_cat' );
				if ( ! is_wp_error( $terms ) ) {
					foreach ( $terms as $term ) {
						$term_ids[] = $term->term_id;
					}
				}
				$applicable_cat = $chart['applicable_categories'] ?? '';
				$applicable_cat = Fns::multiselect_settings_field_value( $applicable_cat );
				$applicable_cat = array_map( 'absint', $applicable_cat );
				$has_cat        = array_intersect( $term_ids, $applicable_cat );
				if ( is_array( $applicable_cat ) && count( $applicable_cat ) && ! count( $has_cat ) ) {
					continue;
				}
			} else {
				$applicable_products = $chart['applicable_products'] ?? '';
				$applicable_products = Fns::multiselect_settings_field_value( $applicable_products );
				$applicable_products = array_map( 'absint', $applicable_products );
				if ( is_array( $applicable_products ) && count( $applicable_products ) && ! in_array( $product_id, $applicable_products ) ) {
					continue;
				}
			}
			$available_size_chart = $chart;
		}

		return $available_size_chart;
	}
}
