<?php
/**
 * Pre-order Functions Class.
 *
 * @package RadiusTheme\SBPRO
 */

namespace RadiusTheme\SBPRO\Modules\PreOrder;

use WC_Order;
use DateTime;
use Exception;
use DateTimeZone;
use RadiusTheme\SB\Helpers\Fns;
use RadiusTheme\SBPRO\Helpers\FnsPro;
use RadiusTheme\SB\Elementor\Helper\RenderHelpers;

defined( 'ABSPATH' ) || exit();

/**
 * Pre-order Functions Class.
 */
class PreOrderFns {
	/**
	 * Returns an array of settings fields.
	 *
	 * @return array
	 */
	public static function settings_field() {
		return [
			'pre_order_label_intro'              => [
				'id'    => 'pre_order_label_intro',
				'type'  => 'title',
				'label' => esc_html__( 'Pre-Order Texts', 'shopbuilder-pro' ),
				'tab'   => 'general',
			],
			'pre_order_button_label'             => [
				'id'          => 'pre_order_button_label',
				'label'       => __( 'Add to Cart Button Label', 'shopbuilder-pro' ),
				'type'        => 'text',
				'value'       => esc_html__( 'Pre-Order', 'shopbuilder-pro' ),
				'placeholder' => esc_html__( 'Ex. Pre-Order', 'shopbuilder-pro' ),
				'help'        => esc_html__( 'Enter the text to override the default \'Add to Cart\' button label. Leave empty for default label.', 'shopbuilder-pro' ),
				'tab'         => 'general',
			],
			'pre_order_closed_text'              => [
				'id'          => 'pre_order_closed_text',
				'label'       => __( 'Pre-Order Closed Text', 'shopbuilder-pro' ),
				'type'        => 'text',
				'value'       => esc_html__( 'Pre-Order Closed', 'shopbuilder-pro' ),
				'placeholder' => esc_html__( 'Ex. Pre-Order Closed', 'shopbuilder-pro' ),
				'help'        => esc_html__( 'Specify the message to be displayed when pre-orders are closed and the product is not auto-converted.', 'shopbuilder-pro' ),
				'tab'         => 'general',
			],
			'pre_order_checkout_button_label'    => [
				'id'          => 'pre_order_checkout_button_label',
				'label'       => __( 'Checkout \'Place Order\' <br />Button Label', 'shopbuilder-pro' ),
				'type'        => 'text',
				'help'        => __( 'Enter the text to override the default Checkout \'Place Order\' button label if cart contains Pre-Order products.  Leave empty for default label.', 'shopbuilder-pro' ),
				'placeholder' => esc_html__( 'Ex. Place Pre-Order Now', 'shopbuilder-pro' ),
				'tab'         => 'general',
			],
			'pre_order_avail_msg'                => [
				'id'          => 'pre_order_avail_msg',
				'label'       => __( 'General Availability Message <br />(with availability date)', 'shopbuilder-pro' ),
				'type'        => 'text_editor',
				'placeholder' => esc_html__( 'Ex. This product will be available by {availability_date}.', 'shopbuilder-pro' ),
				'value'       => esc_html__( 'On Pre-Order: Expected availability by {availability_date}.', 'shopbuilder-pro' ),
				'help'        => __( 'Customize the message displayed for pre-ordered items. Use the <code>{availability_date}</code> placeholder to dynamically show the specified availability date.', 'shopbuilder-pro' ),
				'tab'         => 'general',
			],
			'pre_order_avail_msg_no_date'        => [
				'id'          => 'pre_order_avail_msg_no_date',
				'label'       => __( 'General Availability Message <br />(if no availability date)', 'shopbuilder-pro' ),
				'type'        => 'text_editor',
				'placeholder' => esc_html__( 'Ex. This product will be available soon.', 'shopbuilder-pro' ),
				'value'       => esc_html__( 'On Pre-Order: This product will be available soon.', 'shopbuilder-pro' ),
				'help'        => __( 'General message to be shown when no date is set.', 'shopbuilder-pro' ),
				'tab'         => 'general',
			],
			'pre_order_settings_intro'           => [
				'id'    => 'pre_order_settings_intro',
				'type'  => 'title',
				'label' => esc_html__( 'Pre-Order Product Management', 'shopbuilder-pro' ),
				'tab'   => 'settings',
			],
			'pre_order_max_allowed_order'        => [
				'id'    => 'pre_order_max_allowed_order',
				'type'  => 'number',
				'value' => 3,
				'size'  => 'small',
				'min'   => 0,
				'max'   => 999,
				'label' => __( 'Set Pre-Order Limit', 'shopbuilder-pro' ),
				'help'  => __( 'Configure the maximum number of items a customer is allowed to pre-order.<br /> Leave empty for no limit.  This setting can be overridden on the product edit page.', 'shopbuilder-pro' ),
				'tab'   => 'settings',
			],
			'pre_order_status'                   => [
				'id'      => 'pre_order_status',
				'label'   => __( 'Define Order Status for <br />Pre-Order Items', 'shopbuilder-pro' ),
				'help'    => esc_html__( 'Select the appropriate status for orders that include pre-order products.', 'shopbuilder-pro' ),
				'type'    => 'select',
				'value'   => 'custom',
				'options' => [
					'default' => esc_html__( 'Default Order Status', 'shopbuilder-pro' ),
					'custom'  => esc_html__( "Custom 'Pre-ordered' Status", 'shopbuilder-pro' ),
				],
				'tab'     => 'settings',
			],
			'pre_order_price'                    => [
				'id'      => 'pre_order_price',
				'label'   => esc_html__( 'Display Pre-Order Product Pricing', 'shopbuilder-pro' ),
				'help'    => esc_html__( 'Choose whether to display the regular price of pre-order products to users.', 'shopbuilder-pro' ),
				'type'    => 'select',
				'value'   => 'hide',
				'options' => [
					'hide' => esc_html__( 'Hide the regular price (only show the pre-order price)', 'shopbuilder-pro' ),
					'show' => esc_html__( 'Show the regular price crossed out (with the pre-order price)', 'shopbuilder-pro' ),
				],
				'tab'     => 'settings',
			],
			'pre_order_auto_convert'             => [
				'id'    => 'pre_order_auto_convert',
				'type'  => 'switch',
				'label' => __( 'Auto-Convert to Standard Product', 'shopbuilder-pro' ),
				'help'  => esc_html__( 'Enable this option to automatically convert pre-order products to standard products once the available date is reached. This setting can be overridden on the product edit page.', 'shopbuilder-pro' ),
				'value' => 'on',
				'tab'   => 'settings',
			],
			'pre_order_user_access'              => [
				'id'    => 'pre_order_user_access',
				'type'  => 'title',
				'label' => esc_html__( 'User Access and Permissions', 'shopbuilder-pro' ),
				'tab'   => 'settings',
			],
			'user_role_management'               => [
				'id'      => 'user_role_management',
				'label'   => __( 'Specify Eligible Users for <br />Pre-Ordering', 'shopbuilder-pro' ),
				'help'    => esc_html__( 'Determine which user groups are allowed to place pre-orders.', 'shopbuilder-pro' ),
				'type'    => 'select',
				'value'   => 'all',
				'options' => [
					'all'      => esc_html__( 'All users', 'shopbuilder-pro' ),
					'specific' => esc_html__( 'Only users with a specific role', 'shopbuilder-pro' ),
				],
				'tab'     => 'settings',
			],
			'pre_order_roles'                    => [
				'id'         => 'pre_order_roles',
				'label'      => esc_html__( 'Select User Roles', 'shopbuilder-pro' ),
				'help'       => esc_html__( 'Please choose user roles for placing pre-orders.', 'shopbuilder-pro' ),
				'type'       => 'select',
				'value'      => [ 'administrator', 'customer' ],
				'options'    => FnsPro::get_all_user_roles(),
				'multiple'   => true,
				'tab'        => 'settings',
				'dependency' => [
					'rules' => [
						[
							'item'     => 'modules.pre_order.user_role_management',
							'value'    => 'specific',
							'operator' => '==',
						],
					],
				],
			],
			'hide_for_guest_users'               => [
				'id'    => 'hide_for_guest_users',
				'type'  => 'switch',
				'label' => __( 'Hide Pre-Order Pricing for <br />Guest Users', 'shopbuilder-pro' ),
				'help'  => esc_html__( 'Enable this option to hide the pre-order price from guest users.', 'shopbuilder-pro' ),
				'tab'   => 'settings',
			],
			'guest_user_price_text'              => [
				'id'         => 'guest_user_price_text',
				'type'       => 'text',
				'value'      => esc_html__( 'Login to see prices', 'shopbuilder-pro' ),
				'label'      => esc_html__( 'Custom Price Text for Guest Users', 'shopbuilder-pro' ),
				'help'       => esc_html__( 'Enter custom text that will show instead of product price. Leave blank to hide the price entirely.', 'shopbuilder-pro' ),
				'tab'        => 'settings',
				'dependency' => [
					'rules' => [
						[
							'item'     => 'modules.pre_order.hide_for_guest_users',
							'value'    => 'on',
							'operator' => '==',
						],
					],
				],
			],
			'pre_order_cart_settings'            => [
				'id'    => 'pre_order_cart_settings',
				'type'  => 'title',
				'label' => esc_html__( 'Shopping Cart Restrictions', 'shopbuilder-pro' ),
				'tab'   => 'settings',
			],
			'prevent_mixing'                     => [
				'id'    => 'prevent_mixing',
				'type'  => 'switch',
				'label' => __( 'Restrict Mixing of Product Types <br />in Cart', 'shopbuilder-pro' ),
				'help'  => esc_html__( 'Activate this setting to prevent the combination of pre-order products with regular products in the same shopping cart.', 'shopbuilder-pro' ),
				'tab'   => 'settings',
				'value' => 'on',
			],
			'prevent_add_multiple'               => [
				'id'    => 'prevent_add_multiple',
				'type'  => 'switch',
				'label' => __( 'Limit Multiple Pre-Order Items <br />in Cart', 'shopbuilder-pro' ),
				'help'  => esc_html__( 'Enable this option to restrict users from adding more than one pre-order product to the cart at a time.', 'shopbuilder-pro' ),
				'tab'   => 'settings',
			],
			'pre_order_countdown_settings'       => [
				'id'    => 'pre_order_countdown_settings',
				'type'  => 'title',
				'label' => esc_html__( 'Countdown Timer', 'shopbuilder-pro' ),
				'tab'   => 'settings',
			],
			'pre_order_show_countdown'           => [
				'id'    => 'pre_order_show_countdown',
				'type'  => 'switch',
				'label' => __( 'Enable Pre-Order Countdown Timer', 'shopbuilder-pro' ),
				'help'  => esc_html__( 'Toggle this setting to enable or disable the countdown timer for pre-order products.', 'shopbuilder-pro' ),
				'tab'   => 'settings',
				'value' => 'on',
			],
			'pre_order_notification_intro'       => [
				'id'   => 'pre_order_notification_intro',
				'type' => 'description',
				'text' => sprintf(
					/* translators: %s: link */
					__( 'You can manage the below emails from the <strong><a href="%s" target="_blank">WooCommerce->Settings->Emails</a></strong> tab.', 'shopbuilder-pro' ),
					esc_url( admin_url( 'admin.php?page=wc-settings&tab=email' ) )
				),
				'tab'  => 'notification',
			],
			'pre_order_received_email'           => [
				'id'    => 'pre_order_received_email',
				'type'  => 'switch',
				'label' => __( 'New Pre-Order Received Email', 'shopbuilder-pro' ),
				'help'  => esc_html__( 'Enable this option to send an email to admin when an order contains a pre-order product is received.', 'shopbuilder-pro' ),
				'tab'   => 'notification',
				'value' => 'on',
			],
			'pre_order_confirmation_email'       => [
				'id'    => 'pre_order_confirmation_email',
				'type'  => 'switch',
				'label' => __( 'Pre-Order Confirmation Email', 'shopbuilder-pro' ),
				'help'  => esc_html__( 'Enable this option to send an email to customers when an order contains a pre-order product is placed.', 'shopbuilder-pro' ),
				'tab'   => 'notification',
				'value' => 'on',
			],
			'pre_order_release_email'            => [
				'id'    => 'pre_order_release_email',
				'type'  => 'switch',
				'label' => __( 'Pre-Order Product Release Email', 'shopbuilder-pro' ),
				'help'  => esc_html__( 'Enable this option to send an email to customers when a pre-order product they purchased is released.', 'shopbuilder-pro' ),
				'tab'   => 'notification',
				'value' => 'on',
			],
			'pre_order_cancellation_email'       => [
				'id'    => 'pre_order_cancellation_email',
				'type'  => 'switch',
				'label' => __( 'Pre-Order Cancellation Email', 'shopbuilder-pro' ),
				'help'  => esc_html__( 'Enable this option to send an email to customers when a pre-order product they purchased is cancelled by the admin.', 'shopbuilder-pro' ),
				'tab'   => 'notification',
				'value' => 'on',
			],
			'pre_order_text_style_intro'         => [
				'id'    => 'pre_order_text_style_intro',
				'type'  => 'title',
				'label' => esc_html__( 'Pre-Order Text', 'shopbuilder-pro' ),
				'tab'   => 'styles',
			],
			'pre_order_text_color'               => [
				'id'    => 'pre_order_text_color',
				'label' => esc_html__( 'Pre-Order Text Color', 'shopbuilder-pro' ),
				'help'  => esc_html__( 'Choose color for Pre-Order Text', 'shopbuilder-pro' ),
				'type'  => 'color',
				'tab'   => 'styles',
			],
			'pre_order_bg_color'                 => [
				'id'    => 'pre_order_bg_color',
				'label' => esc_html__( 'Pre-Order Background Color', 'shopbuilder-pro' ),
				'help'  => esc_html__( 'Choose background color for Pre-Order Text', 'shopbuilder-pro' ),
				'type'  => 'color',
				'tab'   => 'styles',
			],
			'pre_order_font_size'                => [
				'id'    => 'pre_order_font_size',
				'type'  => 'text',
				'label' => esc_html__( 'Font Size', 'shopbuilder-pro' ),
				'help'  => esc_html__( 'Enter Font Size. Ex. 16px', 'shopbuilder-pro' ),
				'tab'   => 'styles',
			],
			'pre_order_font_weight'              => [
				'id'      => 'pre_order_font_weight',
				'type'    => 'select',
				'options' => [
					100 => esc_html__( '100 (Thin)', 'shopbuilder-pro' ),
					200 => esc_html__( '200 (Extra Light)', 'shopbuilder-pro' ),
					300 => esc_html__( '300 (Light)', 'shopbuilder-pro' ),
					400 => esc_html__( '400 (Normal)', 'shopbuilder-pro' ),
					500 => esc_html__( '500 (Medium)', 'shopbuilder-pro' ),
					600 => esc_html__( '600 (Semi Bold)', 'shopbuilder-pro' ),
					700 => esc_html__( '700 (Bold)', 'shopbuilder-pro' ),
					800 => esc_html__( '800 (Extra Bold)', 'shopbuilder-pro' ),
					900 => esc_html__( '900 (Black)', 'shopbuilder-pro' ),
				],
				'value'   => 400,
				'label'   => esc_html__( 'Font Weight', 'shopbuilder-pro' ),
				'help'    => esc_html__( 'Choose Font Weight.', 'shopbuilder-pro' ),
				'tab'     => 'styles',
			],
			'pre_order_radius'                   => [
				'id'    => 'pre_order_radius',
				'type'  => 'text',
				'label' => esc_html__( 'Border Radius', 'shopbuilder-pro' ),
				'help'  => esc_html__( 'Enter Border Radius. Ex. 10px | 5px 10px | 0 5px 5px 0', 'shopbuilder-pro' ),
				'tab'   => 'styles',
			],
			'pre_order_padding'                  => [
				'id'    => 'pre_order_padding',
				'type'  => 'text',
				'label' => esc_html__( 'Padding', 'shopbuilder-pro' ),
				'help'  => esc_html__( 'Enter Padding. Ex. 10px | 5px 10px | 0 5px 5px 0', 'shopbuilder-pro' ),
				'tab'   => 'styles',
			],
			'pre_order_margin'                   => [
				'id'    => 'pre_order_margin',
				'type'  => 'text',
				'label' => esc_html__( 'Margin', 'shopbuilder-pro' ),
				'help'  => esc_html__( 'Enter Margin. Ex. 10px | 5px 10px | 0 5px 5px 0', 'shopbuilder-pro' ),
				'tab'   => 'styles',
			],
			'pre_order_badge_style_intro'        => [
				'id'    => 'pre_order_badge_style_intro',
				'type'  => 'title',
				'label' => esc_html__( 'Pre-Order Badge', 'shopbuilder-pro' ),
				'tab'   => 'styles',
			],
			'pre_order_badge_text_color'         => [
				'id'    => 'pre_order_badge_text_color',
				'label' => esc_html__( 'Pre-Order Badge Text Color', 'shopbuilder-pro' ),
				'help'  => esc_html__( 'Choose color for Pre-Order badge text', 'shopbuilder-pro' ),
				'type'  => 'color',
				'tab'   => 'styles',
			],
			'pre_order_badge_bg_color'           => [
				'id'    => 'pre_order_badge_bg_color',
				'label' => esc_html__( 'Pre-Order Badge Background Color', 'shopbuilder-pro' ),
				'help'  => esc_html__( 'Choose background color for Pre-Order badge text', 'shopbuilder-pro' ),
				'type'  => 'color',
				'tab'   => 'styles',
			],
			'pre_order_counter_style_intro'      => [
				'id'         => 'pre_order_counter_style_intro',
				'type'       => 'title',
				'label'      => esc_html__( 'Pre-Order Counter', 'shopbuilder-pro' ),
				'tab'        => 'styles',
				'dependency' => [
					'rules' => [
						[
							'item'     => 'modules.pre_order.pre_order_show_countdown',
							'value'    => 'on',
							'operator' => '==',
						],
					],
				],
			],
			'pre_order_counter_background_color' => [
				'id'         => 'pre_order_counter_background_color',
				'label'      => esc_html__( 'Counter Background Color', 'shopbuilder-pro' ),
				'help'       => esc_html__( 'Change Notification background color', 'shopbuilder-pro' ),
				'type'       => 'color',
				'tab'        => 'styles',
				'dependency' => [
					'rules' => [
						[
							'item'     => 'modules.pre_order.pre_order_show_countdown',
							'value'    => 'on',
							'operator' => '==',
						],
					],
				],
			],

			'pre_order_counter_text_color'       => [
				'id'         => 'pre_order_counter_text_color',
				'label'      => esc_html__( 'Counter Text Color', 'shopbuilder-pro' ),
				'help'       => esc_html__( 'Text Color', 'shopbuilder-pro' ),
				'type'       => 'color',
				'tab'        => 'styles',
				'dependency' => [
					'rules' => [
						[
							'item'     => 'modules.pre_order.pre_order_show_countdown',
							'value'    => 'on',
							'operator' => '==',
						],
					],
				],
			],
			'pre_order_counter_button_margin'    => [
				'id'         => 'pre_order_counter_button_margin',
				'type'       => 'text',
				'label'      => esc_html__( 'Wrapper Margin', 'shopbuilder-pro' ),
				'help'       => esc_html__( 'Enter countdown wrapper margin. Example: 0 15px 15px 15px', 'shopbuilder-pro' ),
				'tab'        => 'styles',
				'dependency' => [
					'rules' => [
						[
							'item'     => 'modules.pre_order.pre_order_show_countdown',
							'value'    => 'on',
							'operator' => '==',
						],
					],
				],
			],
		];
	}

	/**
	 * Get pre-order settings data.
	 *
	 * @return array
	 */
	public static function get_pre_order_settings_data() {
		$settings = Fns::get_options( 'modules', 'pre_order' );

		return [
			'button_label'     => RenderHelpers::get_data( $settings, 'pre_order_button_label', '' ),
			'closed_label'     => RenderHelpers::get_data( $settings, 'pre_order_closed_text', esc_html__( 'Pre-Order Closed', 'shopbuilder-pro' ) ),
			'checkout_label'   => RenderHelpers::get_data( $settings, 'pre_order_checkout_button_label', '' ),
			'message'          => RenderHelpers::get_data( $settings, 'pre_order_avail_msg', '' ),
			'message_no_date'  => RenderHelpers::get_data( $settings, 'pre_order_avail_msg_no_date', '' ),
			'limit'            => RenderHelpers::get_data( $settings, 'pre_order_max_allowed_order', 0 ),
			'order_status'     => RenderHelpers::get_data( $settings, 'pre_order_status', 'custom' ),
			'show_price'       => RenderHelpers::get_data( $settings, 'pre_order_price', 'hide' ),
			'auto_convert'     => RenderHelpers::get_data( $settings, 'pre_order_auto_convert', 'on' ),
			'user_access'      => RenderHelpers::get_data( $settings, 'user_role_management', 'all' ),
			'user_role'        => RenderHelpers::get_data( $settings, 'pre_order_roles', [ 'administrator' ] ),
			'guest_pricing'    => RenderHelpers::get_data( $settings, 'hide_for_guest_users', '' ),
			'guest_text'       => RenderHelpers::get_data( $settings, 'guest_user_price_text', '' ),
			'prevent_mixing'   => RenderHelpers::get_data( $settings, 'prevent_mixing', '' ),
			'prevent_multiple' => RenderHelpers::get_data( $settings, 'prevent_add_multiple', '' ),
			'show_countdown'   => RenderHelpers::get_data( $settings, 'pre_order_show_countdown', '' ),
			'new_email'        => RenderHelpers::get_data( $settings, 'pre_order_received_email', '' ),
			'confirm_email'    => RenderHelpers::get_data( $settings, 'pre_order_confirmation_email', '' ),
			'release_email'    => RenderHelpers::get_data( $settings, 'pre_order_release_email', '' ),
			'cancel_email'     => RenderHelpers::get_data( $settings, 'pre_order_cancellation_email', '' ),
		];
	}

	/**
	 * Get product meta.
	 *
	 * @param int $product_id Product ID.
	 *
	 * @return array
	 */
	public static function get_pre_order_meta_data( $product_id ) {
		$meta_mapping = self::product_meta_data_mapping();
		$product_meta = [];

		foreach ( $meta_mapping as $meta_key => $custom_field_id ) {
			$product_meta[ $meta_key ] = get_post_meta( $product_id, $custom_field_id, true );
		}

		return $product_meta;
	}

	/**
	 * Maps product meta-keys to their corresponding custom field IDs.
	 *
	 * @return array
	 */
	public static function product_meta_data_mapping() {
		return [
			'preorder_stock'            => '_rtsb_preorder_stock',
			'preorder_limit'            => '_rtsb_max_preorder_allowed',
			'preorder_availability'     => '_rtsb_preorder_avail_date',
			'preorder_price_option'     => '_rtsb_preorder_pricing_options',
			'preorder_price'            => '_rtsb_preorder_price',
			'preorder_percent_discount' => '_rtsb_preorder_percentage_discount',
			'preorder_fixed_discount'   => '_rtsb_preorder_fixed_discount',
			'preorder_percent_increase' => '_rtsb_preorder_percentage_increase',
			'preorder_fixed_increase'   => '_rtsb_preorder_fixed_increase',
			'preorder_auto_convert'     => '_rtsb_preorder_auto_convert',
		];
	}

	/**
	 * Get pre-order availability date for a specific product.
	 *
	 * @param int $product_id Product ID.
	 *
	 * @return string
	 */
	public static function get_pre_order_date( $product_id ) {
		return self::get_pre_order_meta_data( $product_id )['preorder_availability'];
	}

	/**
	 * Get pre-order limit for a specific product.
	 *
	 * @param int $product_id Product ID.
	 *
	 * @return string
	 */
	public static function get_pre_order_limit( $product_id ) {
		$limit = self::get_pre_order_meta_data( $product_id )['preorder_limit'];

		if ( empty( $limit ) ) {
			$limit = self::get_pre_order_settings_data()['limit'];
		}

		return $limit;
	}

	/**
	 * Get a pre-order message for a specific product.
	 *
	 * @param int $product_id Product ID.
	 *
	 * @return string
	 */
	public static function get_pre_order_message( $product_id ) {
		$message      = self::get_pre_order_settings_data()['message'];
		$availability = self::get_pre_order_date( $product_id );
		$date_text    = '<span>' . implode( ' - ', self::get_pre_order_formatted_date( $availability ) ) . '</span>';

		if ( empty( $availability ) ) {
			$message = self::get_pre_order_settings_data()['message_no_date'];
		}

		return str_replace( '{availability_date}', $date_text, $message );
	}

	/**
	 * Get formatted date-time based on WordPress settings.
	 *
	 * @param string $date Date to format.
	 *
	 * @return array
	 */
	public static function get_pre_order_formatted_date( $date ) {
		if ( empty( $date ) ) {
			return [];
		}

		$timestamp = strtotime( $date );

		return [
			'date' => date_i18n( get_option( 'date_format' ), $timestamp ),
			'time' => date_i18n( get_option( 'time_format' ), $timestamp ),
		];
	}

	/**
	 * Check if pre-order is allowed for a specific product.
	 *
	 * @param int $product_id Product ID.
	 * @param int $variation_id Product variation ID.
	 *
	 * @return bool
	 * @throws Exception If there is an error in determining the pre-order status.
	 */
	public static function is_pre_order_allowed( $product_id, $variation_id = '' ) {
		if ( ! empty( $variation_id ) ) {
			$product_id = $variation_id;
		}

		$closed = self::is_pre_order_closed( $product_id );

		if ( $closed ) {
			return false;
		}

		$limit = self::get_pre_order_limit( $product_id );

		return $limit < 0 || ( $limit - 1 ) >= 0;
	}

	/**
	 * Display a notice related to the product's pre-order status.
	 *
	 * @param object $product The WooCommerce product object.
	 * @param int    $amount  The maximum amount available for pre-order.
	 * @param string $type  Notice Type.
	 *
	 * @return void
	 */
	public static function pre_order_product_notice( $product, $amount, $type = 'limit' ) {
		$product_title = $product->get_title();
		$message       = '';

		if ( 'limit' === $type ) {
			if ( $amount <= 0 ) {
				/* translators: %s: product title */
				$notice  = esc_html__( 'Sorry, "%s" has reached its pre-order limit.', 'shopbuilder-pro' );
				$message = sprintf( $notice, $product_title );
			} else {
				/* translators: %1$s: product title, %2$s: product amount */
				$notice  = esc_html__( '"%1$s" is available for pre-order. You can add a maximum of %2$d unit to your cart.', 'shopbuilder-pro' );
				$message = sprintf( $notice, $product_title, intval( $amount ) );
			}
		} elseif ( 'cart_mix' === $type ) {
			$message = esc_html__( 'Sorry, you cannot add Pre-Order and other products in the same order. Please check-out separately.', 'shopbuilder-pro' );
		} elseif ( 'cart_multi' === $type ) {
			$message = esc_html__( 'Sorry, you can only add one Pre-Order product at a time. Please complete your current Pre-Order first.', 'shopbuilder-pro' );
		}

		wc_add_notice( '<strong>' . $message . '</strong>', 'error' );
	}

	/**
	 * Check if back order stock for a specific product.
	 *
	 * @param int $id Product ID.
	 * @param int $stock Product stock.
	 * @param int $variation_id Product variation ID.
	 *
	 * @return int|null
	 */
	public static function check_pre_order_stock( $id, $stock, $variation_id = '' ) {
		if ( ! empty( $variation_id ) ) {
			$id = $variation_id;
		}

		$limit = self::get_pre_order_limit( $id );

		if ( ! empty( $limit ) || $limit > 0 ) {
			return intval( $stock ) + intval( $limit );
		}

		return null;
	}

	/**
	 * Check if a product is on pre-order.
	 *
	 * @param mixed $product The product object.
	 * @return bool
	 */
	public static function is_on_pre_order( $product ) {
		if ( ! $product ) {
			return false;
		}

		return 'preorder' === $product->get_stock_status();
	}

	/**
	 * Check if a variable product is on pre-order.
	 *
	 * @param mixed $product The product object.
	 * @return bool
	 */
	public static function variation_is_on_pre_order( $product ) {
		if ( ! $product ) {
			return false;
		}

		if ( 'variable' !== $product->get_type() ) {
			return false;
		}

		$variation_ids  = $product->get_children();
		$is_on_preorder = false;

		foreach ( $variation_ids as $variation_id ) {
			$stock_status = get_post_meta( $variation_id, '_stock_status', true );

			if ( 'preorder' === $stock_status ) {
				$is_on_preorder = true;

				break;
			}
		}

		return $is_on_preorder;
	}

	/**
	 * Check if a variation product is on pre-order.
	 *
	 * @param int $variation_id The variation product ID.
	 *
	 * @return bool
	 */
	public static function variation_id_is_on_pre_order( $variation_id ) {
		$is_on_preorder = false;
		$stock_status   = get_post_meta( $variation_id, '_stock_status', true );

		if ( 'preorder' === $stock_status ) {
			$is_on_preorder = true;
		}

		return $is_on_preorder;
	}

	/**
	 * Checks if the pre-order is closed and returns the time remaining until the pre-order end date.
	 *
	 * @param int $product_id The ID of the product to check.
	 *
	 * @throws Exception If there is an error creating the DateTime objects.
	 * @return array
	 */
	public static function check_pre_order_status( $product_id ) {
		$status = [
			'closed'         => false,
			'time_remaining' => '',
			'formatted_time' => '',
		];

		$preorder_end_datetime = self::get_pre_order_meta_data( $product_id )['preorder_availability'] ?? '';

		if ( ! $preorder_end_datetime ) {
			$status['time_remaining'] = null;

			return $status;
		}

		$timezone_string = get_option( 'timezone_string' );

		if ( empty( $timezone_string ) ) {
			$gmt_offset = get_option( 'gmt_offset' );
			$gmt_offset = intval( $gmt_offset );
			$timezone   = new DateTimeZone( sprintf( 'Etc/GMT%+d', -$gmt_offset ) );
		} else {
			$timezone = new DateTimeZone( $timezone_string );
		}

		try {
			$current_datetime      = new DateTime( 'now', $timezone );
			$preorder_end_datetime = new DateTime( $preorder_end_datetime, $timezone );
		} catch ( Exception $e ) {
			return $status;
		}

		$interval = $current_datetime->diff( $preorder_end_datetime );

		if ( $interval->invert ) {
			$status['closed'] = true;

			return $status;
		}

		$time_remaining = [];

		if ( $interval->d > 0 ) {
			$time_remaining['days'] = $interval->d;
		}
		if ( $interval->h > 0 ) {
			$time_remaining['hours'] = $interval->h;
		}
		if ( $interval->i > 0 ) {
			$time_remaining['minutes'] = $interval->i;
		}
		if ( $interval->s > 0 ) {
			$time_remaining['seconds'] = $interval->s;
		}

		$status['time_remaining'] = $time_remaining;
		$status['formatted_time'] = $preorder_end_datetime->format( 'Y-m-d H:i' );

		return $status;
	}

	/**
	 * Check if a pre-order is closed for a given product.
	 *
	 * @param int $product_id The ID of the product.
	 *
	 * @throws Exception If there is an error in checking the pre-order status.
	 * @return bool
	 */
	public static function is_pre_order_closed( $product_id ) {
		$pre_order_status = self::check_pre_order_status( $product_id );

		if ( $pre_order_status['closed'] ) {
			self::auto_convert_to_standard( $product_id );
		}

		return $pre_order_status['closed'];
	}

	/**
	 * Automatically convert a pre-order product to a standard product.
	 *
	 * @param int $product_id The ID of the product.
	 *
	 * @return void
	 */
	private static function auto_convert_to_standard( $product_id ) {
		$meta_data = self::get_pre_order_meta_data( $product_id );

		if ( 'yes' === $meta_data['preorder_auto_convert'] ) {
			$product = wc_get_product( $product_id );

			if ( $product ) {
				update_post_meta( $product_id, '_rtsb_preorder_auto_convert', 'no' );
				delete_post_meta( $product_id, '_rtsb_preorder_avail_date' );

				$product->set_stock_status( 'instock' );
				$product->save();
			}
		}
	}

	/**
	 * Check if an order contains pre-order products.
	 *
	 * @param WC_Order $order The order object.
	 *
	 * @return bool
	 */
	public static function order_contains_pre_order_products( $order ) {
		if ( ! $order instanceof WC_Order ) {
			return false;
		}

		foreach ( $order->get_items() as $item ) {
			$product_id = $item->get_variation_id() ?: $item->get_product_id();
			$product    = wc_get_product( $product_id );

			if ( $product && self::is_on_pre_order( $product ) ) {
				return true;
			}
		}

		return false;
	}
}
