<?php
/**
 * New Pre-order Received Email Class.
 *
 * @package RadiusTheme\SBPRO
 */

namespace RadiusTheme\SBPRO\Modules\PreOrder\Notification;

use WC_Email;
use WC_Order;

defined( 'ABSPATH' ) || exit();

/**
 * New Pre-order Received Email Class.
 */
class PreOrderReceivedEmail extends WC_Email {
	/**
	 * Email content.
	 *
	 * @var string
	 */
	public $email_content;

	/**
	 * Class Constructor.
	 */
	public function __construct() {
		$this->id            = 'rtsb_new_pre_order_received';
		$this->title         = esc_html__( 'ShopBuilder: New Pre-Order Received', 'shopbuilder-pro' );
		$this->description   = esc_html__( 'This email is sent to admin when a new pre-order is received.', 'shopbuilder-pro' );
		$this->heading       = esc_html__( 'New Pre-Order received #{order_number}', 'shopbuilder-pro' );
		$this->subject       = esc_html__( '[{site_title}]: New Pre-Order #{order_number}', 'shopbuilder-pro' );
		$this->email_content = $this->get_option(
			'email_content',
			sprintf(
				/* translators: line break */
				__( 'Hi admin! %1$s%2$sA new pre-order has been received from {customer_name}. %1$s%2$s{order_details}%1$s%2$sBest regards,%1$s{site_title}', 'shopbuilder-pro' ),
				"\n",
				"\n"
			)
		);
		$this->template_html = 'emails/pre-order-received.php';

		add_action( 'rtsb/module/pre_order/order_received_email', [ $this, 'trigger' ], 10, 1 );

		parent::__construct();

		$this->email_type = 'html';
		$this->recipient  = $this->get_option( 'recipient', get_option( 'admin_email' ) );
	}

	/**
	 * Trigger the email notification for a new order.
	 *
	 * @param WC_Order $order The order object.
	 *
	 * @return void
	 */
	public function trigger( $order ) {
		if ( ! $order ) {
			return;
		}

		$this->object  = $order;
		$customer_name = $this->object->get_formatted_billing_full_name();
		$order_number  = $this->object->get_order_number();

		$this->placeholders = array_merge(
			$this->placeholders,
			[
				'{customer_name}' => $customer_name,
				'{order_number}'  => $order_number,
				'{order_date}'    => wc_format_datetime( $this->object->get_date_created() ),
			]
		);

		if ( ! $this->is_enabled() || ! $this->get_recipient() ) {
			return;
		}

		$this->send( $this->get_recipient(), $this->get_subject(), $this->get_content(), $this->get_headers(), $this->get_attachments() );
	}

	/**
	 * Get the HTML content for the email.
	 *
	 * @return string The HTML content.
	 */
	public function get_content_html() {
		return wc_get_template_html(
			$this->template_html,
			[
				'order'         => $this->object,
				'email_heading' => $this->get_heading(),
				'sent_to_admin' => true,
				'plain_text'    => false,
				'email'         => $this,
				'store_email'   => $this->get_from_address(),
			],
			'',
			rtsbpro()->get_plugin_template_path()
		);
	}

	/**
	 * Initialize the form fields for the plugin settings.
	 *
	 * @return void
	 */
	public function init_form_fields() {
		$this->placeholders = array_merge(
			$this->placeholders,
			[
				'{customer_name}' => __( 'Customer Name', 'shopbuilder-pro' ),
				'{order_number}'  => __( 'Order Number', 'shopbuilder-pro' ),
				'{order_date}'    => __( 'Order Date', 'shopbuilder-pro' ),
				'{order_details}' => __( 'Order Details', 'shopbuilder-pro' ),
			]
		);

		/* translators: %s: list of placeholders */
		$placeholder_text  = sprintf( __( 'Available placeholders: %s', 'shopbuilder-pro' ), '<code>' . implode( '</code>, <code>', array_keys( $this->placeholders ) ) . '</code>' );
		$this->form_fields = [
			'enabled'       => [
				'title'   => __( 'Enable/Disable', 'shopbuilder-pro' ),
				'type'    => 'checkbox',
				'label'   => __( 'Enable this email notification', 'shopbuilder-pro' ),
				'default' => 'yes',
			],
			'recipient'     => [
				'title'       => __( 'Recipient(s)', 'shopbuilder-pro' ),
				'type'        => 'text',
				/* translators: %s: WP admin email */
				'description' => sprintf( __( 'Enter recipients (comma separated) for this email. Defaults to %s.', 'shopbuilder-pro' ), '<code>' . esc_attr( get_option( 'admin_email' ) ) . '</code>' ),
				'placeholder' => '',
				'default'     => esc_attr( get_option( 'admin_email' ) ),
				'desc_tip'    => true,
			],
			'subject'       => [
				'title'       => __( 'Subject', 'shopbuilder-pro' ),
				'type'        => 'text',
				'desc_tip'    => true,
				'description' => $placeholder_text,
				'placeholder' => $this->get_default_subject(),
				'default'     => $this->get_default_subject(),
			],
			'heading'       => [
				'title'       => __( 'Email heading', 'shopbuilder-pro' ),
				'type'        => 'text',
				'desc_tip'    => true,
				'description' => $placeholder_text,
				'placeholder' => $this->get_default_heading(),
				'default'     => $this->get_default_heading(),
			],
			'email_content' => [
				'title'       => __( 'Email Content', 'shopbuilder-pro' ),
				'description' => __( 'Text to appear in the main email body.', 'shopbuilder-pro' ) . ' ' . $placeholder_text . '.',
				'css'         => 'width:800px; height: 185px;',
				'placeholder' => '',
				'type'        => 'textarea',
				'default'     => $this->email_content,
			],
		];
	}
}
