<?php
/**
 * Pre-order Confirmation Email Class.
 *
 * @package RadiusTheme\SBPRO
 */

namespace RadiusTheme\SBPRO\Modules\PreOrder\Notification;

use WC_Email;
use WC_Order;

defined( 'ABSPATH' ) || exit();

/**
 * Pre-order Confirmation Email Class.
 */
class OrderConfirmationEmail extends WC_Email {
	/**
	 * Email content.
	 *
	 * @var string
	 */
	public $email_content;

	/**
	 * Class Constructor.
	 */
	public function __construct() {
		$this->id             = 'rtsb_pre_order_confirmed';
		$this->title          = esc_html__( 'ShopBuilder: Pre-Order Confirmed', 'shopbuilder-pro' );
		$this->description    = esc_html__( 'This email is sent to customer when a pre-order is placed and confirmed.', 'shopbuilder-pro' );
		$this->heading        = esc_html__( 'Pre-Order confirmed #{order_number}', 'shopbuilder-pro' );
		$this->subject        = esc_html__( '[{site_title}]: Your Pre-Order is confirmed #{order_number}', 'shopbuilder-pro' );
		$this->customer_email = true;
		$this->email_content  = $this->get_option(
			'email_content',
			sprintf(
				/* translators: line break */
				__( 'Hi {customer_name}! %1$s%2$sYour pre-order has been received and is now being processed. You will receive another email when the product becomes available. %1$s%2$s{order_details} %1$sThank you for your purchase.%1$s%2$sBest regards,%1$s{site_title}', 'shopbuilder-pro' ),
				"\n",
				"\n"
			)
		);
		$this->template_html = 'emails/pre-order-confirmed.php';

		add_action( 'rtsb/module/pre_order/order_confirmed_email', [ $this, 'trigger' ], 10, 1 );

		parent::__construct();

		$this->email_type = 'html';
	}

	/**
	 * Trigger the email notification for a new order.
	 *
	 * @param WC_Order $order The order object.
	 *
	 * @return void
	 */
	public function trigger( $order ) {
		if ( ! $order ) {
			return;
		}

		$this->object    = $order;
		$customer_name   = $this->object->get_formatted_billing_full_name();
		$this->recipient = $this->object->get_billing_email();
		$order_number    = $this->object->get_order_number();

		$this->placeholders = array_merge(
			$this->placeholders,
			[
				'{customer_name}' => $customer_name,
				'{order_number}'  => $order_number,
				'{order_date}'    => wc_format_datetime( $this->object->get_date_created() ),
			]
		);

		if ( ! $this->is_enabled() || ! $this->get_recipient() ) {
			return;
		}

		$this->send( $this->get_recipient(), $this->get_subject(), $this->get_content(), $this->get_headers(), $this->get_attachments() );
	}

	/**
	 * Get the HTML content for the email.
	 *
	 * @return string The HTML content.
	 */
	public function get_content_html() {
		return wc_get_template_html(
			$this->template_html,
			[
				'order'         => $this->object,
				'email_heading' => $this->get_heading(),
				'sent_to_admin' => true,
				'plain_text'    => false,
				'email'         => $this,
				'store_email'   => $this->get_from_address(),
			],
			'',
			rtsbpro()->get_plugin_template_path()
		);
	}

	/**
	 * Initialize the form fields for the plugin settings.
	 *
	 * @return void
	 */
	public function init_form_fields() {
		$this->placeholders = array_merge(
			$this->placeholders,
			[
				'{customer_name}' => __( 'Customer Name', 'shopbuilder-pro' ),
				'{order_number}'  => __( 'Order Number', 'shopbuilder-pro' ),
				'{order_date}'    => __( 'Order Date', 'shopbuilder-pro' ),
				'{order_details}' => __( 'Order Details', 'shopbuilder-pro' ),
			]
		);

		/* translators: %s: list of placeholders */
		$placeholder_text  = sprintf( __( 'Available placeholders: %s', 'shopbuilder-pro' ), '<code>' . implode( '</code>, <code>', array_keys( $this->placeholders ) ) . '</code>' );
		$this->form_fields = [
			'enabled'       => [
				'title'   => __( 'Enable/Disable', 'shopbuilder-pro' ),
				'type'    => 'checkbox',
				'label'   => __( 'Enable this email notification', 'shopbuilder-pro' ),
				'default' => 'yes',
			],
			'subject'       => [
				'title'       => __( 'Subject', 'shopbuilder-pro' ),
				'type'        => 'text',
				'desc_tip'    => true,
				'description' => $placeholder_text,
				'placeholder' => $this->get_default_subject(),
				'default'     => $this->get_default_subject(),
			],
			'heading'       => [
				'title'       => __( 'Email heading', 'shopbuilder-pro' ),
				'type'        => 'text',
				'desc_tip'    => true,
				'description' => $placeholder_text,
				'placeholder' => $this->get_default_heading(),
				'default'     => $this->get_default_heading(),
			],
			'email_content' => [
				'title'       => __( 'Email Content', 'shopbuilder-pro' ),
				'description' => __( 'Text to appear in the main email body.', 'shopbuilder-pro' ) . ' ' . $placeholder_text . '.',
				'css'         => 'width:800px; height: 185px;',
				'placeholder' => '',
				'type'        => 'textarea',
				'default'     => $this->email_content,
			],
		];
	}
}
