<?php
/**
 * Pre-order Product Class.
 *
 * @package RadiusTheme\SBPRO
 */

namespace RadiusTheme\SBPRO\Modules\PreOrder\Frontend;

use Exception;
use WC_Product;
use RadiusTheme\SB\Helpers\Fns;
use RadiusTheme\SBPRO\Helpers\FnsPro;
use RadiusTheme\SBPRO\Traits\SingletonTrait;
use RadiusTheme\SBPRO\Modules\AddOns\AddOnsFns;
use RadiusTheme\SB\Elementor\Helper\RenderHelpers;
use RadiusTheme\SBPRO\Modules\PreOrder\PreOrderFns;
use RadiusTheme\SBPRO\Modules\AddOns\Frontend\Product as AddOnsProduct;

defined( 'ABSPATH' ) || exit();

/**
 * Pre-order Single Product Page Class.
 */
class Product {
	/**
	 * Singleton Trait.
	 */
	use SingletonTrait;

	/**
	 * Module Class Constructor.
	 */
	private function __construct() {
		/**
		 * Actions.
		 */
		add_filter( 'woocommerce_product_single_add_to_cart_text', [ $this, 'pre_order_button_label' ], 99, 2 );
		add_filter( 'woocommerce_product_add_to_cart_text', [ $this, 'pre_order_shop_button_label' ], 99, 2 );
		add_filter( 'rtsb/elementor/render/add_to_cart_text', [ $this, 'pre_order_shop_button_label' ], 99, 2 );
		add_action( 'woocommerce_is_purchasable', [ $this, 'disable_add_to_cart' ], 0, 2 );
		add_action( 'woocommerce_get_price_html', [ $this, 'custom_pricing' ], 10, 2 );

		/**
		 * Filters.
		 */
		add_filter( 'woocommerce_cart_item_price', [ $this, 'disable_price' ], 99, 2 );
		add_filter( 'woocommerce_product_add_to_cart_url', [ $this, 'modify_add_to_cart_url' ], 10, 2 );
		add_filter( 'rtsb/elementor/render/cart_attributes', [ $this, 'change_add_to_cart_args' ], 10, 2 );
		add_filter( 'woocommerce_loop_add_to_cart_args', [ $this, 'change_add_to_cart_args' ], 10, 2 );
		add_action( 'woocommerce_product_get_price', [ $this, 'pre_order_pricing' ], 99, 2 );
		add_filter( 'woocommerce_product_variation_get_price', [ $this, 'pre_order_pricing' ], 99, 2 );
		add_filter( 'woocommerce_product_variation_get_sale_price', [ $this, 'pre_order_pricing' ], 99, 2 );
		add_filter( 'woocommerce_available_variation', [ $this, 'pre_order_variation_data' ], 99 );
		add_filter( 'woocommerce_variation_prices', [ $this, 'woocommerce_variation_prices_array' ], 5, 2 );
		add_filter( 'woocommerce_get_availability_text', [ $this, 'pre_order_text' ], 10, 2 );
		add_filter( 'woocommerce_get_availability_class', [ $this, 'pre_order_class' ], 10, 2 );
		add_filter( 'rtsb/elementor/product_stock_icons', [ $this, 'pre_order_icon' ], 10, 3 );
	}

	/**
	 * Modify the "Add to Cart" URL for pre-order products.
	 *
	 * @param string     $permalink The original permalink URL.
	 * @param WC_Product $product The product object.
	 *
	 * @return string
	 */
	public function modify_add_to_cart_url( $permalink, $product ) {
		if ( PreOrderFns::is_on_pre_order( $product ) ) {
			$permalink = esc_url( get_permalink( $product->get_id() ) );
		}

		return $permalink;
	}

	/**
	 * Modify the "Add to Cart" button arguments for pre-order products in the product loop.
	 *
	 * @param array      $args The original arguments for the "Add to Cart" button.
	 * @param WC_Product $product The product object.
	 *
	 * @return array
	 * @throws Exception If there is an error in determining the pre-order status.
	 */
	public function change_add_to_cart_args( $args, $product ) {
		if ( PreOrderFns::is_on_pre_order( $product ) || PreOrderFns::variation_is_on_pre_order( $product ) ) {
			$args['href']  = esc_url( get_permalink( $product->get_id() ) );
			$args['class'] = str_replace(
				[ 'ajax_add_to_cart', 'rtsb-mini-cart', 'rtsb-add-to-cart-btn', 'variable-product' ],
				[ '', '', 'rtsb-pre-order-btn', 'variable-product rtsb-pre-order-btn' ],
				$args['class']
			);

			if ( ! empty( $args['title'] ) ) {
				$args['title'] = $this->pre_order_button_label( $args['title'], $product );
			}
		}

		return $args;
	}

	/**
	 * Modify the button text for pre-order products.
	 *
	 * @param string     $button_label The default button label.
	 * @param WC_Product $product The product object.
	 *
	 * @return string
	 * @throws Exception If there is an error in determining the pre-order status.
	 */
	public function pre_order_button_label( $button_label, $product ) {
		$product_id = $product->get_id();

		return $this->get_custom_label( $product, $product_id, $button_label );
	}

	/**
	 * Modify the shop button text for pre-order products.
	 *
	 * @param string     $button_label The default button label.
	 * @param WC_Product $product The product object.
	 *
	 * @return string
	 * @throws Exception If there is an error in determining the pre-order status.
	 */
	public function pre_order_shop_button_label( $button_label, $product ) {
		$product_id = $product->get_id();

		if ( $product->is_type( 'variable' ) ) {
			$variations = $product->get_available_variations();

			foreach ( $variations as $variation ) {
				$variation_id = $variation['variation_id'];
				$button_label = $this->get_custom_label( wc_get_product( $variation['variation_id'] ), $variation_id, $button_label );
			}
		} else {
			$button_label = $this->get_custom_label( $product, $product_id, $button_label );
		}

		return $button_label;
	}

	/**
	 * Disable add to cart for closed pre-order products.
	 *
	 * @param bool       $status The current add to cart status.
	 * @param WC_Product $product The product object.
	 *
	 * @return bool
	 * @throws Exception If there is an error in determining the pre-order status.
	 */
	public function disable_add_to_cart( $status, $product ) {
		$product_id = $product->get_id();

		if ( PreOrderFns::is_on_pre_order( $product )
			&& ! PreOrderFns::is_pre_order_allowed( $product_id ) ) {
			$status = false;
		}

		$has_user_restriction = 'specific' === ( PreOrderFns::get_pre_order_settings_data()['user_access'] ?? null );
		$user_roles           = PreOrderFns::get_pre_order_settings_data()['user_role'];

		if ( $has_user_restriction && ( ! FnsPro::is_visible_for_user( $user_roles ) ) ) {
			$status = false;
		}

		return $status;
	}

	/**
	 * Disable certain features for non-logged-in users.
	 *
	 * @param string     $price The product price.
	 * @param WC_Product $product The product object.
	 *
	 * @return string
	 */
	public function disable_price( $price, $product ) {
		if ( ! $product instanceof WC_Product ) {
			return $price;
		}

		return $this->guest_price_data()['disabled'] ? $this->guest_price_data()['price'] : $price;
	}

	/**
	 * Modify the availability text based on pre-order status.
	 *
	 * @param string     $text    The original text.
	 * @param WC_Product $product The product object.
	 *
	 * @return string
	 * @throws Exception If there is an error in determining the pre-order status.
	 */
	public function pre_order_text( $text, $product ) {
		$product_id = $product->get_id();
		$html       = '';
		$countdown  = false;

		if ( PreOrderFns::is_on_pre_order( $product ) && PreOrderFns::is_pre_order_allowed( $product_id ) ) {
			$text      = wp_kses_post( PreOrderFns::get_pre_order_message( $product_id ) );
			$countdown = PreOrderFns::get_pre_order_settings_data()['show_countdown'];
		}

		$html .= ! empty( $text ) ? '<span class="rtsb-pre-order-text">' . $text . '</span>' : '';

		if ( $countdown ) {
			$html = $html . '</span>' . $this->render_countdown( $product_id );
		}

		return $html;
	}

	/**
	 * Renders countdown timer.
	 *
	 * @param int $product_id The Product ID.
	 *
	 * @return string
	 * @throws Exception If there is an error creating the DateTime objects.
	 */
	private function render_countdown( $product_id ) {
		$remaining_time = PreOrderFns::check_pre_order_status( $product_id )['formatted_time'] ?? [];

		if ( empty( $remaining_time ) ) {
			return '';
		}

		$classes        = [];
		$counter_layout = 'horizontal';
		$classes[]      = 'rtsb-countdown-' . esc_attr( $counter_layout );

		return '<span class="rtsb-pre-order-timer rtsb-countdown-campaign ' . esc_attr( implode( ' ', $classes ) ) . ' " data-date="' . esc_attr( $remaining_time ) . '">';
	}

	/**
	 * Pre-Order calculated pricing and message.
	 *
	 * @param string     $price   The product price.
	 * @param WC_Product $product The product object.
	 *
	 * @return string
	 * @throws Exception If there is an error in determining the pre-order status.
	 */
	public function custom_pricing( $price, $product ) {
		if ( ! $product ) {
			return $price;
		}

		if ( ! PreOrderFns::is_on_pre_order( $product ) ) {
			return $price;
		}

		$html          = '';
		$product_id    = $product->get_id();
		$crossed_price = '';
		$show_price    = PreOrderFns::get_pre_order_settings_data()['show_price'];
		$price         = $product->get_regular_price();
		$price         = AddOnsProduct::convert_currency( self::calculate_preorder_price( $product_id, floatval( $price ) ) );
		//$price         = apply_filters( 'rtsb/module/pre_order/pre_order_price', $price, $product );

		if ( 'show' === $show_price ) {
			$crossed_price .= $product->get_regular_price();
		}

		if ( $this->guest_price_data()['disabled'] ) {
			return $this->guest_price_data()['price'];
		}

		if ( PreOrderFns::is_pre_order_closed( $product_id ) ) {
			$html .= '<span class="rtsb-pre-order-closed">' . esc_html( PreOrderFns::get_pre_order_settings_data()['closed_label'] ) . '</span>';
			$html .= wc_format_sale_price( $crossed_price, $price );

			return $html;
		} else {
			return wc_format_sale_price( $crossed_price, $price );
		}
	}

	/**
	 * Calculate the pre-order pricing for a product.
	 *
	 * @param float      $price    The original price of the product.
	 * @param WC_Product $product  The product object.
	 *
	 * @return float
	 */
	public function pre_order_pricing( $price, $product ) {
		if ( ! $product instanceof WC_Product ) {
			return $price;
		}

		if ( ! apply_filters( 'rtsb/calculate/pre_order/price', true, $product ) ) {
			return $price;
		}

		$type            = $product->get_type();
		$product_id      = $product->get_id();
		$is_on_pre_order = 'variation' === $type
			? PreOrderFns::variation_id_is_on_pre_order( $product_id )
			: PreOrderFns::is_on_pre_order( $product );

		if ( ! $is_on_pre_order ) {
			return $price;
		}

		if ( ( FnsPro::is_module_active( 'product_add_ons' ) )
			&& ( 'variation' === $type || 'simple' === $type ) && AddOnsFns::product_has_addons( $product, 'variation' === $type ) ) {
			return $price;
		}

		$regular_price = $product->get_regular_price();

		if ( ! empty( $regular_price ) ) {
			$price = AddOnsProduct::convert_currency( self::calculate_preorder_price( $product_id, floatval( $regular_price ) ) );

			return apply_filters( 'rtsb/module/pre_order/pre_order_price', $price, $product );
		}

		return apply_filters( 'rtsb/module/pre_order/pre_order_price', $price, $product );
	}

	/**
	 * Adds pre-order data to product variations.
	 *
	 * @param array $variations The variation data.
	 *
	 * @return array
	 */
	public function pre_order_variation_data( $variations ) {
		if ( PreOrderFns::is_on_pre_order( wc_get_product( $variations['variation_id'] ) ) ) {
			$variations['rtsb_is_pre_order']    = 'yes';
			$variations['rtsb_pre_order_label'] = PreOrderFns::get_pre_order_settings_data()['button_label'];

			$product         = wc_get_product( $variations['variation_id'] );
			$regular_price   = floatval( $product->get_regular_price() );
			$pre_order_price = AddOnsProduct::convert_currency( self::calculate_preorder_price( $variations['variation_id'], $regular_price ) );

			if ( ! empty( $pre_order_price ) ) {
				$variations['price_html'] = '<span class="price">' . wc_format_sale_price( $regular_price, $pre_order_price ) . '</span>';
			}
		}

		return $variations;
	}

	/**
	 * Modify the WooCommerce variation prices array.
	 *
	 * @param array      $prices The WooCommerce variation prices array.
	 * @param WC_Product $product The product object.
	 *
	 * @return array The modified prices array.
	 */
	public function woocommerce_variation_prices_array( $prices, $product ) {
		if ( ! PreOrderFns::variation_is_on_pre_order( $product ) ) {
			return $prices;
		}

		if ( is_array( $prices['price'] ) ) {
			foreach ( $prices['price'] as $product_id => $price ) {
				$prices['price'][ $product_id ] = $this->get_pre_order_price_for_variation( $price, $product_id, $product );
			}
		}

		return $prices;
	}

	/**
	 * Calculates the pre-order price for a product variation.
	 *
	 * @param float      $price The original price of the product variation.
	 * @param int        $variation_id The ID of the product variation.
	 * @param WC_Product $product The product object.
	 *
	 * @return float
	 */
	private function get_pre_order_price_for_variation( $price, $variation_id, $product ) {
		if ( ! PreOrderFns::variation_is_on_pre_order( $product ) ) {
			return $price;
		}

		$variation = wc_get_product( $variation_id );

		$regular_price = $variation->get_price();

		if ( $regular_price ) {
			return self::calculate_preorder_price( $variation_id, floatval( $regular_price ) );
		}

		return $price;
	}

	/**
	 * Adjust the product class based on pre-order status.
	 *
	 * @param string     $class   The original class.
	 * @param WC_Product $product Product object.
	 *
	 * @return string
	 * @throws Exception If there is an error in determining the pre-order status.
	 */
	public function pre_order_class( $class, $product ) {
		$new_class = $this->adjust_pre_order_display(
			$product,
			[
				'pre_order' => 'available-on-pre-order',
			]
		);

		return ! empty( $new_class ) ? $new_class : $class;
	}

	/**
	 * Adjust the product icon based on pre-order status.
	 *
	 * @param string $icon        The original icon.
	 * @param array  $controllers The controllers for managing icons.
	 * @param object $product     Product object.
	 *
	 * @return string
	 * @throws Exception If there is an error in determining the pre-order status.
	 */
	public function pre_order_icon( $icon, $controllers, $product ) {
		$new_icon = $this->adjust_pre_order_display(
			$product,
			[
				'pre_order' => Fns::icons_manager( $controllers['stock_pre_order_icon'] ),
			]
		);

		return ! empty( $new_icon ) ? $new_icon : $icon;
	}

	/**
	 * Custom the button label for pre-order products.
	 *
	 * @param WC_Product $product The product object.
	 * @param int        $product_id The product ID.
	 * @param string     $button_label The default button label.
	 *
	 * @return string
	 * @throws Exception If there is an error in determining the pre-order status.
	 */
	private function get_custom_label( $product, $product_id, $button_label ) {
		if ( PreOrderFns::is_on_pre_order( $product ) && PreOrderFns::is_pre_order_allowed( $product_id ) ) {
			$button_label = esc_html( PreOrderFns::get_pre_order_settings_data()['button_label'] );
		}

		return $button_label;
	}

	/**
	 * Adjust the pre-order display property based on product and context.
	 *
	 * @param object $product Product object.
	 * @param array  $context The context for adjusting the display property.
	 *
	 * @return string
	 * @throws Exception If there is an error in determining the pre-order status.
	 */
	private function adjust_pre_order_display( $product, $context = [] ) {
		$product_id = $product->get_id();
		$prop       = '';

		if ( PreOrderFns::is_on_pre_order( $product ) && PreOrderFns::is_pre_order_allowed( $product_id ) ) {
			$prop = $context['pre_order'];
		}

		return $prop;
	}

	/**
	 * Calculate the pre-order price based on the provided arguments.
	 *
	 * @param int   $product_id Product ID.
	 * @param float $regular_price Product regular price.
	 *
	 * @return float|bool
	 */
	public static function calculate_preorder_price( int $product_id, float $regular_price ) {
		if ( ! $product_id || ! $regular_price ) {
			return $regular_price;
		}
		// if ( ! apply_filters( 'rtsb/calculate/pre_order/price', true, wc_get_product( $product_id ) ) ) {
		// return $regular_price;
		// }
		$product_meta        = PreOrderFns::get_pre_order_meta_data( $product_id );
		$pricing_option      = RenderHelpers::get_data( $product_meta, 'preorder_price_option', 'regular' );
		$fixed_price         = RenderHelpers::get_data( $product_meta, 'preorder_price', 0 );
		$discount_percentage = RenderHelpers::get_data( $product_meta, 'preorder_percent_discount', 0 );
		$discount_fixed      = RenderHelpers::get_data( $product_meta, 'preorder_fixed_discount', 0 );
		$increase_percentage = RenderHelpers::get_data( $product_meta, 'preorder_percent_increase', 0 );
		$increase_fixed      = RenderHelpers::get_data( $product_meta, 'preorder_fixed_increase', 0 );

		switch ( $pricing_option ) {
			case 'fixed':
				return $fixed_price;

			case 'discount_percentage':
				return $regular_price * ( 1 - $discount_percentage / 100 );

			case 'discount_fixed':
				return max( 0, $regular_price - $discount_fixed ); // Ensuring the price doesn't go below zero.

			case 'increase_percentage':
				return $regular_price * ( 1 + $increase_percentage / 100 );

			case 'increase_fixed':
				return $regular_price + $increase_fixed;

			case 'regular':
			default:
				return $regular_price;
		}
	}

	/**
	 * Retrieve price data for guest users.
	 *
	 * @return array
	 */
	private function guest_price_data() {
		if ( is_user_logged_in() ) {
			return [ 'disabled' => false ];
		}

		$settings       = PreOrderFns::get_pre_order_settings_data();
		$price_disabled = 'on' === $settings['guest_pricing'];
		$custom_price   = $settings['guest_text'];
		$price_html     = '';

		if ( $price_disabled ) {
			$price_html = ! empty( $custom_price ) ? '<span class="rtsb-price-disabled"><a href="' . get_permalink( wc_get_page_id( 'myaccount' ) ) . '">' . esc_html( $custom_price ) . '</a></span>' : '';
		}

		return [
			'disabled' => $price_disabled,
			'price'    => $price_html,
		];
	}
}
