<?php
/**
 * Pre-order Order Class.
 *
 * @package RadiusTheme\SBPRO
 */

namespace RadiusTheme\SBPRO\Modules\PreOrder\Frontend;

use RadiusTheme\SBPRO\Traits\SingletonTrait;
use RadiusTheme\SBPRO\Modules\PreOrder\PreOrderFns;

defined( 'ABSPATH' ) || exit();

/**
 * Pre-order Cart Page Class.
 */
class Order {
	/**
	 * Singleton Trait.
	 */
	use SingletonTrait;

	/**
	 * Module Class Constructor.
	 */
	private function __construct() {
		/**
		 * Thank you page.
		 */
		add_action( 'woocommerce_thankyou', [ $this, 'change_order_status' ] );
		add_action( 'rtsb_woocommerce_thankyou', [ $this, 'change_order_status' ] );

		/**
		 * Order meta.
		 */
		add_action( 'woocommerce_checkout_create_order_line_item', [ $this, 'add_order_meta' ], 99, 4 );
		add_filter( 'woocommerce_order_item_get_formatted_meta_data', [ $this, 'get_order_meta' ], 10, 4 );
	}

	/**
	 * Change the order status to pre-ordered if one or more items are on backorder.
	 *
	 * @param int $order_id The ID of the order to check and update.
	 *
	 * @return void
	 */
	public function change_order_status( $order_id ) {
		$order = wc_get_order( $order_id );

		if ( ! $order || $order->has_status( 'wc-rtsb-preordered' ) ) {
			return;
		}

		if ( $order->get_meta( '_rtsb_pre_order_email_sent' ) ) {
			return;
		}

		$pre_order_email = false;

		foreach ( $order->get_items() as $item ) {
			$product_id = $item->get_variation_id() ?: $item->get_product_id();
			$product    = wc_get_product( $product_id );

			if ( $this->set_pre_order_status() && PreOrderFns::is_on_pre_order( $product ) ) {
				$pre_order_email = true;
				$order->update_status( 'wc-rtsb-preordered', esc_html__( 'Order status changed to pre-ordered because one or more items are on pre-order.', 'shopbuilder-pro' ) );

				break;
			}
		}

		if ( $pre_order_email ) {
			$this->send_custom_emails( $order );

			$order->update_meta_data( '_rtsb_pre_order_email_sent', true );
			$order->save();
		}
	}

	/**
	 * Send custom email.
	 *
	 * @param object $order The order object.
	 *
	 * @return void
	 */
	private function send_custom_emails( $order ) {
		$settings = PreOrderFns::get_pre_order_settings_data();

		WC()->mailer();

		if ( 'on' === $settings['new_email'] ) {
			do_action( 'rtsb/module/pre_order/order_received_email', $order );
		}

		if ( 'on' === $settings['confirm_email'] ) {
			do_action( 'rtsb/module/pre_order/order_confirmed_email', $order );
		}
	}

	/**
	 * Add order meta-data for a product if it is on pre-order.
	 *
	 * @param mixed  $item The item being added to the order.
	 * @param string $cart_item_key The key of the item in the cart.
	 * @param array  $values The values of the item.
	 * @param object $order The order object.
	 *
	 * @return void
	 */
	public function add_order_meta( $item, $cart_item_key, $values, $order ) {
		$product    = $values['data'];
		$product_id = $product->get_id();

		if ( PreOrderFns::is_on_pre_order( $product ) ) {
			$availability   = PreOrderFns::get_pre_order_date( $product_id );
			$formatted_date = implode( ' - ', PreOrderFns::get_pre_order_formatted_date( $availability ) );

			if ( empty( $availability ) ) {
				$formatted_date = esc_html__( 'This product will be available soon.', 'shopbuilder-pro' );
			}

			$pre_order_availability = '<span class="rtsb-pre-order-availability-text">' . esc_html( $formatted_date ) . '</span>';
			$text                   = '<span class="rtsb-pre-order-text-badge">' . esc_html__( 'Pre-Order', 'shopbuilder-pro' ) . '</span>';

			$item->add_meta_data( 'rtsb_preorder_text', $text, true );
			$item->add_meta_data( 'rtsb_preorder_availability', $pre_order_availability, true );
			$order->update_meta_data( '_rtsb_pre_order_product_id', $product_id );
		}
	}

	/**
	 * Retrieves the order metadata and formats it.
	 *
	 * @param array $formatted_meta The formatted meta data.
	 *
	 * @return array
	 */
	public function get_order_meta( $formatted_meta ) {
		foreach ( $formatted_meta as $key => $meta ) {
			if ( 'rtsb_preorder_availability' === $meta->key ) {
				$meta->display_key = '<span class="rtsb-order-meta order-status status-rtsb-pre-ordered"><span class="wc-item-meta-label rtsb-stock-text">' . esc_html__( 'Availability Date', 'shopbuilder-pro' ) . '</span></span>';
			}

			if ( 'rtsb_preorder_text' === $meta->key ) {
				$meta->display_key = '<span class="rtsb-order-meta order-status status-rtsb-pre-ordered"><span class="wc-item-meta-label rtsb-stock-text">' . esc_html__( 'Order Type', 'shopbuilder-pro' ) . '</span></span>';
			}
		}

		return $formatted_meta;
	}

	/**
	 * Set Pre-order status.
	 *
	 * @return bool
	 */
	private function set_pre_order_status() {
		$status = PreOrderFns::get_pre_order_settings_data()['order_status'];

		return 'custom' === $status;
	}
}
