<?php
/**
 * Pre-order Module Common Class.
 *
 * @package RadiusTheme\SBPRO
 */

namespace RadiusTheme\SBPRO\Modules\PartialPay;

use DateTime;
use WC_Order;
use Exception;
use DateMalformedStringException;
use RadiusTheme\SBPRO\Traits\SingletonTrait;
use RadiusTheme\SBPRO\Modules\PartialPay\Notification\{
	PartialPayOnHoldEmail,
	OrderConfirmationEmail,
	PartialPayReminderEmail,
	PartialPayReceivedEmail
};

defined( 'ABSPATH' ) || exit();

/**
 * PartialPay Module Common Class.
 */
class PartialPayCommon {
	/**
	 * Singleton Trait.
	 */
	use SingletonTrait;

	/**
	 * Module Class Constructor.
	 */
	private function __construct() {
		$this->add_notification_emails();
		$this->add_notification_schedule();
	}

	/**
	 * Add Pre-Order emails.
	 *
	 * @return void
	 */
	public function add_notification_emails() {
		add_filter( 'woocommerce_email_classes', [ $this, 'register_partial_pay_emails' ] );
		add_filter( 'woocommerce_locate_core_template', [ $this, 'locate_templates' ], 10, 2 );

		// Disable default emails.
		$settings = PartialPayFns::get_settings_data();

		if ( 'on' === $settings['new_email'] ) {
			add_filter( 'woocommerce_email_enabled_new_order', [ $this, 'disable_default_emails' ], 10, 2 );
		}

		if ( 'on' === $settings['confirm_email'] ) {
			add_filter( 'woocommerce_email_enabled_customer_processing_order', [ $this, 'disable_default_emails' ], 10, 2 );
		}
	}

	/**
	 * Registers the custom partial pay email classes with WooCommerce.
	 *
	 * @param array $email_classes An array of existing WooCommerce email classes.
	 *
	 * @return array
	 */
	public function register_partial_pay_emails( $email_classes ) {
		$settings = PartialPayFns::get_settings_data();

		if ( 'on' === $settings['new_email'] ) {
			$email_classes['RTSB_Email_New_Partial_Pay_Received'] = new PartialPayReceivedEmail();
		}

		if ( 'on' === $settings['confirm_email'] ) {
			$email_classes['RTSB_Email_Partial_Pay_Confirmed'] = new OrderConfirmationEmail();
		}

		if ( 'on' === $settings['reminder_email'] ) {
			$email_classes['RTSB_Email_Partial_Pay_Reminder'] = new PartialPayReminderEmail();
		}

		if ( 'on' === $settings['on_hold_email'] ) {
			$email_classes['RTSB_Email_Partial_Pay_On_Hold'] = new PartialPayOnHoldEmail();
		}

		return $email_classes;
	}

	/**
	 * Locates custom email templates.
	 *
	 * @param string $core_file The path to the core template file.
	 * @param string $template The requested template file.
	 *
	 * @return string
	 */
	public function locate_templates( $core_file, $template ) {
		$custom_templates = [
			'emails/partial-pay-received.php',
			'emails/partial-pay-confirmed.php',
			'emails/partial-pay-reminder.php',
			'emails/partial-pay-on-hold.php',
		];

		if ( in_array( $template, $custom_templates, true ) ) {
			$core_file = rtsbpro()->get_plugin_template_path() . $template;
		}

		return $core_file;
	}

	/**
	 * Add Partial Pay scheduling.
	 *
	 * @return void
	 */
	public function add_notification_schedule() {
		/**
		 * Partial Pay scheduling.
		 */
		register_activation_hook( RTSBPRO_FILE, [ $this, 'start_due_reminder_schedule' ] );
		register_deactivation_hook( RTSBPRO_FILE, [ $this, 'end_due_reminder_schedule' ] );

		// Manually start the schedule.
		$this->start_due_reminder_schedule();

		/**
		 * Partial Pay.
		 */
		add_action( 'rtsb/module/partial_pay/due_reminder', [ $this, 'due_reminder' ] );
	}

	/**
	 * Schedule the availability check.
	 *
	 * @return void
	 */
	public function start_due_reminder_schedule() {
		if ( ! wp_next_scheduled( 'rtsb/module/partial_pay/due_reminder' ) ) {
			wp_schedule_event( time(), 'twicedaily', 'rtsb/module/partial_pay/due_reminder' );
		}
	}

	/**
	 * Clear the scheduled availability check.
	 *
	 * @return void
	 */
	public function end_due_reminder_schedule() {
		wp_clear_scheduled_hook( 'rtsb/module/partial_pay/due_reminder' );
	}

	/**
	 * Check pre-order availability.
	 *
	 * @return void
	 * @throws Exception If there is an error.
	 */
	public function due_reminder() {
		$settings = PartialPayFns::get_settings_data();

		if ( 'on' !== $settings['reminder_email'] ) {
			return;
		}

		$days_before_due = ! empty( $settings['due_date_reminder_days'] ) ? absint( $settings['due_date_reminder_days'] ) : 7;
		$target_date     = ( new DateTime( 'now', wp_timezone() ) )->modify( "+{$days_before_due} days" )->format( 'Y-m-d' );
		$today           = new DateTime( 'now', wp_timezone() );

		$orders = wc_get_orders(
			[
				'post_type'  => 'shop_order',
				'status'     => [ 'wc-rtsb-partial-pay', 'on-hold', 'processing' ],
				'limit'      => -1,
				// phpcs:ignore WordPress.DB.SlowDBQuery.slow_db_query_meta_query
				'meta_query' => [
					[
						'key'     => '_rtsb_partial_expiry_date',
						'value'   => $target_date,
						'compare' => '=',
						'type'    => 'DATE',
					],
				],
			]
		);

		foreach ( $orders as $order ) {
			$expiry_date_raw = $order->get_meta( '_rtsb_partial_expiry_date' );

			if ( empty( $expiry_date_raw ) ) {
				continue;
			}

			$expiry_date = DateTime::createFromFormat( 'Y-m-d', $expiry_date_raw, wp_timezone() );

			if ( ! $expiry_date instanceof DateTime ) {
				continue;
			}

			$expiry_date = $expiry_date->format( 'Y-m-d' );

			if ( $today > $expiry_date ) {
				$order->update_status( 'on-hold', __( 'Partial payment overdue. Order set to on-hold.', 'shopbuilder-pro' ) );
				$order->add_order_note( __( 'Balance payment was not completed by due date. Order status changed to on-hold.', 'shopbuilder-pro' ) );

				WC()->mailer();
				do_action( 'rtsb/module/partial_pay/on_hold_email', $order );

				continue;
			}

			foreach ( $order->get_items() as $item ) {
				$product = $item->get_product();

				if ( ! $product ) {
					continue;
				}

				$sent_key = '_rtsb_due_reminder_sent_' . $product->get_id() . '_' . $order->get_id();

				if ( 'yes' === get_post_meta( $order->get_id(), $sent_key, true ) ) {
					continue;
				}

				if ( $this->should_send_due_reminder( $order, $product, $settings, $sent_key ) ) {
					WC()->mailer();
					do_action( 'rtsb/module/partial_pay/reminder_email', $order );

					$order->add_order_note( __( 'Sent automated remaining balance email to customer.', 'shopbuilder-pro' ) );
					update_post_meta( $order->get_id(), $sent_key, 'yes' );
				}
			}
		}
	}

	/**
	 * Check if email should be sent.
	 *
	 * @param object $order WooCommerce order object.
	 * @param object $product WooCommerce product object.
	 * @param array  $settings Partial Pay settings.
	 * @param string $sent_key Sent key.
	 *
	 * @return bool
	 * @throws DateMalformedStringException If the date is malformed.
	 */
	private function should_send_due_reminder( $order, $product, $settings, $sent_key ) {
		$product_metas        = $product->get_meta( '_rtsb_product_partial_pay', true );
		$global_due_days      = $settings['payment_date'];
		$global_reminder_days = $settings['days_to_send_mail'];

		$send_mail_days = isset( $product_metas['send_mail'] ) ? absint( $product_metas['send_mail'] ) : absint( $global_reminder_days );
		$due_days       = isset( $product_metas['due_date'] ) ? absint( $product_metas['due_date'] ) : absint( $global_due_days );

		if ( $send_mail_days < 1 || $due_days < 1 ) {
			return false;
		}

		if ( 'yes' === $order->get_meta( $sent_key ) ) {
			return false;
		}

		$timezone      = wp_timezone();
		$order_created = new DateTime( $order->get_date_created(), $timezone );

		$due_date       = ( clone $order_created )->modify( "+$due_days days" );
		$reminder_start = ( clone $due_date )->modify( "-$send_mail_days days" );
		$current_date   = new DateTime( 'now', $timezone );
		$today          = $current_date->format( 'Y-m-d' );
		$reminder_start = $reminder_start->format( 'Y-m-d' );
		$due_date       = $due_date->format( 'Y-m-d' );

		return $today >= $reminder_start && $today <= $due_date;
	}

	/**
	 * Disables default emails for orders containing pre-order products.
	 *
	 * @param bool     $enabled Whether the email is enabled. Default is true.
	 * @param WC_Order $order   The WooCommerce order object.
	 *
	 * @return bool
	 */
	public function disable_default_emails( $enabled, $order ) {
		if ( ! $order || ! PartialPayFns::order_contains_partial_pay_products( $order ) ) {
			return $enabled;
		}

		do_action( 'rtsb/module/partial_pay/order_received_email', $order );
		do_action( 'rtsb/module/partial_pay/order_confirmed_email', $order );

		if ( true === apply_filters( 'rtsb/partial_pay/enabled_default_emails', false, $order ) ) {
			return $enabled;
		}

		return false;
	}
}
