<?php
/**
 * Partial Pay Order Handler Class.
 *
 * @package RadiusTheme\SBPRO
 */

namespace RadiusTheme\SBPRO\Modules\PartialPay\Frontend;

use RadiusTheme\SB\Helpers\Fns;
use RadiusTheme\SBPRO\Helpers\FnsPro;
use RadiusTheme\SBPRO\Modules\AddOns\Frontend\Product as AddOnsProduct;
use RadiusTheme\SBPRO\Modules\PartialPay\Notification\PartialPayReminderEmail;
use RadiusTheme\SBPRO\Modules\PreOrder\Frontend\Product;
use RadiusTheme\SBPRO\Modules\PreOrder\PreOrderFns;
use WC_Order_Item_Fee;
use WC_Data_Exception;
use WC_Order_Item_Product;
use RadiusTheme\SBPRO\Traits\SingletonTrait;
use RadiusTheme\SBPRO\Modules\PartialPay\PartialPayFns;

defined( 'ABSPATH' ) || exit();

/**
 * Partial Pay Order Handler Class.
 */
class OrderHandler {
	/**
	 * Singleton Trait.
	 */
	use SingletonTrait;

	/**
	 * Module Class Constructor.
	 */
	private function __construct() {
		/**
		 * Actions.
		 */
		add_action( 'woocommerce_thankyou', [ $this, 'handle_order_status' ] );
		add_action( 'rtsb_woocommerce_thankyou', [ $this, 'handle_order_status' ] );
		add_action( 'woocommerce_checkout_create_order_line_item', [ $this, 'add_item_meta' ], 99, 4 );
		add_action( 'woocommerce_checkout_order_created', [ $this, 'add_order_meta' ] );
		add_action( 'woocommerce_order_action_rtsb_send_partial_payment_email', [ $this, 'handle_send_partial_payment_email' ] );
		add_action( 'woocommerce_before_pay_action', [ $this, 'handle_balance_payment' ] );
		add_action( 'before_woocommerce_pay_form', [ $this, 'balance_payment_notice' ] );
		add_action( 'template_redirect', [ $this, 'maybe_redirect_with_flag' ] );

		/**
		 * Filters.
		 */
		add_filter( 'woocommerce_order_item_get_formatted_meta_data', [ $this, 'get_order_meta' ], 10, 4 );
		add_filter( 'woocommerce_hidden_order_itemmeta', [ $this, 'hide_admin_meta' ], 10, 1 );
		add_filter( 'woocommerce_order_actions', [ $this, 'maybe_add_partial_payment_email_action' ], 10, 2 );
		add_filter( 'woocommerce_before_calculate_totals', [ $this, 'apply_custom_price' ], 99 );
		add_filter( 'woocommerce_my_account_my_orders_actions', [ $this, 'add_custom_action_button' ], 10, 2 );
		add_filter( 'woocommerce_pay_order_button_text', [ $this, 'change_pay_button_text' ] );
		add_filter( 'woocommerce_get_order_item_totals', [ $this, 'modify_cart_info' ], 20, 2 );
		add_filter( 'woocommerce_valid_order_statuses_for_payment', [ $this, 'allow_partial_pay_status' ], 10, 2 );
		add_filter( 'woocommerce_order_get_items', [ $this, 'adjust_order_items' ], 20 );
		add_filter( 'woocommerce_order_formatted_line_subtotal', [ $this, 'show_partial_paid_label' ], 20, 2 );
	}

	/**
	 * Handle order status update.
	 *
	 * @param int $order_id The ID of the order to check and update.
	 *
	 * @return void
	 */
	public function handle_order_status( $order_id ) {
		$order = wc_get_order( $order_id );

		$due_paid = $order->get_meta( '_rtsb_partial_due_paid' );

		if ( $order->has_status( 'on-hold' ) && $due_paid ) {
			$order->update_status( 'processing', __( 'Payment received. Order resumed.', 'shopbuilder-pro' ) );
			$order->add_order_note( __( 'Balance payment received. Order resumed.', 'shopbuilder-pro' ) );

			return;
		}

		if ( ! $order || $order->has_status( 'wc-rtsb-partial-pay' ) ) {
			return;
		}

		$is_remaining_payment = false;

		foreach ( $order->get_items() as $item ) {
			$product_id = $item->get_variation_id() ?: $item->get_product_id();
			$product    = wc_get_product( $product_id );

			if ( ! $product ) {
				continue;
			}

			foreach ( $order->get_items( 'fee' ) as $fee_item_id => $fee_item ) {
				if ( $fee_item->get_name() === 'Remaining Balance Due' ) {
					$order->remove_item( $fee_item_id );
					$is_remaining_payment = true;
				}
			}

			$total_quantity = 0;

			// Case 1: This is a remaining balance payment.
			if ( $is_remaining_payment ) {
				if ( ! $due_paid ) {
					$this->update_order_items( $order, $total_quantity );
				}
			}

			// Case 2: This is a partial payment.
			if ( ! $is_remaining_payment ) {
				$is_on_pre_order = Fns::is_module_active( 'pre_order' ) && 'variation' === $product->get_type()
					? PreOrderFns::variation_id_is_on_pre_order( $product_id )
					: PreOrderFns::is_on_pre_order( $product );

				if ( ! $is_on_pre_order
					&& PartialPayFns::product_has_partial_pay( $product, 'variation' === $product->get_type() ) ) {
					$order->update_status(
						'wc-rtsb-partial-pay',
						esc_html__( 'Order status changed to partially paid because one or more items are partially paid.', 'shopbuilder-pro' )
					);

					break;
				}
			}
		}
	}

	/**
	 * Add order meta-data for a product if it is on pre-order.
	 *
	 * @param mixed  $item The item being added to the order.
	 * @param string $cart_item_key The key of the item in the cart.
	 * @param array  $values The values of the item.
	 * @param object $order The order object.
	 *
	 * @return void
	 */
	public function add_item_meta( $item, $cart_item_key, $values, $order ) {
		if ( $order->get_meta( '_rtsb_partial_meta_applied' ) ) {
			return;
		}

		$product_data = $values['data'];
		$is_variation = 'variation' === $product_data->get_type();
		$product_id   = $is_variation ? $product_data->get_parent_id() : $product_data->get_id();
		$product      = wc_get_product( $product_data->get_id() );

		if ( PartialPayFns::product_has_partial_pay( $product, $is_variation ) ) {
			$full_price     = FnsPro::get_module_compatible_price( $product );
			$variation_id   = $values['variation_id'] ?? 0;
			$partial_amount = PartialPayFns::get_partial_amount( $product_id, $variation_id );

			$paid     = $partial_amount[0]['amount'] ?? 0;
			$due      = floatval( $full_price ) - floatval( $paid );
			$due_date = $partial_amount[0]['due_date'] ?? '';

			$item->add_meta_data( '_rtsb_is_partial_payment', '<span class="rtsb-pre-order-availability-text partial-pay">Active</span>' );
			$item->add_meta_data( 'rtsb_partial_total', wc_price( $full_price ) );
			$item->add_meta_data( 'rtsb_partial_paid', wc_price( $paid ) );
			$item->add_meta_data( 'rtsb_partial_due', wc_price( $due ) );
			$item->add_meta_data( '_rtsb_partial_due_date', $due_date );
			$item->add_meta_data( '_rtsb_partial_total_raw', $full_price );
			$item->add_meta_data( '_rtsb_partial_paid_raw', $paid );
			$item->add_meta_data( 'rtsb_partial_formatted_due_date', '<span class="rtsb-pre-order-availability-text">' . gmdate( 'F j, Y', strtotime( $due_date ) ) . '</span>' );

			$order->update_meta_data( '_rtsb_partial_due_paid', false );
			$order->update_meta_data( '_rtsb_partial_expiry_date', $due_date );
			$order->update_meta_data( '_rtsb_partial_installments', $partial_amount );
			$order->update_meta_data( '_rtsb_partial_pay_order_product_id', $product_id );
			$order->update_meta_data( '_rtsb_partial_meta_applied', true );
		}
	}

	/**
	 * Add order meta-data for partial pay.
	 *
	 * @param object $order The order object.
	 *
	 * @return void
	 */
	public function add_order_meta( $order ) {
		if ( $order->get_meta( '_rtsb_order_meta_applied' ) ) {
			return;
		}

		$due   = 0;
		$total = 0;

		foreach ( $order->get_items() as $item ) {
			$partial_total = floatval( $item->get_meta( '_rtsb_partial_total_raw' ) );
			$partial_paid  = floatval( $item->get_meta( '_rtsb_partial_paid_raw' ) );
			$total        += floatval( $item->get_meta( '_rtsb_partial_total_raw' ) );

			if ( $partial_total && $partial_paid < $partial_total ) {
				$due += $partial_total - $partial_paid;
			}

			if ( '<span class="rtsb-pre-order-availability-text partial-pay">Active</span>' === $item->get_meta( '_rtsb_is_partial_payment' ) ) {
				$item->update_meta_data( '_rtsb_original_order_id', $order->get_id() );

				break;
			}
		}

		$order->update_meta_data( '_rtsb_partial_total_price', $total );
		$order->update_meta_data( '_rtsb_order_meta_applied', true );

		if ( $due ) {
			$order->update_meta_data( '_rtsb_partial_due', $due );
		}

		$order->save();
	}

	/**
	 * Send partial payment email.
	 *
	 * @param object $order The order object.
	 *
	 * @return void
	 */
	public function handle_send_partial_payment_email( $order ) {
		$customer_email = $order->get_billing_email();
		$product_id     = $order->get_meta( '_rtsb_partial_pay_order_product_id' );
		$product        = wc_get_product( $product_id );
		$partial_amount = $order->get_meta( '_rtsb_partial_installments' );

		if ( empty( $partial_amount ) || ! is_array( $partial_amount ) ) {
			$order->add_order_note( __( 'Payment link email not sent — no installment data found.', 'shopbuilder-pro' ) );

			return;
		}

		$paid  = $partial_amount[0]['amount'] ?? 0;
		$total = $product->get_price();
		$due   = floatval( $total ) - floatval( $paid );

		if ( $due <= 0 || ! $customer_email ) {
			$order->add_order_note( __( 'Payment link email not sent — no due amount or missing email.', 'shopbuilder-pro' ) );
			return;
		}

		WC()->mailer();
		do_action( 'rtsb/module/partial_pay/reminder_email', $order );

		$order->add_order_note( __( 'Sent remaining balance email to customer.', 'shopbuilder-pro' ) );
	}

	/**
	 * Handle partial payment checkout.
	 *
	 * @param object $order The order object.
	 *
	 * @return void
	 * @throws WC_Data_Exception Exception.
	 */
	public function handle_balance_payment( $order ) {
		// phpcs:ignore WordPress.Security.NonceVerification.Recommended
		if ( empty( $_REQUEST['pay_for_order'] ) && empty( $_REQUEST['rtsb_partial_pay_order'] && $order->get_status() !== 'rtsb-partial-pay' ) ) {
			return;
		}

		if ( $order->get_meta( '_rtsb_adjustment_done' ) ) {
			return;
		}

		if ( $order->get_meta( '_rtsb_partial_due_paid' ) ) {
			wc_add_notice( __( 'You have no remaining balance to pay for this order.', 'shopbuilder-pro' ), 'notice' );
			wp_safe_redirect( wc_get_page_permalink( 'myaccount' ) );

			exit;
		}

		// Get payment details.
		$original_total = $order->get_meta( '_rtsb_partial_total_price' ) ?: $order->get_total();
		$installments   = $order->get_meta( '_rtsb_partial_installments' );
		$paid           = $installments[0]['amount'] ?? 0;
		$due            = floatval( $original_total ) - floatval( $paid );

		if ( $due <= 0 ) {
			wc_add_notice( __( 'You have no remaining balance to pay for this order.', 'shopbuilder-pro' ), 'notice' );
			wp_safe_redirect( wc_get_page_permalink( 'myaccount' ) );

			exit;
		}

		$total_quantity = 0;
		$total_due      = $due;

		foreach ( $order->get_items() as $item ) {
			$total_quantity += $item->get_quantity();
			$per_item_due    = $original_total - $paid;
			$total_due       = $per_item_due * $total_quantity;

			$item->set_total( 0 );
			$item->set_subtotal( 0 );
		}

		$fee = new WC_Order_Item_Fee();

		$fee->set_name( __( 'Remaining Balance Due', 'shopbuilder-pro' ) );
		$fee->set_amount( $total_due );
		$fee->set_total( $total_due );
		$fee->set_tax_class( '' );
		$fee->set_tax_status( 'none' );

		$order->add_item( $fee );
		$order->calculate_totals();
		$order->add_order_note(
			sprintf(
				/* translators: %s: Total due. */
				__( 'Remaining balance payment initialized. Due: %s', 'shopbuilder-pro' ),
				wc_price( $total_due ) . ' (' . $total_quantity . ' <small class="times">×</small> ' . wc_price( $order->get_meta( '_rtsb_partial_due' ) ) . ')'
			)
		);
		$order->update_meta_data( '_rtsb_adjustment_done', true );
		$order->save();
	}

	/**
	 * Display notice for balance payment.
	 *
	 * @return void
	 */
	public function balance_payment_notice() {
		if ( $this->is_partial_payment_order_pay_page() ) {
			wc_print_notice( __( 'You are now paying the remaining balance of your order.', 'shopbuilder-pro' ), 'notice' );
		}
	}

	/**
	 * Capture a partial-payment flag for non-logged-in users and store in cookie.
	 *
	 * @return void
	 */
	public function maybe_redirect_with_flag() {
		// phpcs:ignore WordPress.Security.NonceVerification.Recommended
		if ( ! is_user_logged_in() && is_wc_endpoint_url( 'order-pay' ) && isset( $_GET['rtsb_partial_pay_order'] ) && isset( $_GET['key'] ) ) {
			$order_id = get_query_var( 'order-pay' );

			setcookie( 'rtsb_partial_pay_flag', '1', time() + 3600, COOKIEPATH, COOKIE_DOMAIN );
			setcookie( 'rtsb_order_id', $order_id, time() + 3600, COOKIEPATH, COOKIE_DOMAIN );
		}

		if ( is_user_logged_in() && isset( $_COOKIE['rtsb_partial_pay_flag'], $_COOKIE['rtsb_order_id'] ) ) {
			$order_id = absint( $_COOKIE['rtsb_order_id'] );
			$order    = wc_get_order( $order_id );

			// phpcs:ignore WordPress.WP.Capabilities.Unknown
			if ( $order && current_user_can( 'view_order', $order_id ) ) {
				setcookie( 'rtsb_partial_pay_flag', '', time() - 3600, COOKIEPATH, COOKIE_DOMAIN );
				setcookie( 'rtsb_order_id', '', time() - 3600, COOKIEPATH, COOKIE_DOMAIN );

				unset( $_COOKIE['rtsb_partial_pay_flag'], $_COOKIE['rtsb_order_id'] );

				wp_safe_redirect( $order->get_checkout_payment_url() . '&rtsb_partial_pay_order=true' );
				exit;
			}
		}
	}

	/**
	 * Retrieves the order metadata and formats it.
	 *
	 * @param array $formatted_meta The formatted meta data.
	 *
	 * @return array
	 */
	public function get_order_meta( $formatted_meta ) {
		foreach ( $formatted_meta as $meta ) {
			if ( '_rtsb_is_partial_payment' === $meta->key ) {
				$meta->display_key = esc_html__( 'Partial Pay Status', 'shopbuilder-pro' );
			}

			if ( 'rtsb_partial_total' === $meta->key ) {
				$meta->display_key = esc_html__( 'Product Price', 'shopbuilder-pro' );
			}

			if ( 'rtsb_partial_paid' === $meta->key ) {
				$meta->display_key = esc_html__( 'Amount Paid', 'shopbuilder-pro' );
			}

			if ( 'rtsb_partial_due' === $meta->key ) {
				$meta->display_key = esc_html__( 'Balance Due', 'shopbuilder-pro' );
			}

			if ( 'rtsb_partial_formatted_due_date' === $meta->key ) {
				$meta->display_key = esc_html__( 'Next Payment Due', 'shopbuilder-pro' );
			}

			if ( '_rtsb_partially_paid' === $meta->key ) {
				$meta->display_key = esc_html__( 'Partially Paid', 'shopbuilder-pro' );
			}

			if ( '_rtsb_balance_due_paid' === $meta->key ) {
				$meta->display_key = esc_html__( 'Balance Paid', 'shopbuilder-pro' );
			}
		}

		return $formatted_meta;
	}

	/**
	 * Hides meta-data from the admin order details view.
	 *
	 * @param array $hidden_meta_data Array of meta-data keys that should be hidden.
	 *
	 * @return array
	 */
	public function hide_admin_meta( $hidden_meta_data ) {
		$hidden_meta_data[] = '_rtsb_partial_due_date';
		$hidden_meta_data[] = '_rtsb_original_order_id';
		$hidden_meta_data[] = '_rtsb_partial_total_raw';
		$hidden_meta_data[] = '_rtsb_partial_paid_raw';

		return $hidden_meta_data;
	}

	/**
	 * Adds the "Send Remaining Balance Email" action to the order actions.
	 *
	 * @param array  $actions The order actions.
	 * @param object $order   The order object.
	 *
	 * @return array
	 */
	public function maybe_add_partial_payment_email_action( $actions, $order ) {
		$has_partial_pay = $order->get_meta( '_rtsb_partial_pay_order_product_id' ) ?? false;
		$due_paid        = $order->get_meta( '_rtsb_partial_due_paid' ) ?? false;

		if ( $has_partial_pay && ! $due_paid ) {
			$actions['rtsb_send_partial_payment_email'] = __( 'Send Remaining Balance Email', 'shopbuilder-pro' );
		}

		return $actions;
	}

	/**
	 * Applies the custom price to the cart item.
	 *
	 * @param Object $cart The cart object.
	 *
	 * @return void
	 */
	public function apply_custom_price( $cart ) {
		if ( is_admin() && ! defined( 'DOING_AJAX' ) ) {
			return;
		}

		foreach ( $cart->get_cart() as $cart_item ) {
			$product = $cart_item['data'];
			if ( isset( $cart_item['_rtsb_remaining_due'] ) ) {
				$product->set_price( floatval( $cart_item['_rtsb_remaining_due'] ) );
			}

			if ( isset( $cart_item['rtsb_partial_pay_installments'] ) ) {
				$product->set_price( floatval( $cart_item['rtsb_partial_pay_installments'][0]['amount'] ) );
			}
		}
	}

	/**
	 * Adds the "Pay Remaining Balance" action to the order actions.
	 *
	 * @param array  $actions The order actions.
	 * @param object $order   The order object.
	 *
	 * @return array
	 */
	public function add_custom_action_button( $actions, $order ) {
		$due           = floatval( $order->get_meta( '_rtsb_partial_due' ) ?? 0 );
		$due_paid      = $order->get_meta( '_rtsb_partial_due_paid' ) ?? false;
		$settings      = PartialPayFns::get_settings_data();
		$balance_label = $settings['labels']['to_pay'];

		if ( 'rtsb-partial-pay' === $order->get_status() || 'rtsb-preordered' === $order->get_status() ) {
			if ( ! $due_paid && $due > 0 ) {
				$pay_balance_action = [
					'rtsb_pay_remaining' => [
						'url'  => add_query_arg(
							[
								'rtsb_partial_pay_order' => 'true',
							],
							$order->get_checkout_payment_url()
						),
						'name' => $balance_label,
					],
				];

				$actions = $pay_balance_action + $actions;
			}

			unset( $actions['pay'] );
		}

		return $actions;
	}

	/**
	 * Changes the button text for the "Pay Remaining Balance" action.
	 *
	 * @param string $button_text The button text.
	 *
	 * @return string
	 */
	public function change_pay_button_text( $button_text ) {
		$settings      = PartialPayFns::get_settings_data();
		$balance_label = $settings['labels']['to_pay'];

		if ( $this->is_partial_payment_order_pay_page() ) {
			return $balance_label;
		}

		return $button_text;
	}

	/**
	 * Modifies the cart info for the remaining balance action.
	 *
	 * @param array  $totals The cart totals.
	 * @param object $order The order object.
	 *
	 * @return array
	 */
	public function modify_cart_info( $totals, $order ) {
		if ( $this->is_partial_payment_order_pay_page() ) {
			$total          = floatval( $order->get_meta( '_rtsb_partial_total_price' ) );
			$installments   = $order->get_meta( '_rtsb_partial_installments' );
			$paid           = $installments[0]['amount'] ?? 0;
			$total_quantity = 0;

			foreach ( $order->get_items() as $item ) {
				if ( $item instanceof WC_Order_Item_Product ) {
					$total_quantity += $item->get_quantity();
				}
			}

			$per_item_due = $total - $paid;
			$total_due    = $per_item_due * $total_quantity;

			unset( $totals['cart_subtotal'], $totals['payment_method'] );

			if ( $total_due > 0 ) {
				$totals['order_total'] = [
					'label' => __( 'Total (Remaining Balance Due)', 'shopbuilder-pro' ),
					'value' => wc_price( $total_due ) . ' (' . $total_quantity . ' <small class="times">×</small> ' . wc_price( $order->get_meta( '_rtsb_partial_due' ) ) . ')',
				];
			}
		}

		return $totals;
	}

	/**
	 * Adds the "rtsb-partial-pay" status to the order statuses.
	 *
	 * @param array  $statuses The order statuses.
	 * @param object $order   The order object.
	 *
	 * @return array
	 */
	public function allow_partial_pay_status( $statuses, $order ) {
		if ( $order && 'rtsb-partial-pay' === $order->get_status() ) {
			$statuses[] = 'rtsb-partial-pay';
		}

		if ( $order && 'rtsb-preordered' === $order->get_status() ) {
			$statuses[] = 'rtsb-preordered';
		}

		// phpcs:ignore WordPress.Security.NonceVerification.Recommended
		if ( isset( $_REQUEST['rtsb_partial_pay_order'] ) && 'true' === $_REQUEST['rtsb_partial_pay_order'] ) {
			$statuses[] = 'on-hold';
		}

		return $statuses;
	}

	/**
	 * Adjusts the order items for the remaining balance action.
	 *
	 * @param array $items The order items.
	 *
	 * @return array
	 */
	public function adjust_order_items( $items ) {
		if ( $this->is_partial_payment_order_pay_page() ) {
			foreach ( $items as $item ) {
				if ( 'line_item' === $item->get_type() ) {
					$item->set_total( 0 );
					$item->set_subtotal( 0 );
				}
			}
		}

		return $items;
	}

	/**
	 * Modifies the subtotal for the remaining balance action.
	 *
	 * @param string $subtotal The subtotal.
	 * @param object $item     The order item.
	 *
	 * @return string
	 */
	public function show_partial_paid_label( $subtotal, $item ) {
		if ( $this->is_partial_payment_order_pay_page() ) {
			$partial_paid = floatval( $item->get_meta( '_rtsb_partial_paid_raw' ) );

			if ( $partial_paid > 0 ) {
				$subtotal = wc_price( $partial_paid ) . ' <small>(' . esc_html__( 'Already paid', 'shopbuilder-pro' ) . ')</small>';
			}
		}

		return $subtotal;
	}

	/**
	 * Update order items.
	 *
	 * @param object $order The order object.
	 * @param int    $total_quantity The total quantity of the order.
	 *
	 * @return void
	 */
	private function update_order_items( $order, $total_quantity ) {
		$total = floatval( $order->get_meta( '_rtsb_partial_total_price' ) ) ?: $order->get_total();

		foreach ( $order->get_items() as $order_item ) {
			if ( $order_item instanceof WC_Order_Item_Product ) {
				$total_quantity += $order_item->get_quantity();
				$total_due       = $total * $total_quantity;

				$order_item->set_subtotal( $total_due );
				$order_item->set_total( $total_due );

				if ( '<span class="rtsb-pre-order-availability-text partial-pay">Active</span>' === $order_item->get_meta( '_rtsb_is_partial_payment' ) ) {
					$partially_paid = $order_item->get_meta( 'rtsb_partial_paid' );
					$due_paid       = $order_item->get_meta( 'rtsb_partial_due' );

					$order_item->update_meta_data( '_rtsb_is_partial_payment', '<span class="rtsb-pre-order-availability-text partial-pay">Fully Paid</span>' );
					$order_item->update_meta_data( '_rtsb_partially_paid', $partially_paid );
					$order_item->update_meta_data( '_rtsb_balance_due_paid', $due_paid );

					$order_item->delete_meta_data( 'rtsb_partial_total' );
					$order_item->delete_meta_data( 'rtsb_partial_paid' );
					$order_item->delete_meta_data( 'rtsb_partial_due' );
					$order_item->delete_meta_data( 'rtsb_partial_formatted_due_date' );
				}
			}
		}

		$order->calculate_totals();
		$order->update_meta_data( '_rtsb_partial_due_paid', true );
		$order->add_order_note(
			sprintf(
			/* Translators: 1: remaining balance, 2: total */
				__( 'Remaining balance of %1$s received. Order total adjusted to reflect full payment of %2$s.', 'shopbuilder-pro' ),
				wc_price( $order->get_meta( '_rtsb_partial_due' ) * $total_quantity ) . ' (' . $total_quantity . ' <small class="times">×</small> ' . wc_price( $order->get_meta( '_rtsb_partial_due' ) ) . ')',
				wc_price( $total * $total_quantity )
			)
		);

		$order->save();
	}

	/**
	 * Check if the current page is the order pay page with partial payment.
	 *
	 * @return bool
	 */
	private function is_partial_payment_order_pay_page() {
		// phpcs:ignore WordPress.Security.NonceVerification.Recommended
		return is_wc_endpoint_url( 'order-pay' ) && isset( $_GET['rtsb_partial_pay_order'] );
	}
}
