<?php
/**
 * PartialPay Cart Class.
 *
 * @package RadiusTheme\SBPRO
 */

namespace RadiusTheme\SBPRO\Modules\PartialPay\Frontend;

use Exception;
use RadiusTheme\SBPRO\Helpers\FnsPro;
use RadiusTheme\SBPRO\Modules\AddOns\Frontend\Product as AddOnsProduct;
use RadiusTheme\SBPRO\Modules\PreOrder\Frontend\Product;
use RadiusTheme\SBPRO\Traits\SingletonTrait;
use RadiusTheme\SBPRO\Modules\AddOns\AddOnsFns;
use RadiusTheme\SBPRO\Modules\PreOrder\PreOrderFns;
use RadiusTheme\SBPRO\Modules\PartialPay\PartialPayFns;

defined( 'ABSPATH' ) || exit();

/**
 * PartialPay Cart Class.
 */
class CartPage {
	/**
	 * Singleton Trait.
	 */
	use SingletonTrait;

	/**
	 * Prevent mixed products.
	 *
	 * @var bool
	 */
	private $prevent_mix = true;

	/**
	 * Prevent multiple products.
	 *
	 * @var bool
	 */
	private $prevent_multiple = true;

	/**
	 * Module Class Constructor.
	 */
	private function __construct() {
		/**
		 * Action.
		 */
		add_action( 'woocommerce_after_cart_item_name', [ $this, 'display_partial_pay_cart_text' ], 9 );

		/**
		 * Filters.
		 */
		add_filter( 'woocommerce_add_cart_item_data', [ $this, 'add_partial_pay_data' ], 20, 2 );
		add_filter( 'woocommerce_get_item_data', [ $this, 'display_partial_pay_data' ], 10, 2 );
		add_filter( 'woocommerce_add_cart_item', [ $this, 'update_cart_price' ], 21 );
		add_filter( 'woocommerce_get_cart_item_from_session', [ $this, 'get_cart_item_from_session' ], 21, 2 );
		add_filter( 'woocommerce_cart_item_name', [ $this, 'partial_pay_checkout_message' ], 10, 2 );
		add_filter( 'woocommerce_add_to_cart_validation', [ $this, 'validate_partial_pay_cart' ], 10, 5 );
		add_filter( 'woocommerce_before_calculate_totals', [ $this, 'validate_partial_cart_totals' ] );
		add_filter( 'woocommerce_available_payment_gateways', [ $this, 'restrict_payment_gateways' ] );
		add_filter( 'woocommerce_coupon_is_valid', [ $this, 'disable_coupons' ] );
//		add_filter( 'rtsb/module/pre_order/pre_order_price', [ $this, 'partial_pay_pre_order_price' ], 10, 2 );
	}

	/**
	 * Partial Pay cart message.
	 *
	 * @param array $cart_item Cart Item.
	 *
	 * @return void
	 * @throws Exception If there is an error in determining the partial pay status.
	 */
	public function display_partial_pay_cart_text( $cart_item ) {
		if ( ! is_cart() ) {
			return;
		}

		$cart_partial_pay_item = $cart_item['rtsb_partial_pay'] ?? false;
		$product               = $cart_item['data'];

		if ( PartialPayFns::is_partial_pay_disabled( $product ) ) {
			return;
		}

		$has_partial_pay = PartialPayFns::product_has_partial_pay( $product, $cart_item['variation_id'] ?? false );

		if ( ! $has_partial_pay || ! $cart_partial_pay_item ) {
			return;
		}

		echo '<span class="rtsb-partial-pay-text-badge">' . esc_html__( 'Partial Pay', 'shopbuilder-pro' ) . '</span>';
	}

	/**
	 * Adds add-on data to the cart item data for a given product.
	 *
	 * @param array $cart_item_data The current cart item data to which add-on data will be added.
	 * @param int   $product_id The ID of the product for which add-on data is being processed.
	 *
	 * @return array
	 */
	public function add_partial_pay_data( $cart_item_data, $product_id ) {
		// Verify nonce and request data.
		if ( ! AddOnsFns::verify_add_to_cart_request( $product_id ) ) {
			return $cart_item_data;
		}

		// phpcs:ignore WordPress.Security.NonceVerification.Recommended
		$has_partial_pay = isset( $_REQUEST['rtsb_partial_pay_option'] ) ? sanitize_text_field( wp_unslash( $_REQUEST['rtsb_partial_pay_option'] ) ) : '';

		if ( 'partial' !== $has_partial_pay ) {
			return $cart_item_data;
		}

		$rules = PartialPayFns::get_partial_pay_rules( $product_id );

		if ( is_array( $rules ) && count( $rules ) > 0 ) {
			// phpcs:ignore WordPress.Security.NonceVerification.Recommended
			$variation_id   = isset( $_REQUEST['variation_id'] ) ? absint( $_REQUEST['variation_id'] ) : 0;
			$partial_amount = PartialPayFns::get_partial_amount( $product_id );
			$product        = $variation_id ? wc_get_product( $variation_id ) : wc_get_product( $product_id );
			$full_price     = FnsPro::get_module_compatible_price( $product );

			if ( ! PartialPayFns::is_valid_partial_payment( $partial_amount, $full_price ) ) {
				return $cart_item_data;
			}

			$cart_item_data['rtsb_partial_pay']              = true;
			$cart_item_data['rtsb_partial_pay_installments'] = $partial_amount;
		}

		return $cart_item_data;
	}

	/**
	 * Display partial pay data in the WooCommerce cart.
	 *
	 * @param array $show_data The current array of cart item data to display.
	 * @param array $cart_item The cart item array containing product data.
	 *
	 * @return array
	 */
	public function display_partial_pay_data( $show_data, $cart_item ) {
		$has_partial_pay = $cart_item['rtsb_partial_pay'] ?? false;

		if ( $has_partial_pay ) {
			$product_id   = $cart_item['product_id'];
			$variation_id = $cart_item['variation_id'] ?? 0;
			$rules        = PartialPayFns::get_partial_pay_rules( $cart_item['product_id'] );

			if ( is_array( $rules ) && count( $rules ) > 0 ) {
				$partial_amount = PartialPayFns::get_partial_amount( $product_id, $variation_id );
				$product        = $variation_id ? wc_get_product( $variation_id ) : wc_get_product( $product_id );
				$full_price     = FnsPro::get_module_compatible_price( $product );

				if ( ! PartialPayFns::is_valid_partial_payment( $partial_amount, $full_price ) ) {
					return $show_data;
				}

				$labels = PartialPayFns::get_settings_data()['labels'];

				$show_data[] = [
					'name'  => esc_html__( 'Product Price', 'shopbuilder-pro' ),
					'value' => wc_price( $full_price ),
				];

				if ( count( $partial_amount ) > 1 ) {
					foreach ( $partial_amount as $inst ) {
						$show_data[] = [
							'name'  => $inst['title'],
							'value' => wc_price( $inst['amount'] ),
						];
					}
				} else {
					$show_data[] = [
						'name'  => $labels['single_label'],
						'value' => wc_price( $partial_amount[0]['amount'] ),
					];
					$show_data[] = [
						'name'  => $labels['remaining_label_single'],
						'value' => wc_price( floatval( $full_price ) - floatval( $partial_amount[0]['amount'] ) ) ,
					];
				}
			}
		}

		return $show_data;
	}

	/**
	 * Update the product price in the cart item based on partial pay.
	 *
	 * @param array $cart_item The cart item data.
	 *
	 * @return array
	 */
	public function update_cart_price( $cart_item ) {
		$product = $cart_item['data'];

		if ( ! PartialPayFns::product_has_partial_pay( $product, 'variation' === $product->get_type() ) ) {
			return $cart_item;
		}

		$price         = floatval( $product->get_price() );
		$regular_price = floatval( $product->get_regular_price() );
		$sale_price    = floatval( $product->get_sale_price() );
		$prices        = compact( 'price', 'regular_price', 'sale_price' );

		return $this->update_product_price( $cart_item, $prices );
	}

	/**
	 * Get the cart item data from the session and update it with partial pay data and prices.
	 *
	 * @param array $cart_item The current cart item data.
	 * @param array $values The session data for the cart item.
	 *
	 * @return array
	 */
	public function get_cart_item_from_session( $cart_item, $values ) {
		$product = $cart_item['data'];

		if ( ! PartialPayFns::product_has_partial_pay( $product, 'variation' === $product->get_type() ) ) {
			return $cart_item;
		}

		$price         = floatval( $product->get_price() );
		$regular_price = floatval( $product->get_regular_price() );
		$sale_price    = floatval( $product->get_sale_price() );
		$prices        = compact( 'price', 'regular_price', 'sale_price' );

		if ( isset( $values['rtsb_partial_pay'] ) ) {
			$cart_item['rtsb_partial_pay'] = $values['rtsb_partial_pay'];
		}

		return $this->update_product_price( $cart_item, $prices );
	}

	/**
	 * Partial pay checkout message.
	 *
	 * @param string $name Name.
	 * @param array  $cart_item Cart Item.
	 *
	 * @return string
	 */
	public function partial_pay_checkout_message( $name, $cart_item ) {
		if ( ! is_checkout() ) {
			return $name;
		}

		$product               = $cart_item['data'];
		$text                  = '<span class="rtsb-checkout-text"><span class="rtsb-partial-pay-text-badge">' . esc_html__( 'Partial Pay', 'shopbuilder-pro' ) . '</span></span>';
		$has_partial_pay       = PartialPayFns::product_has_partial_pay( $product, $cart_item['variation_id'] ?? false );
		$cart_partial_pay_item = $cart_item['rtsb_partial_pay'] ?? false;

		if ( ! $has_partial_pay || ! $cart_partial_pay_item ) {
			return $name;
		}

		return $name . $text;
	}

	/**
	 * Perform cart validation for partial pay products.
	 *
	 * @param bool  $passed_validation Whether the validation has passed.
	 * @param int   $product_id        The ID of the product being added to the cart.
	 * @param int   $quantity          The quantity of the product being added to the cart.
	 * @param int   $variation_id      The ID of the product variation being added to the cart.
	 * @param array $variation         The variation attributes of the product being added to the cart.
	 *
	 * @return bool
	 * @throws Exception If there is an error in determining the pre-order status.
	 */
	public function validate_partial_pay_cart( $passed_validation, $product_id, $quantity, $variation_id = '', $variation = [] ) {
		$id      = $variation_id ?: $product_id;
		$product = wc_get_product( $id );

		if ( ! $product ) {
			return $passed_validation;
		}

		if ( $this->cart_contains_mixed_products() ) {
			PartialPayFns::partial_pay_product_notice();

			return true;
		}

		if ( $this->cart_contains_multiple_partial_products() ) {
			PartialPayFns::partial_pay_product_notice( 'cart_multi' );

			return true;
		}

		return $passed_validation;
	}

	/**
	 * Perform cart calculations for partial pay products.
	 *
	 * @param object $cart The cart object.
	 *
	 * @return void
	 */
	public function validate_partial_cart_totals( $cart ) {
		if ( is_admin() && ! defined( 'DOING_AJAX' ) ) {
			return;
		}

		if ( $this->cart_contains_mixed_products() ) {
			foreach ( $cart->get_cart() as $cart_item_key => $cart_item ) {
				$product = $cart_item['data'];

				if ( ! PartialPayFns::product_has_partial_pay( $product, 'variation' === $product->get_type() ) ) {
					$cart->remove_cart_item( $cart_item_key );
					PartialPayFns::partial_pay_product_notice();

					break;
				}
			}
		}

		if ( $this->cart_contains_multiple_partial_products() ) {
			$partial_pay_found = false;

			foreach ( $cart->get_cart() as $cart_item_key => $cart_item ) {
				$product = $cart_item['data'];

				if ( PartialPayFns::product_has_partial_pay( $product, 'variation' === $product->get_type() ) ) {
					if ( $partial_pay_found ) {
						$cart->remove_cart_item( $cart_item_key );
						PartialPayFns::partial_pay_product_notice( 'cart_multi' );
					} else {
						$partial_pay_found = true;
					}
				}
			}
		}
	}

	/**
	 * Restrict payment gateways.
	 *
	 * @param array $gateways Gateways.
	 *
	 * @return array
	 */
	public function restrict_payment_gateways( $gateways ) {
		if ( is_admin() || ! is_checkout() ) {
			return $gateways;
		}

		// phpcs:ignore WordPress.Security.NonceVerification.Recommended
		if ( ! empty( $_REQUEST['pay_for_order'] ) && ! empty( $_REQUEST['rtsb_partial_pay_order'] ) ) {
			return $gateways;
		}

		if ( ! $this->is_partial_pay_cart() ) {
			return $gateways;
		}

		$settings     = PartialPayFns::get_settings_data();
		$restrictions = $settings['restrict_payment_method'];

		$decoded             = array_map(
			function ( $item ) {
				return json_decode( stripslashes( $item ), true );
			},
			$restrictions
		);
		$restricted_gateways = array_column( $decoded, 'value' );

		foreach ( $gateways as $gateway_id => $gateway ) {
			if ( in_array( $gateway_id, $restricted_gateways, true ) ) {
				unset( $gateways[ $gateway_id ] );
			}
		}

		return $gateways;
	}

	/**
	 * Disable coupons.
	 *
	 * @param bool $valid Valid.
	 *
	 * @return bool
	 * @throws Exception Throws Exception for invalid coupons.
	 */
	public function disable_coupons( $valid ) {
		if ( ! WC()->cart ) {
			return $valid;
		}

		$settings = PartialPayFns::get_settings_data();
		$disable  = $settings['disable_coupons'];

		if ( 'on' !== $disable ) {
			return $valid;
		}

		foreach ( WC()->cart->get_cart() as $cart_item ) {
			$product = $cart_item['data'];

			$partial_meta = $product->get_meta( '_rtsb_product_partial_pay', true );

			if ( ! empty( $partial_meta ) ) {
				throw new Exception( esc_html__( 'Sorry, coupons cannot be used with Partial Pay products.', 'shopbuilder-pro' ) );
			}
		}

		return $valid;
	}

	/**
	 * Partial pay pre-order price.
	 *
	 * @param string $price Price.
	 * @param object $product Product Object.
	 *
	 * @return string
	 */
	public static function partial_pay_pre_order_price( $price, $product ) {
		if ( ! PartialPayFns::product_has_partial_pay( $product, 'variation' === $product->get_type() ) ) {
			return $price;
		}

		$product_id   = $product->get_id();
		$variation_id = 'variation' === $product->get_type() ? $product->get_id() : null;
		$rules        = PartialPayFns::get_partial_pay_rules( $product_id );

		if ( is_array( $rules ) && count( $rules ) > 0 ) {
			$partial_amount = PartialPayFns::get_partial_amount( $product_id, $variation_id );
			$product        = $variation_id ? wc_get_product( $variation_id ) : wc_get_product( $product_id );
			$full_price     = $product->get_price();

			if ( ! PartialPayFns::is_valid_partial_payment( $partial_amount, $full_price ) ) {
				return $price;
			}

			if ( count( $partial_amount ) > 1 ) {
				foreach ( $partial_amount as $inst ) {
					$price = $inst['amount'];
				}
			} else {
				$price = $partial_amount[0]['amount'];
			}
		}

		return $price;
	}

	/**
	 * Update the product price in the cart item data based on the given prices.
	 *
	 * @param array $cart_item The cart item data.
	 * @param array $prices The prices (regular, sale, and current) to update.
	 *
	 * @return array
	 */
	public function update_product_price( $cart_item, $prices ) {
		$result_data = $this->price_calculate( $cart_item, $prices );

		$cart_item['data']->set_price( $result_data['price'] );

		return $cart_item;
	}

	/**
	 * Calculate the total price of the product in the cart item based on the given prices.
	 *
	 * @param array $cart_data The cart item data.
	 * @param array $prices The base prices (regular, sale, and current).
	 *
	 * @return array
	 */
	public function price_calculate( $cart_data, $prices ) {
		$product_id     = $cart_data['product_id'];
		$variation_id   = 'variation' === $cart_data['data']->get_type() ? $cart_data['variation_id'] : null;
		$partial_amount = PartialPayFns::get_partial_amount( $product_id, $variation_id );
		$partial_price  = $partial_amount[0]['amount'] ?? 0;

		$prices['price']                   = $partial_price;
		$prices['rtsb_partial_pay_amount'] = $partial_amount;

		/**
		 * Pre-Order compatibility.
		 */
		if ( FnsPro::is_module_active( 'pre_order' ) && PreOrderFns::is_on_pre_order( wc_get_product( $product_id ) ) ) {
			$prices = AddOnsFns::apply_preorder_price( $prices, $cart_data['data'] );
		}

		return $prices;
	}

	/**
	 * Check the contents of the cart for mixed and multiple partial pay products.
	 *
	 * @return bool[]
	 */
	private function analyze_cart_content() {
		$has_partial   = false;
		$has_regular   = false;
		$partial_count = 0;

		foreach ( WC()->cart->get_cart() as $cart_item ) {
			$product = $cart_item['data'];

			$has_partial_pay = PartialPayFns::product_has_partial_pay( $product, $cart_item['variation_id'] ?? false );

			if ( $has_partial_pay ) {
				$has_partial = true;
				$partial_count++;
			} else {
				$has_regular = true;
			}
		}

		return [
			'cart_contains_mixed'    => $has_partial && $has_regular,
			'cart_contains_multiple' => $partial_count > 1,
		];
	}

	/**
	 * Check if the cart contains mixed products (partial pay and regular).
	 *
	 * @return bool
	 */
	private function cart_contains_mixed_products() {
		return $this->prevent_mix && $this->analyze_cart_content()['cart_contains_mixed'];
	}

	/**
	 * Check if the cart contains multiple partial pay products.
	 *
	 * @return bool
	 */
	private function cart_contains_multiple_partial_products() {
		return $this->prevent_multiple && $this->analyze_cart_content()['cart_contains_multiple'];
	}

	/**
	 * Check if the cart contains partial pay products.
	 *
	 * @return bool
	 */
	private function is_partial_pay_cart() {
		if ( ! WC()->cart ) {
			return false;
		}

		foreach ( WC()->cart->get_cart() as $cart_item ) {
			$product_id   = $cart_item['product_id'];
			$variation_id = $cart_item['variation_id'] ?? false;
			$product      = wc_get_product( $variation_id ?: $product_id );

			$has_partial_pay = PartialPayFns::product_has_partial_pay( $product, $variation_id );

			if ( $has_partial_pay ) {
				return true;
			}
		}

		return false;
	}
}
