<?php
/**
 * Product Add-Ons Module Custom Fields Class.
 *
 * @package RadiusTheme\SB
 */

namespace RadiusTheme\SBPRO\Modules\PartialPay\Admin;

use RadiusTheme\SB\Helpers\Fns;
use RadiusTheme\SBPRO\Traits\SingletonTrait;
use RadiusTheme\SBPRO\Helpers\AdminInputsFns;

defined( 'ABSPATH' ) || exit();

/**
 * Product Add-Ons Module Admin Class.
 */
class CustomFields {
	/**
	 * Singleton Trait.
	 */
	use SingletonTrait;

	/**
	 * Class Constructor.
	 */
	private function __construct() {
		$this->add_custom_data_tab();
		$this->add_tab_options();
		$this->save_tab_options();
	}

	/**
	 * Adds a custom tab to the WooCommerce product data tabs.
	 *
	 * @return void
	 */
	private function add_custom_data_tab() {
		add_filter( 'woocommerce_product_data_tabs', [ $this, 'add_partial_pay_tab' ] );
	}


	/**
	 * Adds meta-boxes to the WooCommerce custom data tab.
	 *
	 * @return void
	 */
	public function add_tab_options() {
		add_action( 'woocommerce_product_data_panels', [ $this, 'render_meta_boxes' ] );
	}

	/**
	 * Saves the Product Add-Ons data when the product is saved.
	 *
	 * @return void
	 */
	public function save_tab_options() {
		add_action( 'woocommerce_process_product_meta', [ $this, 'save_partial_pay_meta' ] );
	}

	/**
	 * Adds the Product Add-Ons tab to the product data tabs.
	 *
	 * @param array $tabs Existing tabs.
	 *
	 * @return array Modified tabs.
	 */
	public function add_partial_pay_tab( $tabs ) {
		$tabs['rtsb_product_partial_pay'] = [
			'label'    => esc_html__( 'Partial Pay', 'shopbuilder-pro' ),
			'target'   => 'rtsb_product_partial_pay',
			'class'    => [ 'show_if_simple', 'show_if_variable' ],
			'priority' => 62,
		];

		return $tabs;
	}

	/**
	 * Renders meta-boxes to the WooCommerce product data panels.
	 *
	 * @return void
	 */
	public function render_meta_boxes() {
		global $post;

		$disable_partial_pay = get_post_meta( $post->ID, '_rtsb_disable_partial_pay', true );
		$enable_partial_pay  = get_post_meta( $post->ID, '_rtsb_enable_partial_pay', true );
		$partial_pay         = get_post_meta( $post->ID, '_rtsb_product_partial_pay', true );
		$disable_checked     = ! empty( $disable_partial_pay ) && 'on' === $disable_partial_pay ? 'on' : 'off';
		$disable_visibility  = 'on' === $disable_checked ? ' hidden' : '';
		$enable_checked      = ! empty( $enable_partial_pay ) && 'on' === $enable_partial_pay ? 'on' : 'off';
		$enable_visibility   = 'on' === $enable_checked ? '' : ' hidden';
		?>
		<div id='rtsb_product_partial_pay' class='panel woocommerce_options_panel rtsb_panel hidden'>
			<div class="rtsb-group-heading">
				<h2><?php esc_html_e( 'Partial Pay Settings', 'shopbuilder-pro' ); ?></h2>
			</div>

			<div id="rtsb_product_partial_pay_options_group" class="options_group">
				<?php wp_nonce_field( rtsb()->nonceText, rtsb()->nonceId ); ?>
				<div class="partial-pay-group-control">
					<div id="rtsb_disable_partial_pay">
						<?php
						Fns::print_html(
							AdminInputsFns::generate_switch_input(
								[
									'id'            => '_rtsb_disable_partial_pay',
									'label'         => esc_html__( 'Disable Global Rules?', 'shopbuilder-pro' ),
									'desc_tip'      => true,
									'aria_label'    => esc_html__( 'Disable Global partial pay', 'shopbuilder-pro' ),
									'description'   => esc_html__( 'Switch on to disable the ShopBuilder partial pay rules for this product.', 'shopbuilder-pro' ),
									'checked'       => esc_attr( $disable_checked ),
									'value'         => 'yes',
									'input_class'   => 'switch-input',
									'wrapper_class' => 'form-field disable-rules',
								]
							),
							true
						);
						?>
					</div>
					<div id="rtsb_enable_partial_pay">
						<?php
						Fns::print_html(
							AdminInputsFns::generate_switch_input(
								[
									'id'            => '_rtsb_enable_partial_pay',
									'label'         => esc_html__( 'Override Global Rules?', 'shopbuilder-pro' ),
									'desc_tip'      => true,
									'aria_label'    => esc_html__( 'Override Global Partial Pay', 'shopbuilder-pro' ),
									'description'   => esc_html__( 'Switch on to override the ShopBuilder global partial pay rules.', 'shopbuilder-pro' ),
									'checked'       => esc_attr( $enable_checked ),
									'value'         => 'yes',
									'input_class'   => 'switch-input',
									'wrapper_class' => 'form-field' . esc_attr( $disable_visibility ),
								]
							),
							true
						);
						?>
					</div>
				</div>
				<div id="rtsb_partial_pay_groups" class="rtsb-partial-pay<?php echo esc_attr( $enable_visibility ); ?>">
					<?php
					$fields = AdminInputsFns::generate_select_input(
						[
							'id'            => '_rtsb_product_partial_pay[amount_type]',
							'label'         => esc_html__( 'Amount Type', 'shopbuilder-pro' ),
							'desc_tip'      => true,
							'aria_label'    => esc_html__( 'Amount type', 'shopbuilder-pro' ),
							'description'   => esc_html__( 'Choose how you want to received the partial payment.', 'shopbuilder-pro' ),
							'value'         => ! empty( $partial_pay['amount_type'] ) ? $partial_pay['amount_type'] : 'fixed',
							'input_class'   => 'short',
							'label_class'   => 'field-label',
							'options'       => [
								'percent' => esc_html__( 'Percentage', 'shopbuilder-pro' ),
								'fixed'   => esc_html__( 'Fixed', 'shopbuilder-pro' ),
							],
							'wrapper_class' => 'form-field amount-type',
						]
					);

//					$fields .= AdminInputsFns::generate_text_input(
//						[
//							'id'            => '_rtsb_product_partial_pay[partial_amount]',
//							'label'         => esc_html__( 'Partial pay amount', 'shopbuilder-pro' ),
//							'desc_tip'      => true,
//							'aria_label'    => esc_html__( 'Partial pay amount', 'shopbuilder-pro' ),
//							'description'   => esc_html__( 'Enter the partial payment amount based on the amount type you chose above. You can separate multiple amounts with a comma. Multiple amounts will serve as multiple installments.', 'shopbuilder-pro' ),
//							'value'         => ! empty( $partial_pay['partial_amount'] ) ? $partial_pay['partial_amount'] : '',
//							'input_class'   => 'short',
//							'label_class'   => 'field-label',
//							'wrapper_class' => 'form-field amount',
//						]
//					);

					$fields .= AdminInputsFns::generate_number_input(
						[
							'id'            => '_rtsb_product_partial_pay[partial_amount]',
							'label'         => esc_html__( 'Partial pay amount', 'shopbuilder-pro' ),
							'desc_tip'      => true,
							'aria_label'    => esc_html__( 'Partial pay amount', 'shopbuilder-pro' ),
							'description'   => esc_html__( 'Enter the partial payment amount based on the amount type you chose above. ', 'shopbuilder-pro' ),
							'value'         => ! empty( $partial_pay['partial_amount'] ) ? $partial_pay['partial_amount'] : '',
							'input_class'   => 'short',
							'label_class'   => 'field-label',
							'wrapper_class' => 'form-field amount',
						]
					);

					$fields .= AdminInputsFns::generate_number_input(
						[
							'id'            => '_rtsb_product_partial_pay[due_date]',
							'label'         => esc_html__( 'Due Date After Payment (Days)', 'shopbuilder-pro' ),
							'desc_tip'      => true,
							'aria_label'    => esc_html__( 'Balance payment due after the order date, in days', 'shopbuilder-pro' ),
							'description'   => esc_html__( 'Set the number of days after the initial payment when the remaining balance is due.', 'shopbuilder-pro' ),
							'value'         => ! empty( $partial_pay['due_date'] ) ? $partial_pay['due_date'] : '',
							'input_class'   => 'short',
							'label_class'   => 'field-label',
							'wrapper_class' => 'form-field amount',
						]
					);

					$fields .= AdminInputsFns::generate_number_input(
						[
							'id'            => '_rtsb_product_partial_pay[send_mail]',
							'label'         => esc_html__( 'Send Reminder Before Due (Days)', 'shopbuilder-pro' ),
							'desc_tip'      => true,
							'aria_label'    => esc_html__( 'Send email after the order date, in days', 'shopbuilder-pro' ),
							'description'   => esc_html__( 'Specify how many days before the due date the customer should receive a reminder email.', 'shopbuilder-pro' ),
							'value'         => ! empty( $partial_pay['send_mail'] ) ? $partial_pay['send_mail'] : '',
							'input_class'   => 'short',
							'label_class'   => 'field-label',
							'wrapper_class' => 'form-field amount',
						]
					);

					Fns::print_html( $fields, true );
					?>
				</div>
			</div>
		</div>
		<?php
	}

	/**
	 * Saves the product add-ons data from the admin form.
	 *
	 * @param int $post_id The ID of the product post.
	 *
	 * @return void
	 */
	public static function save_partial_pay_meta( $post_id ) {
		// Check if post_id is valid.
		if ( empty( $post_id ) && ! is_numeric( $post_id ) ) {
			return;
		}

		// Verify nonce.
		check_admin_referer( rtsb()->nonceText, rtsb()->nonceId );

		$disable_partial_pay = isset( $_REQUEST['_rtsb_disable_partial_pay'] ) ? sanitize_text_field( wp_unslash( $_REQUEST['_rtsb_disable_partial_pay'] ) ) : 'off';
		$enable_partial_pay  = isset( $_REQUEST['_rtsb_enable_partial_pay'] ) ? sanitize_text_field( wp_unslash( $_REQUEST['_rtsb_enable_partial_pay'] ) ) : 'off';
		$disable_partial_pay = 'yes' === $disable_partial_pay ? 'on' : $disable_partial_pay;
		$enable_partial_pay  = 'yes' === $enable_partial_pay ? 'on' : $enable_partial_pay;
		$partial_pay         = isset( $_REQUEST['_rtsb_product_partial_pay'] ) ? map_deep( wp_unslash( $_REQUEST['_rtsb_product_partial_pay'] ), 'sanitize_text_field' ) : [];

		if ( ! empty( $disable_partial_pay ) && 'on' === $disable_partial_pay ) {
			update_post_meta( $post_id, '_rtsb_disable_partial_pay', $disable_partial_pay );
		} else {
			delete_post_meta( $post_id, '_rtsb_disable_partial_pay' );
		}

		if ( ! empty( $enable_partial_pay ) && 'on' === $enable_partial_pay ) {
			update_post_meta( $post_id, '_rtsb_enable_partial_pay', $enable_partial_pay );
		} else {
			delete_post_meta( $post_id, '_rtsb_enable_partial_pay' );
		}

		update_post_meta( $post_id, '_rtsb_product_partial_pay', $partial_pay );
	}
}
