<?php
/**
 * Customize My Account Module Class.
 *
 * @package RadiusTheme\SB
 */

namespace RadiusTheme\SBPRO\Modules\GiftCard;

use RadiusTheme\SB\Helpers\Fns;
use RadiusTheme\SBPRO\Helpers\FnsPro;
use RadiusTheme\SBPRO\Traits\SingletonTrait;
use WC_Product_Simple;

defined( 'ABSPATH' ) || exit();

/**
 * Back-order Module Class.
 */
class GiftCardUserAccount {
	/**
	 * Singleton Trait.
	 */
	use SingletonTrait;

	/**
	 * Module Class Constructor.
	 */
	private function __construct() {
		// Register new endpoint to use for My Account page.
		$this->add_endpoint();
		// Add new query var.
		// add_filter( 'query_vars', [ $this, 'gift_cards_query_vars' ] );
		 add_filter( 'woocommerce_get_query_vars', [ $this, 'gift_cards_query_vars' ], 0 );
		// Insert the new endpoint into the My Account menu.
		add_filter( 'woocommerce_account_menu_items', [ $this, 'add_gift_cards_link_my_account' ] );
		// Add content to the new endpoint.
		add_action( 'rtsb/view/card/information', [ $this, 'view_gift_card_information' ] );
		add_action( 'rtsb/view/card/activity', [ $this, 'view_gift_card_activity' ], 11 );
		// Gift Card Endpoint.
		add_action( 'woocommerce_account_rtsb-gift-cards_endpoint', [ $this, 'gift_cards_content' ] );
		add_action( 'woocommerce_account_rtsb-view-card_endpoint', [ $this, 'view_gift_cards_content' ] );
	}
	/**
	 * Register new endpoint to use for My Account page.
	 */
	public function add_endpoint() {
		add_rewrite_endpoint( 'rtsb-gift-cards', EP_ROOT | EP_PAGES );
		add_rewrite_endpoint( 'rtsb-view-card', EP_ROOT | EP_PAGES );
	}
	/**
	 * Add new query var.
	 *
	 * @param $vars
	 *
	 * @return array
	 */
	public function gift_cards_query_vars( $vars ) {
		$vars['rtsb-gift-cards'] = 'rtsb-gift-cards';
		$vars['rtsb-view-card']  = 'rtsb-view-card';
		return $vars;
	}
	/**
	 * Insert the new endpoint into the My Account menu.
	 *
	 * @param $items
	 *
	 * @return array
	 */
	public function add_gift_cards_link_my_account( $items ) {

		$item_position = ( array_search( 'orders', array_keys( $items ), true ) );
		$items_part1   = array_slice( $items, 0, $item_position + 1 );
		$items_part2   = array_slice( $items, $item_position );

		/**
		 * APPLY_FILTERS: yith_wcgc_my_account_menu_item_title
		 *
		 * Filter the "Gift Cards" menu item title on "My Account".
		 *
		 * @param string the "Gift Cards" menu item title
		 *
		 * @return string
		 */
		$items_part1['rtsb-gift-cards'] = esc_html_x( 'Gift Cards', 'my account endpoint title', 'shopbuilder-pro' );
		$items                          = array_merge( $items_part1, $items_part2 );

		return $items;
	}
	/**
	 *  Add content to the new endpoint.
	 */
	public function gift_cards_content() {
		$current_endpoint = WC()->query->get_current_endpoint();
		$paged            = absint( get_query_var( $current_endpoint ) ) ?: 1;
		$per_page         = apply_filters( 'rtsb/gift/cards/per/page', 5 );
		$args             = [
			'post_type'      => GiftCardFns::GIFT_CARD_POST_TYPE,
			'post_status'    => 'publish',
			'meta_key'       => 'purchase_by',
			'meta_value'     => get_current_user_id(),
			'orderby'        => 'ID',
			'order'          => 'DESC',
			'posts_per_page' => $per_page,
			'paged'          => $paged,
		];
		$query            = new \WP_Query( $args );
		// Pagination.
		$total_pages = $query->max_num_pages;
		$data        = apply_filters(
			'rtsb/view/gift/card/list',
			[
				'gift_cards'  => $query->posts,
				'paged'       => $paged,
				'total_pages' => $total_pages,
			],
			$query
		);
		Fns::load_template( 'gift-card/gift-card-list', $data, false, '', rtsbpro()->get_plugin_template_path() );

		wp_reset_postdata();
	}

	/**
	 * Add new query var.
	 *
	 * @param $vars
	 *
	 * @return array
	 */
	public function view_gift_cards_content( $vars ) {
		$cardId   = absint( get_query_var( 'rtsb-view-card' ) ) ?: 0;
		$giftCard = GiftCardFns::get_card_by_id( $cardId );
		if ( $giftCard->exists() ) {
			$this->view_gift_card_information( $giftCard );
			$this->view_gift_card_activity( $giftCard );
		} else {
			FnsPro::custom_notice( esc_html__( 'Invalid Gift Card.', 'shopbuilder-pro' ) );
		}
	}
	/**
	 * Add new query var.
	 *
	 * @param $giftCard
	 *
	 * @return array
	 */
	public function view_gift_card_information( $giftCard ) {
		$expiration_date = $giftCard->get_card_data( 'expiration_date' );
		$format          = get_option( 'date_format' ) . ' ' . get_option( 'time_format' );
		$expiration_date = ! empty( $expiration_date ) ? date_i18n( $format, strtotime( $expiration_date ) ) : esc_html__( 'Lifetime', 'shopbuilder-pro' );
		$data            = apply_filters(
			'rtsb/view/gift/card/information',
			[
				'order_id'          => $giftCard->get_card_data( 'order_id' ),
				'card_number'       => $giftCard->get_card_data( 'card_number' ),
				'virtual'           => $giftCard->get_card_data( 'virtual' ),
				'expiration_date'   => $expiration_date,
				'product_name'      => $giftCard->get_card_data( 'product_name' ),
				'remaining_balance' => wc_price( $giftCard->get_card_data( 'remaining_balance' ) ),
				'card_amount'       => wc_price( $giftCard->get_card_data( 'card_amount' ) ),
				'email'             => $giftCard->get_card_data( 'recipient_email' ),
				'sender_name'       => $giftCard->get_card_data( 'sender_name' ),
				'recipient_name'    => $giftCard->get_card_data( 'recipient_name' ),
				'card_message'      => $giftCard->get_card_data( 'card_message' ),
			],
			$giftCard
		);
		Fns::load_template( 'gift-card/gift-card-info', $data, false, '', rtsbpro()->get_plugin_template_path() );
	}

	/**
	 * Add new query var.
	 *
	 * @param $giftCard
	 *
	 * @return array
	 */
	public function view_gift_card_activity( $giftCard ) {
		$data = apply_filters(
			'rtsb/view/gift/card/activity',
			[
				'activity' => $giftCard->get_card_activity(),
			],
			$giftCard
		);
		Fns::load_template( 'gift-card/gift-card-activity', $data, false, '', rtsbpro()->get_plugin_template_path() );
	}
}
