<?php
/**
 * Customize My Account Module Class.
 *
 * @package RadiusTheme\SB
 */

namespace RadiusTheme\SBPRO\Modules\GiftCard;

use RadiusTheme\SB\Helpers\Fns;
use RadiusTheme\SBPRO\Helpers\AdminInputsFns;
use RadiusTheme\SBPRO\Helpers\FnsPro;
use RadiusTheme\SBPRO\Traits\SingletonTrait;

defined( 'ABSPATH' ) || exit();

/**
 * Gift Card Admin Module Class.
 */
class GiftCardCPT {
	/**
	 * Singleton Trait.
	 */
	use SingletonTrait;

	/**
	 * Module Class Constructor.
	 *
	 * Initializes the Gift Card product type by adding necessary filters and actions.
	 */
	private function __construct() {
		$this->register_gift_card_post_type();
		add_action( 'woocommerce_checkout_order_processed', [ $this, 'create_gift_cards_for_old_checkout' ], 11, 3 );
		add_action( 'woocommerce_store_api_checkout_order_processed', [ $this, 'create_gift_cards_for_block_checkout' ], 11 );
		$options = GiftCardFns::get_options();
		$status  = $options['publish_card'] ?? [];
		if ( is_array( $status ) && in_array( 'processing', $status, true ) ) {
			add_action( 'woocommerce_order_status_processing', [ $this, 'gift_card_publish' ] );
		}
		if ( is_array( $status ) && in_array( 'completed', $status, true ) ) {
			add_action( 'woocommerce_order_status_completed', [ $this, 'gift_card_publish' ] );
		}
		add_filter( 'manage_' . GiftCardFns::GIFT_CARD_POST_TYPE . '_posts_columns', [ $this, 'customize_column' ] );
		add_action( 'manage_' . GiftCardFns::GIFT_CARD_POST_TYPE . '_posts_custom_column', [ $this, 'custom_column_content' ], 10, 2 );
		add_action( 'add_meta_boxes', [ $this, 'add_custom_meta_box' ] );
		add_action( 'save_post', [ $this, 'save_meta_fields' ] );
		add_filter( 'parent_file', [ $this, 'highlight_wc_menu_for_gift_cards' ] );
	}
	/**
	 * Adds a custom meta box to the post edit screen.
	 */
	public function add_custom_meta_box() {
		add_meta_box(
			'rtsb_gift_card_meta_box',
			esc_html__( 'Gift Card Details', 'shopbuilder-pro' ),
			[ $this, 'render_meta_box' ],
			GiftCardFns::GIFT_CARD_POST_TYPE,
			'normal',
			'high'
		);
		add_meta_box(
			'rtsb_gift_card_send_email',
			esc_html__( 'Gift Card Info', 'shopbuilder-pro' ),
			[ $this, 'send_card_email_meta' ],
			GiftCardFns::GIFT_CARD_POST_TYPE,
			'side',
			'high'
		);
	}

	/**
	 * Highlights the WooCommerce menu item in the admin when viewing the custom post type.
	 *
	 * Ensures that the WooCommerce menu stays selected when managing the 'gift_card' custom post type.
	 *
	 * @param string $parent_file The current parent file slug.
	 * @return string The updated parent file slug.
	 */
	public function highlight_wc_menu_for_gift_cards( $parent_file ) {
		global $current_screen;
		if ( isset( $current_screen->post_type ) && GiftCardFns::GIFT_CARD_POST_TYPE === $current_screen->post_type ) {
			return 'woocommerce';
		}
		return $parent_file;
	}

	/**
	 * Saves custom meta field values when the post is saved.
	 *
	 * @param int $post_id The ID of the post being saved.
	 */
	public function save_meta_fields( $post_id ) {
		// Use existing nonce action.
		$nonce_action = 'update-post_' . $post_id;
		// Verify nonce.
		if ( ! isset( $_POST['_wpnonce'] ) || ! wp_verify_nonce( sanitize_text_field( wp_unslash( $_POST['_wpnonce'] ) ), $nonce_action ) ) {
			return;
		}
		// Prevent autosave overwrite.
		if ( defined( 'DOING_AUTOSAVE' ) && DOING_AUTOSAVE ) {
			return;
		}
		// Check user permissions.
		if ( ! current_user_can( 'edit_post', $post_id ) ) {
			return;
		}

		$virtual         = isset( $_POST['rtsb_gift_card_as_virtual'] ) ? sanitize_text_field( wp_unslash( $_POST['rtsb_gift_card_as_virtual'] ) ) : 'no';
		$recipient_email = isset( $_POST['rtsb_gift_card_recipient_email'] ) ? sanitize_text_field( wp_unslash( $_POST['rtsb_gift_card_recipient_email'] ) ) : '';
		$sender_name     = isset( $_POST['rtsb_gift_card_sender_name'] ) ? sanitize_text_field( wp_unslash( $_POST['rtsb_gift_card_sender_name'] ) ) : '';
		$recipient_name  = isset( $_POST['rtsb_gift_card_recipient_name'] ) ? sanitize_text_field( wp_unslash( $_POST['rtsb_gift_card_recipient_name'] ) ) : '';
		$card_message    = isset( $_POST['rtsb_gift_card_message'] ) ? sanitize_text_field( wp_unslash( $_POST['rtsb_gift_card_message'] ) ) : '';
		$card_note       = isset( $_POST['rtsb_gift_card_note'] ) ? sanitize_text_field( wp_unslash( $_POST['rtsb_gift_card_note'] ) ) : '';
		// rtsb_gift_card_note.
		$expire_date       = isset( $_POST['expiration_date'] ) ? sanitize_text_field( wp_unslash( $_POST['expiration_date'] ) ) : '';
		$card_amount       = isset( $_POST['rtsb_gift_card_amount'] ) ? wc_format_decimal( wp_unslash( $_POST['rtsb_gift_card_amount'] ) ) : 0;
		$remaining_balance = isset( $_POST['rtsb_gift_card_remaining_balance'] ) ? wc_format_decimal( wp_unslash( $_POST['rtsb_gift_card_remaining_balance'] ) ) : '';

		if ( null === $remaining_balance || '' === $remaining_balance ) {
			$remaining_balance = $card_amount;
		} else {
			$remaining_balance = $remaining_balance > $card_amount ? $card_amount : $remaining_balance;
		}

		update_post_meta( $post_id, 'rtsb_gift_card_as_virtual', $virtual );
		update_post_meta( $post_id, 'rtsb_gift_card_recipient_email', $recipient_email );
		update_post_meta( $post_id, 'rtsb_gift_card_sender_name', $sender_name );
		update_post_meta( $post_id, 'rtsb_gift_card_recipient_name', $recipient_name );
		update_post_meta( $post_id, 'rtsb_gift_card_message', $card_message );
		update_post_meta( $post_id, 'rtsb_gift_card_note', $card_note );

		update_post_meta( $post_id, 'expiration_date', $expire_date );
		update_post_meta( $post_id, 'rtsb_gift_card_amount', $card_amount );
		update_post_meta( $post_id, 'rtsb_gift_card_remaining_balance', $remaining_balance );
	}
	/**
	 * Renders the custom meta box with input fields.
	 *
	 * @param \WP_Post $post The current post object.
	 */
	public function render_meta_box( $post ) {
		$giftCard = GiftCardFns::get_card_by_id( $post->ID );
		if ( ! $giftCard->exists() ) {
			return;
		}
		// Retrieve meta values.
		wp_enqueue_style( 'gift-card-metabox' );
		wp_enqueue_style( 'rtsb-product-admin' );
		wp_enqueue_script( 'rtsb-gift-card-admin' );
		$order_id          = $giftCard->get_card_data( 'order_id' );
		$expiration_date   = $giftCard->get_card_data( 'expiration_date' );
		$remaining_balance = $giftCard->get_card_data( 'remaining_balance' );
		$card_amount       = $giftCard->get_card_data( 'card_amount' );

		$virtual        = $giftCard->get_card_data( 'virtual' );
		$email          = $giftCard->get_card_data( 'recipient_email' );
		$sender_name    = $giftCard->get_card_data( 'sender_name' );
		$recipient_name = $giftCard->get_card_data( 'recipient_name' );
		$card_message   = $giftCard->get_card_data( 'card_message' );
		$card_note      = $giftCard->get_card_data( 'card_note' );
		$product_name   = $giftCard->get_card_data( 'product_name' );
		// Order ID field.
		?>
		<div class="rtsb-gift-card-meta-wrapper">
			<div class="rtsb-meta-left">
				<?php
				Fns::print_html(
					AdminInputsFns::generate_switch_input(
						[
							'id'            => 'rtsb_gift_card_as_virtual',
							'label'         => esc_html__( 'Virtual?', 'shopbuilder-pro' ),
							'desc_tip'      => true,
							'aria_label'    => esc_html__( 'Virtual Card', 'shopbuilder-pro' ),
							'description'   => esc_html__( 'Switch on to make virtual card.', 'shopbuilder-pro' ),
							'checked'       => esc_attr( $virtual ),
							'value'         => 'on',
							'input_class'   => 'switch-input',
							'label_class'   => 'field-label',
							'wrapper_class' => 'form-field ',
						]
					),
					true
				);
				?>
				<span class="description" ><?php esc_html_e( 'Switch on to make virtual card.', 'shopbuilder-pro' ); ?></span>

				<?php
					Fns::print_html(
						AdminInputsFns::generate_date_picker_input(
							$post->ID,
							[
								'id'            => 'expiration_date',
								'label'         => esc_html__( 'Expiration Date', 'shopbuilder-pro' ),
								'desc_tip'      => true,
								'value'         => $expiration_date,
								'class'         => 'rtsb-input-field',
								'description'   => esc_html__( 'Set the expiration date for this gift card.', 'shopbuilder-pro' ),
								'label_class'   => 'field-label',
								'wrapper_class' => 'form-field',
							],
							true,
							false
						),
						true
					);
				?>
				<span class="description"><?php esc_html_e( 'Set the expiration date for this gift card.', 'shopbuilder-pro' ); ?></span>
				<?php
				Fns::print_html(
					AdminInputsFns::generate_number_input(
						[
							'id'                => 'rtsb_gift_card_amount',
							'label'             => esc_html__( 'Card Amount', 'shopbuilder-pro' ),
							'desc_tip'          => false,
							'aria_label'        => esc_html__( 'Card Amount', 'shopbuilder-pro' ),
							'description'       => esc_html__( 'Total Gift amount loaded onto the gift card.', 'shopbuilder-pro' ),
							'value'             => $card_amount,
							'min'               => '0',
							'step'              => '0.01',
							'input_class'       => 'rtsb-input-field',
							'label_class'       => 'field-label',
							'wrapper_class'     => 'form-row input-price',
							'custom_attributes' => [
								'readonly' => ! empty( $order_id ),
							],
						]
					),
					true
				);
				?>
				<span class="description" ><?php esc_html_e( 'Total Gift amount loaded onto the gift card.', 'shopbuilder-pro' ); ?></span>
				<br/>
				<?php
				if ( '' !== $remaining_balance ) {
					Fns::print_html(
						AdminInputsFns::generate_number_input(
							[
								'id'                => 'rtsb_gift_card_remaining_balance',
								'label'             => esc_html__( 'Available Balance', 'shopbuilder-pro' ),
								'desc_tip'          => true,
								'aria_label'        => esc_html__( 'Available Balance', 'shopbuilder-pro' ),
								'description'       => esc_html__( 'Current balance left on this gift card. Remaining Balance will not be greater than the Card Amount.', 'shopbuilder-pro' ),
								'value'             => $remaining_balance,
								'min'               => 0,
								'step'              => 1,
								'input_class'       => 'rtsb-input-field',
								'label_class'       => 'field-label',
								'wrapper_class'     => 'form-row input-price',
								'custom_attributes' => [
									'readonly' => ! empty( $order_id ),
								],
							]
						),
						true
					);
					?>
					<span class="description"><?php esc_html_e( 'Current balance left on this gift card. Remaining Balance will not be greater than the Card Amount.', 'shopbuilder-pro' ); ?></span>
					<?php
				}

				Fns::print_html(
					AdminInputsFns::generate_text_input(
						[
							'id'            => 'rtsb_gift_card_recipient_email',
							'label'         => esc_html__( 'Recipient\'s email', 'shopbuilder-pro' ),
							'desc_tip'      => true,
							'aria_label'    => esc_html__( 'Recipient\'s email', 'shopbuilder-pro' ),
							'placeholder'   => esc_html__( 'Email', 'shopbuilder-pro' ),
							'value'         => $email,
							'input_class'   => 'rtsb-input-field',
							'label_class'   => 'field-label',
							'wrapper_class' => 'form-row rtsb-virtual-fields',
						]
					),
					true
				);
		?>
				<br/>
				<?php
				Fns::print_html(
					AdminInputsFns::generate_text_input(
						[
							'id'            => 'rtsb_gift_card_sender_name',
							'label'         => esc_html__( 'Sender Name', 'shopbuilder-pro' ),
							'desc_tip'      => true,
							'aria_label'    => esc_html__( 'Sender Name', 'shopbuilder-pro' ),
							'placeholder'   => esc_html__( 'Sender Name', 'shopbuilder-pro' ),
							'value'         => $sender_name,
							'input_class'   => 'rtsb-input-field',
							'label_class'   => 'field-label',
							'wrapper_class' => 'form-row rtsb-virtual-fields',
						]
					),
					true
				);
				?>
				<br/>
				<?php
				Fns::print_html(
					AdminInputsFns::generate_text_input(
						[
							'id'            => 'rtsb_gift_card_recipient_name',
							'label'         => esc_html__( 'Recipient Name', 'shopbuilder-pro' ),
							'desc_tip'      => true,
							'aria_label'    => esc_html__( 'Recipient Name', 'shopbuilder-pro' ),
							'placeholder'   => esc_html__( 'Recipient Name', 'shopbuilder-pro' ),
							'value'         => $recipient_name,
							'input_class'   => 'rtsb-input-field',
							'label_class'   => 'field-label',
							'wrapper_class' => 'form-row rtsb-virtual-fields',
						]
					),
					true
				);
				?>
				<br/>
				<?php
				Fns::print_html(
					AdminInputsFns::generate_textarea_input(
						[
							'id'            => 'rtsb_gift_card_message',
							'label'         => esc_html__( 'Card Message', 'shopbuilder-pro' ),
							'desc_tip'      => true,
							'aria_label'    => esc_html__( 'Card Message', 'shopbuilder-pro' ),
							'description'   => esc_html__( 'Card Message.', 'shopbuilder-pro' ),
							'placeholder'   => esc_html__( 'Message.', 'shopbuilder-pro' ),
							'value'         => $card_message,
							'input_class'   => 'rtsb-input-field',
							'label_class'   => 'field-label',
							'wrapper_class' => 'form-field input-description  rtsb-virtual-fields',
						]
					),
					true
				);
				?>
				<?php
				Fns::print_html(
					AdminInputsFns::generate_textarea_input(
						[
							'id'            => 'rtsb_gift_card_note',
							'label'         => esc_html__( 'Internal note', 'shopbuilder-pro' ),
							'desc_tip'      => true,
							'aria_label'    => esc_html__( 'Internal note', 'shopbuilder-pro' ),
							'description'   => esc_html__( 'Internal note', 'shopbuilder-pro' ),
							'placeholder'   => esc_html__( 'Private Note', 'shopbuilder-pro' ),
							'value'         => $card_note,
							'input_class'   => 'rtsb-input-field',
							'label_class'   => 'field-label',
							'wrapper_class' => 'form-field input-description ',
						]
					),
					true
				);
				?>
				<span class="description" ><?php esc_html_e( 'Internal note – Visible only to admins.', 'shopbuilder-pro' ); ?></span>
			</div>

			<?php if ( ! empty( $post->ID ) ) { ?>
				<div class="rtsb-gift-card-design rtsb-virtual-fields " style=" margin-top: 10px;">
					<h3 style="font-size: 30px;margin-left: auto; margin-right: auto; margin-bottom: 20px; margin-top: 0;"> Card Preview </h3>
					<?php
					$format          = get_option( 'date_format' ) . ' ' . get_option( 'time_format' );
					$expiration_date = ! empty( $expiration_date ) ? date_i18n( $format, strtotime( $expiration_date ) ) : esc_html__( 'Lifetime', 'shopbuilder-pro' );
					$data            = [
						'card_number'       => get_the_title( $post->ID ),
						'product_name'      => $product_name,
						'recipient_email'   => $email,
						'card_amount'       => wp_strip_all_tags( wc_price( $card_amount ) ),
						'sender_name'       => $sender_name,
						'recipient_name'    => $recipient_name,
						'card_message'      => $card_message,
						'remaining_balance' => $remaining_balance,
						'expiration_date'   => $expiration_date,
						'redeem_url'        => get_permalink( wc_get_page_id( 'shop' ) ),
					];
					Fns::load_template( 'gift-card/gift-card-template-for-email', $data, false, '', rtsbpro()->get_plugin_template_path() );
					?>
				</div>
			<?php } ?>

		</div>
		<?php
		$data = [
			'activity' => $giftCard->get_card_activity(),
		];
		Fns::load_template( 'gift-card/gift-card-activity', $data, false, '', rtsbpro()->get_plugin_template_path() );
	}

	/**
	 * Renders the custom meta box with input fields.
	 *
	 * @param \WP_Post $post The current post object.
	 */
	public function send_card_email_meta( $post ) {
		$giftCard = GiftCardFns::get_card_by_id( $post->ID );
		if ( ! $giftCard->exists() ) {
			return;
		}
		$order_id = $giftCard->get_card_data( 'order_id' );
		if ( $order_id ) {
			?>
			<p>
				<label class="field-label" ><?php esc_html_e( 'Order ID', 'shopbuilder-pro' ); ?>:  </label>
				<?php Fns::print_html( GiftCardFns::get_order_edit_link( $order_id ) ); ?>
			</p>
			<?php
		}
		if ( 'on' !== $giftCard->get_card_data( 'virtual' ) ) {
			echo esc_html__( 'Physical Card', 'shopbuilder-pro' );
			return;
		}
		$email = $giftCard->get_card_data( 'recipient_email' );
		if ( ! empty( $email ) ) {
			$email .= '<br/><a class="rtsb-send-gift-card-email" href="#" data-send-email="' . $post->ID . '">' . esc_html__( 'Send Gift Card', 'shopbuilder-pro' ) . '</a>';
		} else {
			$email = '--';
		}

		Fns::print_html( $email );
	}
	/**
	 * Adds a custom column to the post type list table.
	 *
	 * @param array $columns Existing columns.
	 * @return array Modified columns.
	 */
	public function customize_column( $columns ) {
		$new_columns = [];
		if ( isset( $columns['title'] ) ) {
			$columns['title'] = esc_html__( 'Card Number', 'shopbuilder-pro' );
		}
		foreach ( $columns as $key => $value ) {
			$new_columns[ $key ] = $value;
			if ( 'title' === $key ) {
				$new_columns['total_balance']     = esc_html__( 'Total Balance', 'shopbuilder-pro' );
				$new_columns['remaining_balance'] = esc_html__( 'Available Balance', 'shopbuilder-pro' );
				$new_columns['recipients_email']  = esc_html__( 'Recipients email', 'shopbuilder-pro' );
				$new_columns['order_id']          = esc_html__( 'Source Order', 'shopbuilder-pro' );
			}
		}
		$new_columns['expiration_date'] = esc_html__( 'Expiration Date', 'shopbuilder-pro' );
		return $new_columns;
	}

	/**
	 * Outputs the content for the custom column.
	 *
	 * @param string $column Column name.
	 * @param int    $post_id Post ID.
	 */
	public function custom_column_content( $column, $post_id ) {
		$giftCard = GiftCardFns::get_card_by_id( $post_id );
		if ( ! $giftCard->exists() ) {
			return;
		}
		if ( 'order_id' === $column ) {
			Fns::print_html( $this->order_link( $post_id ) );
		} elseif ( 'remaining_balance' === $column ) {
			Fns::print_html( wc_price( $giftCard->get_card_data( 'remaining_balance' ) ) );
		} elseif ( 'total_balance' === $column ) {
			Fns::print_html( wc_price( $giftCard->get_card_data( 'card_amount' ) ) );
		} elseif ( 'recipients_email' === $column ) {
			if ( 'on' !== $giftCard->get_card_data( 'virtual' ) ) {
				echo esc_html__( 'Physical Card', 'shopbuilder-pro' );
				return;
			}
			$email = $giftCard->get_card_data( 'recipient_email' );
			if ( ! empty( $email ) ) {
				$email .= '<br/><a class="rtsb-send-gift-card-email" href="#" data-send-email="' . $post_id . '">' . esc_html__( 'Send Gift Card', 'shopbuilder-pro' ) . '</a>';
			} else {
				$email = '--';
			}

			Fns::print_html( $email );
		} elseif ( 'expiration_date' === $column ) {
			$date = $giftCard->get_card_data( 'expiration_date' );
			if ( ! empty( $date ) ) {
				$format   = get_option( 'date_format' ) . ' ' . get_option( 'time_format' );
				$dateTime = date_i18n( $format, strtotime( $date ) );
			} else {
				$dateTime = esc_html__( 'Lifetime', 'shopbuilder-pro' );
			}
			Fns::print_html( $dateTime );
		}
	}

	/**
	 * @param $post_id
	 * @return string
	 */
	private function order_link( $post_id ) {
		$giftCard = GiftCardFns::get_card_by_id( $post_id );
		$order_id = $giftCard->get_card_data( 'order_id' );
		if ( ! empty( $order_id ) ) {
			return GiftCardFns::get_order_edit_link( $order_id );
		} else {
			return esc_html__( 'Created manually', 'shopbuilder-pro' );
		}
	}
	/**
	 * Registers the "Gift Card" post type under the WooCommerce menu.
	 *
	 * - No archive or single page (admin use only).
	 * - Supports title, editor, custom fields, and revisions.
	 * - Accessible only from the WooCommerce menu.
	 */
	public function register_gift_card_post_type() {
		$labels = [
			'name'                  => _x( 'Gift Cards', 'Post type general name', 'shopbuilder-pro' ),
			'singular_name'         => _x( 'Gift Card', 'Post type singular name', 'shopbuilder-pro' ),
			'menu_name'             => _x( 'Gift Cards', 'Admin Menu text', 'shopbuilder-pro' ),
			'add_new'               => __( 'Add New Gift Card', 'shopbuilder-pro' ),
			'add_new_item'          => __( 'Add New Gift Card', 'shopbuilder-pro' ),
			'new_item'              => __( 'New Gift Card', 'shopbuilder-pro' ),
			'edit_item'             => __( 'Edit Gift Card', 'shopbuilder-pro' ),
			'view_item'             => __( 'View Gift Card', 'shopbuilder-pro' ),
			'all_items'             => __( 'All Gift Cards', 'shopbuilder-pro' ),
			'search_items'          => __( 'Search Gift Cards', 'shopbuilder-pro' ),
			'not_found'             => __( 'No gift cards found.', 'shopbuilder-pro' ),
			'not_found_in_trash'    => __( 'No gift cards found in Trash.', 'shopbuilder-pro' ),
			'featured_image'        => _x( 'Gift Card Image', 'Overrides the “Featured Image” phrase for this post type. Added in 4.3', 'shopbuilder-pro' ),
			'set_featured_image'    => _x( 'Set Gift Card Image', 'Overrides the “Set featured image” phrase for this post type. Added in 4.3', 'shopbuilder-pro' ),
			'remove_featured_image' => _x( 'Remove Gift Card Image', 'Overrides the “Remove featured image” phrase for this post type. Added in 4.3', 'shopbuilder-pro' ),
			'use_featured_image'    => _x( 'Use as Gift Card Image', 'Overrides the “Use as featured image” phrase for this post type. Added in 4.3', 'shopbuilder-pro' ),
		];
		$args   = [
			'labels'              => $labels,
			'public'              => false,
			'show_ui'             => true,
			'show_in_menu'        => false,
			'capability_type'     => 'post',
			'supports'            => [ 'title' ],
			'menu_icon'           => 'dashicons-gift',
			'has_archive'         => false,
			'publicly_queryable'  => false,
			'exclude_from_search' => true,
			'show_in_nav_menus'   => false,
		];
		register_post_type( GiftCardFns::GIFT_CARD_POST_TYPE, $args );
	}
	/**
	 * Update gift card post status by meta key and value (order_id).
	 *
	 * @param int    $order_id The WooCommerce order ID.
	 * @param string $new_status The new post status (e.g., 'publish', 'draft', etc.).
	 */
	private function update_gift_card_status_by_order_id( $order_id, $new_status, $old_status = 'draft' ) {
		// Get the gift card posts using meta query by order ID.
		$gift_cards = get_posts(
			[
				'post_type'      => GiftCardFns::GIFT_CARD_POST_TYPE,
				'posts_per_page' => -1, // Retrieve all posts.
				'post_status'    => $old_status, // Get posts with draft status.
				'meta_key'       => 'order_id', // Meta key to query by.
				'meta_value'     => $order_id,  // Order ID meta value.
				'fields'         => 'ids', // Only return post IDs for performance.
			]
		);

		$current = FnsPro::currentTimestamp();
		WC()->mailer();
		// Loop through the posts and update their status.
		if ( ! empty( $gift_cards ) ) {
			foreach ( $gift_cards as $post_id ) {
				wp_update_post(
					[
						'ID'          => $post_id,
						'post_status' => $new_status, // Change the post status.
					]
				);
				$giftCard   = GiftCardFns::get_card_by_id( $post_id );
				$product_id = (int) $giftCard->get_card_data( 'product_id' );
				$validity   = (int) get_post_meta( $product_id, '_rtsb_gift_card_validity', true );
				if ( ! metadata_exists( 'post', $post_id, 'expiration_date' ) && ! empty( $validity ) ) {
					$expire      = $current + ( $validity * DAY_IN_SECONDS ); // Expiration timestamp.
					$expire_date = gmdate( 'Y-m-d\TH:i', $expire ); // Format as Y-m-d.
					update_post_meta( $post_id, 'expiration_date', $expire_date );
				}
				do_action( 'rtsb/gift/card/after/' . $new_status, $post_id, $order_id, $old_status );
			}
		}
	}
	/**
	 * Generates a unique gift card number.
	 *
	 * @return string The generated gift card number.
	 */
	public function gift_card_publish( $order_id ) {
		$this->update_gift_card_status_by_order_id( $order_id, 'publish' );
	}
	/**
	 * Creates gift card posts when an order is created.
	 *
	 * @param int $order_id The WooCommerce order ID.
	 */
	private function create_gift_cards( $order ) {
		if ( ! $order ) {
			return;
		}
		// Prevent duplicate gift card creation.
		if ( $order->get_meta( '_rtsb_gift_cards_created' ) ) {
			return;
		}

		// Get the Customer billing email.
		$billing_email = $order->get_billing_email();
		foreach ( $order->get_items() as $item ) {
			$product = $item->get_product();
			// Check if the product is a gift card.
			if ( $product && $product->get_type() === GiftCardFns::GIFT_CARD_KEY ) {
				$quantity        = $item->get_quantity();
				$amount          = $item->get_subtotal() / $quantity;
				$recipient       = $item->get_meta( 'rtsb_gift_card_recipient_email', true );
				$virtualCardinfo = [
					'rtsb_gift_card_recipient_email' => $recipient ?: $billing_email,
					'rtsb_gift_card_sender_name'     => $item->get_meta( 'rtsb_gift_card_sender_name', true ),
					'rtsb_gift_card_recipient_name'  => $item->get_meta( 'rtsb_gift_card_recipient_name', true ),
					'rtsb_gift_card_message'         => $item->get_meta( 'rtsb_gift_card_message', true ),
				];
				for ( $i = 0; $i < $quantity; $i++ ) {
					$cardId   = $this->create_gift_card_post( $order, $product, $amount );
					$giftCard = GiftCardFns::get_card_by_id( $cardId );
					$giftCard->update_card_info( $virtualCardinfo );
				}
			}
		}
		// Mark order as processed to avoid duplicate gift card creation.
		$order->update_meta_data( '_rtsb_gift_cards_created', true );
		$order->save();
	}
	/**
	 * Creates gift card posts when an order is created.
	 *
	 * @param int $order_id The WooCommerce order ID.
	 */
	public function create_gift_cards_for_old_checkout( $order_id, $posted_data, $order ) {
		$this->create_gift_cards( $order );
	}
	/**
	 * Creates gift card posts when an order is created.
	 *
	 * @param int $order_id The WooCommerce order ID.
	 */
	public function create_gift_cards_for_block_checkout( $order ) {
		$this->create_gift_cards( $order );
	}
	/**
	 * Creates a single gift card post.
	 *
	 * @param object $order   The order.
	 * @param object $product The product.
	 * @param float  $amount     The gift card amount.
	 */
	private function create_gift_card_post( $order, $product, $amount ) {
		$order_id      = $order->get_id();
		$card_number   = GiftCardFns::generate_card_number();
		$virtual       = $product && $product->is_virtual() ? 'on' : '';
		$customer_user = $order->get_customer_id();
		if ( empty( $customer_user ) ) {
			$customer_user = $order->get_billing_email();
		}
		return wp_insert_post(
			[
				'post_title'  => $card_number,
				'post_type'   => GiftCardFns::GIFT_CARD_POST_TYPE,
				'post_status' => 'draft',
				'post_parent' => $product->get_id(),
				'meta_input'  => [
					'order_id'                         => $order_id,
					'product_id'                       => $product->get_id(),
					'validity'                         => null,
					'rtsb_gift_card_remaining_balance' => $amount,
					'rtsb_gift_card_amount'            => $amount,
					'rtsb_gift_card_as_virtual'        => $virtual,
					'purchase_by'                      => $customer_user,
				],
			]
		);
	}
}
