<?php
/**
 * Customize My Account Module Class.
 *
 * @package RadiusTheme\SB
 */

namespace RadiusTheme\SBPRO\Modules\GiftCard;

use RadiusTheme\SB\Helpers\Fns;
use RadiusTheme\SBPRO\Traits\SingletonTrait;
use WC_Order;
use WP_Query;

defined( 'ABSPATH' ) || exit();

/**
 * Gift Card Admin Module Class.
 */
class GiftCardAdmin {
	/**
	 * Singleton Trait.
	 */
	use SingletonTrait;

	/**
	 * Module Class Constructor.
	 *
	 * Initializes the Gift Card product type by adding necessary filters and actions.
	 */
	private function __construct() {
		add_action( 'admin_menu', [ $this, 'add_gift_card_submenu' ] );
		// Save the Gift Card product type when a product is saved.
		add_action( 'woocommerce_process_product_meta', [ $this, 'save_gift_card_product_type' ] );
		// Customize WooCommerce product data tabs for Gift Card products.
		add_filter( 'woocommerce_product_data_tabs', [ $this, 'woocommerce_product_data_tabs' ], 1 );
		/**
		 * Add an option to let the admin set the gift card as a physical good or digital goods
		 */
		add_filter( 'product_type_options', [ $this, 'add_type_option' ] );

		// Custom.
		add_action( 'woocommerce_product_data_panels', [ $this, 'show_gift_card_product_settings' ] );
		/**
		 * Show gift cards code and amount in order's totals section, in edit order page
		 */
		add_action( 'woocommerce_admin_order_totals_after_tax', [ $this, 'show_gift_cards_total_before_order_totals' ] );
		// Save Type.
		add_action( 'save_post', [ $this, 'save_gift_card' ], 1, 2 );
		// Change Status.
		add_action( 'woocommerce_order_status_changed', [ $this, 'handle_refunded_amount' ], 10, 4 );
		add_filter( 'views_edit-rtsb_gift_card', [ $this, 'gift_card_view' ] );
		add_action( 'pre_get_posts', [ $this, 'filter_posts_by_my_view' ] );
		add_action( 'admin_enqueue_scripts', [ $this, 'enqueue_gift_card_admin_js' ] );
	}
	/**
	 * Adds "Gift Cards" as a submenu under WooCommerce.
	 */
	public function enqueue_gift_card_admin_js( $hook ) {
		$screen = get_current_screen();

		if ( ! $screen ) {
			return;
		}

		// Replace 'gift_card' with your actual post type
		if ( 'edit-rtsb_gift_card' === $screen->id || 'rtsb_gift_card' === $screen->post_type ) {
			$version = ( defined( 'WP_DEBUG' ) && WP_DEBUG ) ? time() : RTSBPRO_VERSION;
			if ( in_array( 'sitepress-multilingual-cms/sitepress.php', get_option( 'active_plugins' ) ) ) {
				$ajaxurl = admin_url( 'admin-ajax.php?lang=' . ICL_LANGUAGE_CODE );
			} else {
				$ajaxurl = admin_url( 'admin-ajax.php' );
			}
			$options = GiftCardFns::get_options();
			$pattern = $options['card_number_pattern'] ?? '';
			wp_register_script( 'rtsb-gift-card-admin', rtsbpro()->get_assets_uri( 'js/backend/gift-card.js' ), [ 'jquery' ], $version, true );
			// Gift Card Admin Params.
			wp_localize_script(
				'rtsb-gift-card-admin',
				'rtsbCardAdminParam',
				[
					'ajaxUrl'            => esc_url( $ajaxurl ),
					'confirmMessage'     => esc_html__( 'Are you sure you want to send the gift card email?', 'shopbuilder-pro' ),
					'afterSendMessage'   => esc_html__( '✅ Card sent successfully.', 'shopbuilder-pro' ),
					'generateButtonText' => esc_html__( 'Generate Card Number', 'shopbuilder-pro' ),
					'cardNumberPattern'  => $pattern,
					rtsb()->nonceId      => wp_create_nonce( rtsb()->nonceText ),
				]
			);
			wp_enqueue_script( 'rtsb-gift-card-admin' );

		}
	}
	/**
	 * Adds "Gift Cards" as a submenu under WooCommerce.
	 */
	public function filter_posts_by_my_view( $query ) {
		if ( ! ( is_admin() && $query->is_main_query() && in_array( $query->get( 'post_type' ), [ GiftCardFns::GIFT_CARD_POST_TYPE ] ) ) ) {
			return;
		}
		$view = ( $_GET['card_view'] ?? '' );
		if ( 'redeemed' === $view ) {
			$query->set(
				'meta_query',
				[
					[
						'key'     => 'rtsb_gift_card_remaining_balance',
						'value'   => 0,
						'compare' => '<=',
						'type'    => 'NUMERIC',
					],
				]
			);
		} elseif ( 'expired' === $view ) {
			$query->set(
				'meta_query',
				[
					'relation' => 'AND',
					[
						'key'     => 'expiration_date', // change to your meta key.
						'value'   => current_time( 'Y-m-d' ), // WP site time.
						'compare' => '<',
						'type'    => 'DATE',
					],
					[
						'key'     => 'rtsb_gift_card_remaining_balance',
						'value'   => 0,
						'compare' => '>',
						'type'    => 'NUMERIC',
					],
				]
			);
		} elseif ( 'active' === $view ) {
			$query->set( 'post_status', 'publish' );
			$query->set(
				'meta_query',
				[
					'relation' => 'AND',
					[
						'key'     => 'rtsb_gift_card_remaining_balance',
						'value'   => '0',
						'compare' => '>',
						'type'    => 'NUMERIC',
					],
					[
						'relation' => 'OR',
						[
							'key'     => 'expiration_date',
							'value'   => current_time( 'Y-m-d' ),
							'compare' => '>',
							'type'    => 'DATE',
						],
						[
							'key'     => 'expiration_date',
							'compare' => 'in',
							'value'   => [ null, '' ],
						],
						[
							'key'     => 'expiration_date',
							'compare' => 'NOT EXISTS',
						],
					],
				]
			);
		}
	}
	/**
	 * Adds "Gift Cards" as a submenu under WooCommerce.
	 */
	public function add_gift_card_submenu() {
		add_submenu_page(
			'woocommerce',
			__( 'Gift Cards', 'shopbuilder-pro' ),
			__( 'Gift Cards', 'shopbuilder-pro' ),
			'manage_woocommerce',
			'edit.php?post_type=rtsb_gift_card'
		);
	}
	/**
	 * Adds "Gift Cards" as a submenu under WooCommerce.
	 */
	public function gift_card_view( $views ) {
		// $trash = $views['trash'] ?? '';
		$all = $views['all'] ?? '';
		unset(
			$views['publish'],
			$views['all']
		);
		$view = ( $_GET['card_view'] ?? '' );
		// Count for Redeemed (balance = 0).
		$redeemed_query = new WP_Query(
			[
				'post_type'      => GiftCardFns::GIFT_CARD_POST_TYPE,
				'post_status'    => 'any',
				'posts_per_page' => -1,
				'fields'         => 'ids',
				'meta_query'     => [
					[
						'key'     => 'rtsb_gift_card_remaining_balance',
						'value'   => 0,
						'compare' => '<=',
						'type'    => 'NUMERIC',
					],
				],
			]
		);
		$redeemed_count = $redeemed_query->found_posts;
		// Count for Expired (expiration_date < today).
		$expired_query = new WP_Query(
			[
				'post_type'      => GiftCardFns::GIFT_CARD_POST_TYPE,
				'post_status'    => 'any',
				'posts_per_page' => -1,
				'fields'         => 'ids',
				'meta_query'     => [
					[
						'relation' => 'AND',
						[
							'key'     => 'expiration_date', // change to your meta key.
							'value'   => current_time( 'Y-m-d' ), // WP site time.
							'compare' => '<',
							'type'    => 'DATE',
						],
						[
							'key'     => 'rtsb_gift_card_remaining_balance',
							'value'   => 0,
							'compare' => '>',
							'type'    => 'NUMERIC',
						],
					],
				],
			]
		);
		$expired_count = $expired_query->found_posts;

		$active_query       = new WP_Query(
			[
				'post_type'      => GiftCardFns::GIFT_CARD_POST_TYPE,
				'post_status'    => 'publish',
				'posts_per_page' => -1,
				'fields'         => 'ids',
				'meta_query'     => [
					'relation' => 'AND',
					[
						'key'     => 'rtsb_gift_card_remaining_balance',
						'value'   => '0',
						'compare' => '>',
						'type'    => 'NUMERIC',
					],
					[
						'relation' => 'OR',
						[
							'key'     => 'expiration_date',
							'value'   => current_time( 'Y-m-d' ),
							'compare' => '>',
							'type'    => 'DATE',
						],
						[
							'key'     => 'expiration_date',
							'compare' => 'in',
							'value'   => [ null, '' ],
						],
						[
							'key'     => 'expiration_date',
							'compare' => 'NOT EXISTS',
						],
					],
				],
			]
		);
		$active_count       = $active_query->found_posts;
		$custom             = [];
		$used_url           = admin_url( 'edit.php?post_type=rtsb_gift_card&card_view=redeemed' );
		$expired_url        = admin_url( 'edit.php?post_type=rtsb_gift_card&card_view=expired' );
		$active_url         = admin_url( 'edit.php?post_type=rtsb_gift_card&card_view=active' );
		$custom['all']      = $all;
		$custom['active']   = '<a href="' . esc_url( $active_url ) . '"' . ( 'active' === $view ? 'class="current"' : null ) . '>' . esc_html__( 'Active', 'shopbuilder-pro' ) . '(' . $active_count . ')</a>';
		$custom['redeemed'] = '<a href="' . esc_url( $used_url ) . '"' . ( 'redeemed' === $view ? 'class="current"' : null ) . '>' . esc_html__( 'Redeemed', 'shopbuilder-pro' ) . '(' . $redeemed_count . ')</a>';
		$custom['expired']  = '<a href="' . esc_url( $expired_url ) . '" ' . ( 'expired' === $view ? 'class="current"' : null ) . ' >' . esc_html__( 'Expired', 'shopbuilder-pro' ) . '(' . $expired_count . ')</a>';
		return $custom + $views;
	}
	/**
	 * Handles gift card refund logic when an order status changes to refunded, cancelled, or failed.
	 *
	 * - Restores the used gift card amount back to the card balance if not already refunded.
	 * - Marks the order to avoid multiple refunds using the `_rtsb_is_gift_card_amount_refunded` meta.
	 * - If the order is refunded, resets the status of gift cards associated with the order to `pending`.
	 *
	 * @param int            $order_id    The ID of the order being updated.
	 * @param string         $from_status The previous order status.
	 * @param string         $to_status   The new order status.
	 * @param false|WC_Order $order     Optional. The WC_Order object. Defaults to false and fetched if not provided.
	 */
	public function handle_refunded_amount( $order_id, $from_status, $to_status, $order = false ) {
		$order = wc_get_order( $order_id );
		// Your logic here, e.g. notify admin or log data.
		$is_gift_card_amount_refunded = $order->get_meta( '_rtsb_is_gift_card_amount_refunded' );
		if ( ( 'cancelled' === $to_status || ( 'refunded' === $to_status ) || ( 'failed' === $to_status ) ) && 'yes' !== $is_gift_card_amount_refunded ) {
			$gift_card_applied = $order->get_meta( 'rtsb_applied_gift_cards' );
			if ( ! empty( $gift_card_applied ) ) {
				foreach ( $gift_card_applied as $gift_card_code => $gift_card_value ) {
					$giftCard = GiftCardFns::get_card_by_code( $gift_card_code );
					if ( ! $giftCard->exists() ) {
						continue;
					}
					 $new_amount = $giftCard->card_balance() + $gift_card_value;
					if ( $new_amount > $giftCard->card_total_amount() ) {
						$new_amount = $giftCard->card_total_amount();
					}
					$giftCard->update_balance( $new_amount );
					$giftCard->add_card_activity( $order_id, $gift_card_value, $new_amount );
				}
				$order->update_meta_data( '_rtsb_is_gift_card_amount_refunded', 'yes' );
				$order->save_meta_data();
			}
		}
		if ( 'refunded' === $to_status ) {
			$post_ids = get_posts(
				[
					'post_type'   => GiftCardFns::GIFT_CARD_POST_TYPE,
					'post_status' => 'any',
					'meta_key'    => 'order_id',
					'meta_value'  => $order_id,
					'numberposts' => -1,
					'fields'      => 'ids',
				]
			);
			if ( ! empty( $post_ids ) ) {
				foreach ( $post_ids as $post_id ) {
					wp_update_post(
						[
							'ID'          => $post_id,
							'post_status' => 'pending',
						]
					);
				}
			}
		}
	}

	/**
	 * Save gift card amount when a product is saved
	 *
	 * @param int    $post_id
	 * @param object $post
	 *
	 * @return mixed|void
	 */
	public function save_gift_card( $post_id, $post ) {

		$product = wc_get_product( $post_id );

		if ( ! is_object( $product ) ) {
			return;
		}

		if ( ! isset( $_POST['product-type'] ) || ( GiftCardFns::GIFT_CARD_KEY !== $_POST['product-type'] ) ) {//phpcs:ignore WordPress.Security.NonceVerification
			return;
		}

		// verify this is not an auto save routine.
		if ( defined( 'DOING_AUTOSAVE' ) && DOING_AUTOSAVE ) {
			return;
		}
		if ( empty( $_REQUEST['woocommerce_meta_nonce'] ) ) {
			return;
		}
		// Check the nonce.
		$nonce = sanitize_text_field( wp_unslash( $_REQUEST['woocommerce_meta_nonce'] ) );
		if ( ! wp_verify_nonce( $nonce, 'woocommerce_save_data' ) ) { // phpcs:ignore WordPress.Security.ValidatedSanitizedInput.InputNotSanitized
			return;
		}

		$validity = sanitize_text_field( wp_unslash( $_REQUEST['rtsb_gift_card_validity'] ?? 0 ) );

		update_post_meta( $post_id, '_rtsb_gift_card_validity', $validity );

		$amounts = isset( $_REQUEST['rtsb_gift_card_amounts'] ) ? array_map( 'floatval', wp_unslash( $_REQUEST['rtsb_gift_card_amounts'] ) ) : [];
		$amounts = array_unique( array_filter( $amounts ) );
		update_post_meta( $post_id, GiftCardFns::GIFT_CARD_AMOUNTS, $amounts );
		/**
		 * DO_ACTION: rtsb_gift_cards_after_product_save
		 *
		 * Allow actions after the gift card product is save.
		 *
		 * @param int $post_id the gift card ID
		 * @param object $post the post object
		 * @param object $product the product object
		 */
		do_action( 'rtsb_gift_cards_after_product_save', $post_id, $post, $product );
	}

	/**
	 * Show controls on backend product page to let create the gift card price
	 */
	public function show_gift_card_product_settings() {
		require_once RTSBPRO_PATH . '/app/Modules/GiftCard/view/gift-cards-product-settings.php';
	}

	/**
	 * Customize WooCommerce product data tabs for Gift Card products.
	 *
	 * @param array $tabs The current product data tabs.
	 *
	 * @return array Modified product data tabs.
	 */
	public function woocommerce_product_data_tabs( $tabs ) {
		 $tabs['inventory']['class'][]       = 'show_if_' . GiftCardFns::GIFT_CARD_KEY;
		 $tabs['variations']['class'][]      = 'hide_if_' . GiftCardFns::GIFT_CARD_KEY;
		 $tabs[ GiftCardFns::GIFT_CARD_KEY ] = [
			 'label'    => __( 'General', 'shopbuilder-pro' ),
			 'target'   => GiftCardFns::GIFT_CARD_KEY,
			 'class'    => [ 'show_if_' . GiftCardFns::GIFT_CARD_KEY ],
			 'priority' => 5,
		 ];
		 return $tabs;
	}
	/**
	 * Add an option to let the admin set the gift card as a physical good or digital goods.
	 *
	 * @param array $array Product type options.
	 *
	 * @return mixed
	 * @since  1.0.0
	 */
	public function add_type_option( $array ) { // phpcs:ignore Universal.NamingConventions.NoReservedKeywordParameterNames.arrayFound
		if ( isset( $array['virtual'] ) ) {
			$css_class     = $array['virtual']['wrapper_class'];
			$add_css_class = 'show_if_rtsb-gift-card';
			$class         = empty( $css_class ) ? $add_css_class : $css_class .= ' ' . $add_css_class;

			$array['virtual']['wrapper_class'] = $class;
		}

		return $array;
	}

	/**
	 * Save the Gift Card product type when a product is updated.
	 *
	 * @param int $post_id The product post ID.
	 */
	public function save_gift_card_product_type( $post_id ) {
        // phpcs:ignore WordPress.Security.NonceVerification.Missing
		if ( sanitize_text_field( wp_unslash( $_POST['product-type'] ?? '' ) ) === GiftCardFns::GIFT_CARD_KEY ) {
			wp_set_object_terms( $post_id, GiftCardFns::GIFT_CARD_KEY, 'product_type' );
			delete_post_meta( $post_id, '_regular_price' );
			delete_post_meta( $post_id, '_sale_price' );
			delete_post_meta( $post_id, '_price' );
		}
	}

	/**
	 * Show gift cards code and amount in order's totals section, in edit order page
	 *
	 * @param int $order_id
	 */
	public function show_gift_cards_total_before_order_totals( $order_id ) {

		$order            = wc_get_order( $order_id );
		$order_gift_cards = $order->get_meta( 'rtsb_applied_gift_cards' );
		$currency         = $order->get_currency();
		if ( ! $order_gift_cards ) {
			return;
		}
		foreach ( $order_gift_cards as $code => $amount ) {
			?>
			<tr>
				<td class="label"><?php echo esc_html__( 'Gift Card: ', 'shopbuilder-pro' ) . esc_html( $code ) . ''; ?></td>
				<td width="1%"></td>
				<td class="total"><?php echo wp_kses_post( wc_price( $amount, [ 'currency' => $currency ] ) ); ?>
				</td>
			</tr>
			<?php
		}
	}
}
