<?php
/**
 * Gift Card Email Class.
 *
 * @package RadiusTheme\SBPRO
 */

namespace RadiusTheme\SBPRO\Modules\GiftCard\Email;

use RadiusTheme\SB\Helpers\Fns;
use RadiusTheme\SBPRO\Modules\GiftCard\GiftCardFns;
use WC_Email;

defined( 'ABSPATH' ) || exit;

/**
 * GiftCardEmail Class - extends WooCommerce email system.
 */
class GiftCardValidityEmail extends WC_Email {
	/**
	 * Email content.
	 *
	 * @var string
	 */
	public $email_content;
	/**
	 * Email content.
	 *
	 * @var string
	 */
	public $before_expiration_days;
	/**
	 * @var int
	 */
	public $gift_card_id;
	/**
	 * Constructor.
	 */
	public function __construct() {
		$this->id             = 'rtsb_send_gift_card_validity_email';
		$this->title          = esc_html__( 'Shopbuilder - Gift Card Expiration Email', 'shopbuilder-pro' );
		$this->description    = esc_html__( 'Notifies the recipient about an upcoming gift card expiration.', 'shopbuilder-pro' );
		$this->template_html  = 'gift-card/gift-card-template-for-email.php';
		$this->customer_email = true;

		$this->subject       = esc_html__( 'Your Gift Card is About to Expire', 'shopbuilder-pro' );
		$this->heading       = esc_html__( 'Don’t Miss Out on Your Gift 🎁', 'shopbuilder-pro' );
		$this->email_content = $this->get_option(
			'email_content',
			sprintf(
			/* translators: line break */
				__( 'Hello {recipient_name},%1$s%1$sJust a quick reminder – your gift card is nearing its expiration date. %1$sBe sure to use it before it’s gone!%1$s%1$sGift Card Details:', 'shopbuilder-pro' ),
				"\n"
			)
		);
		$this->before_expiration_days = $this->get_option( 'send_before_expiration_days', 7 );
		$this->placeholders           = [
			'{recipient_name}' => '',
			'{expiration}'     => '',
			'{site_title}'     => '',
			'{site_address}'   => '',
			'{site_url}'       => '',
			'{store_email}'    => '',
		];
		add_action( 'rtsb/gift/card/reminder', [ $this, 'trigger' ], 10, 2 );
		parent::__construct();
		$this->email_type = 'html';
	}

	/**
	 * Trigger the email.
	 *
	 * @param int $post_id Gift card post ID.
	 * @param int $current Order ID.
	 */
	public function trigger( $post_id, $current ) {
		if ( ! $post_id ) {
			return;
		}
		$this->gift_card_id = $post_id;
		$giftCard           = GiftCardFns::get_card_by_id( $post_id );
		$email_send         = $giftCard->get_card_data( 'reminder_send' );
		if ( $email_send ) {
			 return;
		}

		$this->object                           = $post_id;
		$this->recipient                        = $giftCard->get_card_data( 'recipient_email' );
		$this->placeholders['{recipient_name}'] = $giftCard->get_card_data( 'recipient_name' );
		if ( ! $this->is_enabled() || ! $this->get_recipient() ) {
			return;
		}

		$this->setup_locale();

		$sendEmail = $this->send(
			$this->get_recipient(),
			$this->get_subject(),
			$this->get_content(),
			$this->get_headers(),
			$this->get_attachments()
		);
		if ( $sendEmail ) {
			$giftCard->update_card_info(
				[
					'rtsb_gift_card_reminder_email_send' => $current,
				]
			);
		}
		$this->restore_locale();
	}

	/**
	 * Get HTML content.
	 *
	 * @return string
	 */
	public function get_content_html() {
		ob_start();
		$giftCard        = GiftCardFns::get_card_by_id( $this->gift_card_id );
		$expiration_date = $giftCard->get_card_data( 'expiration_date' );
		$format          = get_option( 'date_format' ) . ' ' . get_option( 'time_format' );
		$expiration_date = ! empty( $expiration_date ) ? date_i18n( $format, strtotime( $expiration_date ) ) : esc_html__( 'Lifetime', 'shopbuilder-pro' );

		$recipient_name    = $giftCard->get_card_data( 'recipient_name' );
		$product_name      = $giftCard->get_card_data( 'product_name' );
		$card_amount       = $giftCard->get_card_data( 'card_amount' );
		$current_amount    = $giftCard->get_card_data( 'remaining_balance' );
		$card_number       = $giftCard->get_card_data( 'card_number' );
		$card_message      = $giftCard->get_card_data( 'card_message' );
		$remaining_balance = $giftCard->get_card_data( 'remaining_balance' );

		$site_title   = $this->get_blogname();
		$site_address = wp_parse_url( home_url(), PHP_URL_HOST );
		$site_url     = wp_parse_url( home_url(), PHP_URL_HOST );
		$store        = $this->get_from_address();

		// Load email body template with the (possibly filtered) data.
		$email_body = nl2br(
			str_replace(
				[
					'{recipient_name}',
					'{site_title}',
					'{expiration}',
					'{site_address}',
					'{site_url}',
					'{store_email}',
				],
				[
					ucwords( $recipient_name ),
					$site_title,
					$expiration_date,
					$site_address,
					$site_url,
					$store,
				],
				$this->email_content
			)
		);
		$data       = apply_filters(
			'rtsb/gift/cards/email/header',
			[
				'heading'         => $recipient_name,
				'recipient_name'  => $recipient_name,
				'current_balance' => wc_price( $current_amount ),
				'card_number'     => $card_number,
				'expiration_date' => $expiration_date,
				'card_message'    => $card_message,
				'email_body'      => $email_body,
			],
			$giftCard
		);

		$email_body   = Fns::load_template( 'gift-card/gift-card-email-body', $data, true, '', rtsbpro()->get_plugin_template_path() );
		$email_footer = esc_html__( 'Best Regards,', 'shopbuilder-pro' ) . '<br/>' . $site_title;
		wc_get_template(
			$this->template_html,
			apply_filters(
				'rtsb/gift/cards/email/data',
				[
					'email_heading'     => $this->get_heading(),
					'post_id'           => $this->gift_card_id,
					'email'             => $this,
					'product_name'      => $product_name,
					'card_amount'       => wc_price( $card_amount ),
					'remaining_balance' => wc_price( $remaining_balance ),
					'card_number'       => $card_number ?: 'ABCD-EFG-EMQ-NP-1GEOGQ',
					'expiration_date'   => $expiration_date,
					'redeem_url'        => get_permalink( wc_get_page_id( 'shop' ) ),
					'email_body'        => $email_body,
					'email_footer'      => $email_footer,
				]
			),
			'',
			rtsbpro()->get_plugin_template_path()
		);
		return ob_get_clean();
	}


	/**
	 * Initialize the form fields for the plugin settings.
	 *
	 * @return void
	 */
	public function init_form_fields() {
		/* translators: %s: list of placeholders */
		$placeholder_text  = sprintf( __( 'Available placeholders: %s', 'shopbuilder-pro' ), '<code>' . implode( '</code>, <code>', array_keys( $this->placeholders ) ) . '</code>' );
		$this->form_fields = [
			'enabled'                     => [
				'title'   => __( 'Enable/Disable', 'shopbuilder-pro' ),
				'type'    => 'checkbox',
				'label'   => __( 'Enable this email notification', 'shopbuilder-pro' ),
				'default' => 'yes',
			],
			'subject'                     => [
				'title'       => __( 'Subject', 'shopbuilder-pro' ),
				'type'        => 'text',
				'desc_tip'    => true,
				'description' => $placeholder_text,
				'placeholder' => $this->get_default_subject(),
				'default'     => $this->get_default_subject(),
			],
			'heading'                     => [
				'title'       => __( 'Email heading', 'shopbuilder-pro' ),
				'type'        => 'text',
				'desc_tip'    => true,
				'description' => $placeholder_text,
				'placeholder' => $this->get_default_heading(),
				'default'     => $this->get_default_heading(),
			],
			'send_before_expiration_days' => [
				'title'       => __( 'Send Reminder Before (Days)', 'shopbuilder-pro' ),
				'type'        => 'number',
				'desc_tip'    => true,
				'description' => __( 'Specify how many days before expiration the email should be sent. For example, enter "7" to send the email 7 days before the gift card expires.', 'shopbuilder-pro' ),
				'placeholder' => 7,
				'default'     => $this->before_expiration_days,
			],
			'email_content'               => [
				'title'       => __( 'Email Content', 'shopbuilder-pro' ),
				'description' => __( 'Text to appear in the main email body.', 'shopbuilder-pro' ) . ' ' . $placeholder_text . '.',
				'css'         => 'width:800px; height: 185px;',
				'placeholder' => '',
				'type'        => 'textarea',
				'default'     => $this->email_content,
			],
		];
	}
}
