<?php
/**
 * Gift Card Email Class.
 *
 * @package RadiusTheme\SBPRO
 */

namespace RadiusTheme\SBPRO\Modules\GiftCard\Email;

use RadiusTheme\SB\Helpers\Fns;
use RadiusTheme\SBPRO\Modules\GiftCard\GiftCardFns;
use WC_Email;

defined( 'ABSPATH' ) || exit;

/**
 * GiftCardEmail Class - extends WooCommerce email system.
 */
class GiftCardUsesActivity extends WC_Email {
	/**
	 * Email content.
	 *
	 * @var string
	 */
	public $email_content;
	/**
	 * @var int
	 */
	public $gift_card_id;
	/**
	 * Constructor.
	 */
	public function __construct() {
		$this->id             = 'rtsb_send_gift_card_activity_email';
		$this->title          = esc_html__( 'Shopbuilder - Gift Card Usage Notification', 'shopbuilder-pro' );
		$this->description    = esc_html__( 'Notifies the recipient when their gift card is used on an order.', 'shopbuilder-pro' );
		$this->template_html  = 'gift-card/gift-card-template-for-email.php';
		$this->customer_email = true;

		$this->subject       = esc_html__( 'Your Gift Card Was Just Used', 'shopbuilder-pro' );
		$this->heading       = esc_html__( 'A Purchase Was Made With Your Gift Card 🎉', 'shopbuilder-pro' );
		$this->email_content = $this->get_option(
			'email_content',
			sprintf(
			/* translators: line break */
				__( 'Hello {recipient_name},%1$s%1$sYour gift card was just used on a recent purchase.%1$sBelow are the details of this transaction, including remaining balance and order summary.%1$s%1$sGift', 'shopbuilder-pro' ),
				"\n"
			)
		);
		$this->placeholders = [
			'{recipient_name}' => '',
			'{site_title}'     => '',
			'{site_address}'   => '',
			'{site_url}'       => '',
			'{store_email}'    => '',
		];
		add_action( 'rtsb/gift/card/activity/email', [ $this, 'trigger' ], 10, 2 );
		parent::__construct();
		$this->email_type = 'html';
	}

	/**
	 * Trigger the email.
	 *
	 * @param object $giftCard Gift card.
	 * @param object $order order.
	 */
	public function trigger( $giftCard, $order ) {
		$this->gift_card_id                     = $giftCard->ID;
		$this->object                           = $order;
		$this->recipient                        = $giftCard->get_card_data( 'recipient_email' );
		$this->placeholders['{recipient_name}'] = $giftCard->get_card_data( 'recipient_name' );
		if ( ! $this->is_enabled() || ! $this->get_recipient() ) {
			return;
		}

		$this->setup_locale();

		$this->send(
			$this->get_recipient(),
			$this->get_subject(),
			$this->get_content(),
			$this->get_headers(),
			$this->get_attachments()
		);
		$this->restore_locale();
	}

	/**
	 * Get HTML content.
	 *
	 * @return string
	 */
	public function get_content_html() {
		ob_start();
		$giftCard        = GiftCardFns::get_card_by_id( $this->gift_card_id );
		$expiration_date = $giftCard->get_card_data( 'expiration_date' );
		$format          = get_option( 'date_format' ) . ' ' . get_option( 'time_format' );
		$expiration_date = ! empty( $expiration_date ) ? date_i18n( $format, strtotime( $expiration_date ) ) : esc_html__( 'Lifetime', 'shopbuilder-pro' );

		$recipient_name    = $giftCard->get_card_data( 'recipient_name' );
		$card_number       = $giftCard->get_card_data( 'card_number' );
		$remaining_balance = $giftCard->get_card_data( 'remaining_balance' );

		$site_title   = $this->get_blogname();
		$site_address = wp_parse_url( home_url(), PHP_URL_HOST );
		$site_url     = wp_parse_url( home_url(), PHP_URL_HOST );
		$store        = $this->get_from_address();

		// Load email body template with the (possibly filtered) data.
		$email_body   = nl2br(
			str_replace(
				[
					'{recipient_name}',
					'{site_title}',
					'{site_address}',
					'{site_url}',
					'{store_email}',
				],
				[
					ucwords( $recipient_name ),
					$site_title,
					$site_address,
					$site_url,
					$store,
				],
				$this->email_content
			)
		);
		$activity     = $giftCard->get_card_activity();
		$email_footer = esc_html__( 'Best Regards,', 'shopbuilder-pro' ) . '<br/>' . $site_title;
		$data         = apply_filters(
			'rtsb/view/gift/card/activity/email',
			[
				'activity'          => $activity,
				'heading'           => $recipient_name,
				'remaining_balance' => wc_price( $remaining_balance ),
				'card_number'       => $card_number,
				'expiration_date'   => $expiration_date,
				'email_body'        => $email_body,
				'email_heading'     => $this->get_heading(),
				'email'             => $this,
				'redeem_url'        => get_permalink( wc_get_page_id( 'shop' ) ),
				'email_footer'      => $email_footer,
			],
			$giftCard
		);
		Fns::load_template( 'gift-card/gift-card-activity-email', $data, false, '', rtsbpro()->get_plugin_template_path() );
		return ob_get_clean();
	}


	/**
	 * Initialize the form fields for the plugin settings.
	 *
	 * @return void
	 */
	public function init_form_fields() {
		/* translators: %s: list of placeholders */
		$placeholder_text  = sprintf( __( 'Available placeholders: %s', 'shopbuilder-pro' ), '<code>' . implode( '</code>, <code>', array_keys( $this->placeholders ) ) . '</code>' );
		$this->form_fields = [
			'enabled'       => [
				'title'   => __( 'Enable/Disable', 'shopbuilder-pro' ),
				'type'    => 'checkbox',
				'label'   => __( 'Enable this email notification', 'shopbuilder-pro' ),
				'default' => 'yes',
			],
			'subject'       => [
				'title'       => __( 'Subject', 'shopbuilder-pro' ),
				'type'        => 'text',
				'desc_tip'    => true,
				'description' => $placeholder_text,
				'placeholder' => $this->get_default_subject(),
				'default'     => $this->get_default_subject(),
			],
			'heading'       => [
				'title'       => __( 'Email heading', 'shopbuilder-pro' ),
				'type'        => 'text',
				'desc_tip'    => true,
				'description' => $placeholder_text,
				'placeholder' => $this->get_default_heading(),
				'default'     => $this->get_default_heading(),
			],
			'email_content' => [
				'title'       => __( 'Email Content', 'shopbuilder-pro' ),
				'description' => __( 'Text to appear in the main email body.', 'shopbuilder-pro' ) . ' ' . $placeholder_text . '.',
				'css'         => 'width:800px; height: 185px;',
				'placeholder' => '',
				'type'        => 'textarea',
				'default'     => $this->email_content,
			],
		];
	}
}
