<?php
/**
 * Main FilterHooks class.
 *
 * @package RadiusTheme\SB
 */

namespace RadiusTheme\SBPRO\Modules\CurrencySwitcher;

use RadiusTheme\SB\Helpers\Fns;
use RadiusTheme\SB\Helpers\Cache;
use RadiusTheme\SB\Models\DataModel;
use RadiusTheme\SBPRO\Traits\SingletonTrait;

defined( 'ABSPATH' ) || exit();

/**
 * Main FilterHooks class.
 */
class CurrencySwitcher extends CurrencyBase {
	/**
	 * Singleton Trait.
	 */
	use SingletonTrait;

	/**
	 * Notifications hooks.
	 */
	private function __construct() {
		parent::__construct();
		// $this->set_currency();
		add_action( 'template_redirect', [ $this, 'set_currency' ], 8 );
		add_action( 'template_redirect', [ $this, 'update_currency_on_cart_page' ] );
		add_action( 'woocommerce_update_options_general', [ $this, 'woocommerce_currency_settings' ], 99, 2 );
		add_action( 'rtsb/after/saved/settings/success/modules/currency_switcher', [ $this, 'repeater_currency_settings' ], 20 );

		// Currency Update Via AJAX.
		add_action( 'wp_ajax_rtsb_update_currency_rates', [ $this, 'get_latest_currency_rates' ] );
		// add_action( 'wp_ajax_nopriv_rtsb_update_currency_rates', [ $this, 'get_latest_currency_rates' ] );

		// Schedules Event.
		add_filter( 'cron_schedules', [ $this, 'currency_custom_cron_schedule' ] );
		add_action( 'update_exchange_rates_hourly', [ $this, 'func_update_exchange_rates_hourly' ] );
		register_deactivation_hook( RTSBPRO_FILE, [ $this, 'deactivate_exchange_rates_hourly' ] );
		$this->currency_cron_schedules();
	}

	/**
	 * @return void
	 */
	public function update_currency_on_cart_page() {
		global $currency_switcher_options; // Define global variable.
		// If the cached result doesn't exist, fetch it from the database.
		$update_on_cart_page = is_cart() && ( $currency_switcher_options['update_currency_on_cart_page'] ?? false );
		$is_geo              = 'on' === ( $currency_switcher_options['enable_geolocation'] ?? 'off' );

		$checkout_currency = $currency_switcher_options['checkout_currency'] ?? 'customer-selected';
		$customer_selected = 'customer-selected' === $checkout_currency;

		if ( $customer_selected || ! ( $update_on_cart_page || is_checkout() ) ) {
			return;
		}
		// return;
		$available_currency = array_keys( $this->get_available_currency() );

		$default_currency = 'default-currency' === $checkout_currency;
		if ( $default_currency && $this->get_default_currency() !== Fns::getSession( 'rtsb_currency' ) ) {
			Fns::setSession( 'rtsb_currency', $this->get_default_currency() );
			Fns::setSession( 'rtsb_currency_changed_on_cart', true );
			Cache::clear_all_cache();
		}

		$customer_country = 'customer-country' === $checkout_currency;

		if ( $customer_country ) {
			$takes_country = $currency_switcher_options['checkout_base_on_customers_country'] ?? '';
			$country       = '';
			if ( 'billing' === $takes_country ) {
				$country = WC()->customer->get_billing_country();
			} elseif ( 'shipping' === $takes_country ) {
				$country = WC()->customer->get_shipping_country();
			}
			$currency = CsFns::get_country_or_currency( $country, 'currency' );
			if ( Fns::getSession( 'rtsb_currency' ) !== $currency && in_array( $currency, $available_currency, true ) ) {
				Fns::setSession( 'rtsb_currency', $currency );
				Fns::setSession( 'rtsb_currency_changed_on_cart', true );
				Cache::clear_all_cache();
			}
		}

		if ( Fns::getSession( 'rtsb_currency_changed_on_cart' ) ) {
			Fns::setSession( 'rtsb_currency_changed_on_cart', false );
			$params = [ rtsb()->nonceId ];
			if ( ! $is_geo ) {
				$params[] = 'currency';// Redirect to the same page.
			}
			wp_safe_redirect( remove_query_arg( $params ) );
			exit;
		}
	}
	/**
	 * @return void
	 */
	public function set_currency() {
		// If the cached result doesn't exist, fetch it from the database.
		global $currency_switcher_options; // Define global variable.
		$is_geo = 'on' === ( $currency_switcher_options['enable_geolocation'] ?? 'off' );

		$available_currency = array_keys( $this->get_available_currency() );
		$nonce_validate     = ! empty( $_GET['currency'] ) && wp_verify_nonce( Fns::get_nonce(), rtsb()->nonceText );

		global $wp;
		$redirect_url = home_url( add_query_arg( [], $wp->request ) );

		if ( $nonce_validate ) {
			$currency = sanitize_text_field( wp_unslash( $_GET['currency'] ) );
			if ( in_array( $currency, $available_currency, true ) ) {
				Fns::setSession( 'rtsb_currency', $currency );
				Fns::setSession( 'rtsb_currency_changed', true );
				Cache::clear_all_cache();
			}
			// Redirect to the same page without the 'currency' query string.
			 wp_safe_redirect( $redirect_url );
			 exit;

		} elseif ( $is_geo && ! Fns::getSession( 'rtsb_currency' ) && ! ( is_cart() || is_checkout() ) ) {
			 $currency = $this->get_geo_currency();
			if ( in_array( $currency, $available_currency, true ) ) {
				 Fns::setSession( 'rtsb_currency', $currency );
				 Fns::setSession( 'rtsb_currency_changed', true );
				// Redirect to the same page without the 'currency' query string.
				Cache::clear_all_cache();
				wp_safe_redirect( $redirect_url );
				 exit;
			}
		}
		if ( ! $this->is_available_currency() ) {
			Fns::removeSession( 'rtsb_currency' );
			Fns::setSession( 'rtsb_currency', CsFns::get_default_currency() );
			Fns::setSession( 'rtsb_currency_changed', true );
		}
	}

	/**
	 * @return void
	 */
	public function woocommerce_currency_settings() {
		$currency     = get_option( 'woocommerce_currency' );
		$currency_pos = get_option( 'woocommerce_currency_pos' );
		$thousand_sep = get_option( 'woocommerce_price_thousand_sep' );
		$decimal_sep  = get_option( 'woocommerce_price_decimal_sep' );
		$num_decimals = get_option( 'woocommerce_price_num_decimals' );
		Fns::set_repeater_options(
			'modules',
			'currency_switcher',
			'currency_options',
			'title',
			$currency,
			[
				'activate_currency'  => 'on',
				'currency_rate'      => 1,
				'currency_position'  => $currency_pos,
				'thousand_separator' => $thousand_sep,
				'decimal_separator'  => $decimal_sep,
				'number_of_decimals' => $num_decimals,
			]
		);
	}

	/**
	 * @return void
	 */
	public function repeater_currency_settings( $rawOptions ) {
		$currency = get_option( 'woocommerce_currency' );
		$modules  = DataModel::source()->get_option( 'modules', [], false );

		if ( empty( $modules['currency_switcher']['currency_options'] ) ) {
			return;
		}

		$options = $modules['currency_switcher']['currency_options'];

		$repeater_options = json_decode( $options, true );
		// Check if options are not empty and are in array format.
		if ( ! is_array( $repeater_options ) ) {
			return;
		}
		// Use array_column to get an array of values from $compare_key.
		$column_values = array_column( $repeater_options, 'title' );
		// Use array_search to find the index of the $compare_value.
		$index = array_search( $currency, $column_values, true );
		if ( false === $index ) {
			return;
		}

		if ( ! empty( $repeater_options[ $index ]['currency_position'] ) ) {
			update_option( 'woocommerce_currency_pos', $repeater_options[ $index ]['currency_position'] );
		}
		if ( ! empty( $repeater_options[ $index ]['thousand_separator'] ) ) {
			update_option( 'woocommerce_price_thousand_sep', $repeater_options[ $index ]['thousand_separator'] );
		}
		if ( ! empty( $repeater_options[ $index ]['decimal_separator'] ) ) {
			update_option( 'woocommerce_price_decimal_sep', $repeater_options[ $index ]['decimal_separator'] );
		}
		if ( ! empty( $repeater_options[ $index ]['number_of_decimals'] ) ) {
			update_option( 'woocommerce_price_num_decimals', $repeater_options[ $index ]['number_of_decimals'] );
		}
	}


	/**
	 * @param $schedules
	 * @return mixed
	 */
	public function currency_custom_cron_schedule( $schedules ) {
		global $currency_switcher_options; // Define global variable.
		$update_time       = absint( $currency_switcher_options['update_exchange_rates'] ?? 24 );
		$key               = 'currency_custom_cron_schedule_' . $update_time;
		$interval          = absint( $update_time ?: 24 ) * 3600;
		$schedules[ $key ] = [
			'interval' => $interval,
			'display'  => esc_html__( 'Custom Schedule Hourly', 'shopbuilder-pro' ),
		];
		return $schedules;
	}

	/**
	 * @return void
	 */
	public function currency_cron_schedules() {
		$event_hook = 'update_exchange_rates_hourly';
		// Check if the cron job is already scheduled.
		$is_scheduled = wp_next_scheduled( $event_hook );
		// $scheduled_time = date( 'Y-m-d H:i:s', $is_scheduled );
		global $currency_switcher_options; // Define global variable.
		$update_time = absint( $currency_switcher_options['update_exchange_rates'] ?? 24 );
		$key         = 'currency_custom_cron_schedule_' . $update_time;
		if ( $is_scheduled ) {
			$schedule = false;
			$cron     = _get_cron_array();
			// update_exchange_rates_hourly.
			if ( ! empty( $cron[ $is_scheduled ]['update_exchange_rates_hourly'] ) ) {
				$current_cron = $cron[ $is_scheduled ]['update_exchange_rates_hourly'];
				foreach ( $current_cron as $uniq_key => $cronhooks ) {
					if ( isset( $cronhooks['schedule'] ) ) {
						$schedule = $cronhooks['schedule'];
						break;
					}
				}
			}
			// Clear any existing scheduled events with the same hook.
			if ( $schedule && $schedule !== $key ) {
				wp_clear_scheduled_hook( $event_hook );
			}
			return; // Cron job is already scheduled, no need to proceed further.
		}
		wp_clear_scheduled_hook( $event_hook );
		wp_schedule_event( time(), $key, $event_hook );
	}

	/** Deactive Shedule
	 *
	 * @return void
	 */
	public function deactivate_exchange_rates_hourly() {
		wp_clear_scheduled_hook( 'update_exchange_rates_hourly' );
	}

	/**
	 * Shedule Event.
	 *
	 * @return void
	 */
	public function func_update_exchange_rates_hourly() {
		global $currency_switcher_options; // Define global variable.
		$exchange_rates = $currency_switcher_options['exchange_rates'] ?? 'manually';
		if ( 'automatically' !== $exchange_rates ) {
			return;
		}
		$api = FetchCurrencyApi::instance();
		$api->automatically_update_all_currency_rate();
	}

	/**
	 * @return void
	 * @throws \Exception
	 */
	public function get_latest_currency_rates() {

		if ( ! wp_verify_nonce( Fns::get_nonce(), rtsb()->nonceText ) ) {
			wp_send_json_error(
				[
					'message' => __( 'Error: Authentication Error', 'shopbuilder-pro' ),
				]
			);
		}
		$currency_options = json_decode( stripslashes_deep( $_REQUEST['currency_options'] ?? '{}' ), true );
		if ( empty( $currency_options ) || ! is_array( $currency_options ) ) {
			wp_send_json_error(
				[
					'message' => __( 'Error: Not Valid Data', 'shopbuilder-pro' ),
				]
			);
		}
		$currencies = [];
		$api        = FetchCurrencyApi::instance();
		$failed     = [];
		foreach ( $currency_options as  $currency ) {

			$key = $currency['title'];
			if ( empty( $currency['title'] ) ) {
				$currencies[] = $currency;
				continue;
			}

			if ( CsFns::get_default_currency() === $currency['title'] ) {
				$currency['currency_rate'] = 1;
				$currencies[]              = $currency;
				continue;
			}

			$rate = $api->convert_to( $key )->get_the_rate();

			if ( ! $rate ) {
				$failed[]     = $key;
				$currencies[] = $currency;
				continue;
			}
			$currency['currency_rate'] = number_format( $rate, 5, '.', '' );
			$currencies[]              = $currency;
		}
		if ( count( $failed ) > 0 ) {
			wp_send_json_error(
				[
					'message'          => __( 'Error: Failed to update exchange rates for ', 'shopbuilder-pro' ) . implode( ', ', $failed ),
					'currency_options' => wp_json_encode( $currencies ),
				]
			);
		} else {
			wp_send_json_success(
				[
					'message'          => __( 'Rates updated successfully. Please save now.', 'shopbuilder-pro' ),
					'currency_options' => wp_json_encode( $currencies ),
				]
			);
		}
	}
}
