<?php
/**
 * Main FilterHooks class.
 *
 * @package RadiusTheme\SB
 */

namespace RadiusTheme\SBPRO\Modules\CheckoutEditor;

use RadiusTheme\SB\Helpers\Fns;
use RadiusTheme\SB\Modules\CheckoutEditor\CheckoutEditorBase;
use RadiusTheme\SBPRO\Traits\SingletonTrait;

defined( 'ABSPATH' ) || exit();

/**
 * Main FilterHooks class.
 */
class CheckoutManager extends CheckoutEditorCustomData {
	/**
	 * Singleton Trait.
	 */
	use SingletonTrait;

	/**
	 * Class construction.
	 */
	private function __construct() {
		parent::__construct();
		add_filter( 'woocommerce_form_field', [ $this, 'customize_form_field' ], 12, 4 );
		add_action( 'woocommerce_process_shop_order_meta', [ $this, 'custom_order_fields_processing' ], 40 );
		add_action( 'woocommerce_checkout_update_order_meta', [ $this, 'checkout_field_update_order_meta' ] );
		add_filter( 'woocommerce_order_details_after_customer_details', [ $this, 'custom_fields_data_for_thankyou_page' ], 20 );
		add_action( 'woocommerce_admin_order_data_after_billing_address', [ $this, 'custom_billing_address_field_display' ], 10 );
		add_action( 'woocommerce_admin_order_data_after_shipping_address', [ $this, 'custom_shipping_address_field_display' ], 10 );
		add_action( 'woocommerce_email_customer_details', [ $this, 'display_custom_data_in_order_email' ], 40 );
		// Footer Script.
		add_action( 'wp_footer', [ $this, 'add_scriptcheckout_page' ], 99 );
	}

	/**
	 * @param \WC_Order $order The order object to retrieve the fields from.
	 * @return void
	 */
	public function display_custom_data_in_order_email( $order ) {
		$data = $this->get_custom_data_for_email( $order );
		foreach ( $data as $field ) {
			if ( empty( $field['fields'] ) ) {
				continue;
			}
			Fns::load_template( 'checkout-custom-fields/email', $field, false, '', rtsbpro()->get_plugin_template_path() );
		}
	}
	/**
	 * Display Custom Checkout Fields Data on Thankyou page
	 */
	public function custom_fields_data_for_thankyou_page( $order ) {
		$data = $this->get_custom_data_for_thankyou_page( $order );
		?>
		<div class="rtsb-custom-information-wrapper">
			<?php
			foreach ( $data as $field ) {
				if ( empty( $field['fields'] ) ) {
					continue;
				}
				Fns::load_template( 'checkout-custom-fields/table', $field, false, '', rtsbpro()->get_plugin_template_path() );
			}
			?>
		</div>
		<?php
	}

	/**
	 * Update the order meta with field value
	 */
	public function custom_order_fields_processing( $order_id ) {
		// Check the nonce.
		if ( empty( $_POST['woocommerce_meta_nonce'] ) || ! wp_verify_nonce( wp_unslash( $_POST['woocommerce_meta_nonce'] ), 'woocommerce_save_data' ) ) { // phpcs:ignore WordPress.Security.ValidatedSanitizedInput.InputNotSanitized
			return;
		}
		$this->checkout_field_update_order_meta( $order_id );
	}

	/**
	 * @return void
	 */
	public function add_scriptcheckout_page() {
		if ( is_checkout() ) {
			?>
			<script>
				jQuery(document).ready(function ($) {
					$('select.rtsb_custom_field').selectWoo();
				});
			</script>
			<?php
		}
	}
	/**
	 * Customize a form field based on its type and options.
	 *
	 * @param string $field  The initial form field markup.
	 * @param string $key    The name or key of the field.
	 * @param array  $args   Arguments to customize the field (e.g., type, options, class).
	 * @param mixed  $value  The current value or values associated with the field.
	 *
	 * @return string        The customized form field HTML markup.
	 */
	public function customize_form_field( $field, $key, $args, $value ) {
		if ( ! empty( $args['required'] ) ) {
			$required = '&nbsp;<abbr class="required" title="' . esc_attr__( 'required', 'shopbuilder-pro' ) . '">*</abbr>';
		} else {
			$required = '&nbsp;<span class="optional">(' . esc_html__( 'optional', 'shopbuilder-pro' ) . ')</span>';
		}
		$args['class'][] = 'form-row';
		switch ( $args['type'] ) {
			case 'heading':
				$field .= sprintf(
					'<%1$s class="checkout-heading %2$s">%3$s</%1$s>',
					esc_html( $args['tag'] ?? 'h3' ),
					esc_html( implode( ' ', ( $args['class'] ?? [] ) ) ),
					esc_html( $args['label'] ?? '' )
				);
				break;
			case 'multiselect':
				$options           = '';
				$custom_attributes = [];
				$value             = is_array( $value ) ? $value : array_map( 'trim', (array) explode( ',', $value ) );
				if ( ! empty( $args['options'] ) ) {
					foreach ( $args['options'] as $option_key => $option_text ) {
						if ( '' === $option_key ) {
							// If we have a blank option, select2 needs a placeholder.
							if ( empty( $args['placeholder'] ) ) {
								$args['placeholder'] = $option_text ?: __( 'Choose an option', 'shopbuilder-pro' );
							}
							$custom_attributes[] = 'data-allow_clear="true"';
						}
						$selected = in_array( $option_key, $value, true );

						$options .= '<option value="' . esc_attr( $option_key ) . '" ' . selected( $selected, 1, false ) . '>' . esc_html( $option_text ) . '</option>';
					}
					$field .= '<p class="woocommerce-multicheckbox-wrapper ' . esc_attr( implode( ' ', $args['class'] ) ) . '" id="' . esc_attr( $key ) . '_field" data-priority="' . esc_attr( ( $args['priority'] ?? 10 ) ) . '">';
					$field .= '<label for="' . esc_attr( $key ) . '" class="' . esc_attr( implode( ' ', $args['label_class'] ) ) . '">' . wp_kses_post( $args['label'] ) . $required . '</label>';
					$field .= '<select id="' . esc_attr( $key ) . '" multiple name="' . esc_attr( $key ) . '[]" id="' . esc_attr( $args['id'] ) . '" class="select rtsb_custom_field " ' . implode( ' ', $custom_attributes ) . ' data-placeholder="' . esc_attr( $args['placeholder'] ) . '">
							' . $options . '
						</select>';
					$field .= '</p>';
				}
				break;
			case 'checkboxgroup':
				$value = is_array( $value ) ? $value : array_map( 'trim', (array) explode( ',', $value ) );
				if ( ! empty( $args['options'] ) ) {
					$field .= ' <p class="woocommerce-checkboxgroup-wrapper ' . esc_attr( implode( ' ', $args['class'] ) ) . ' " id="' . esc_attr( $key ) . '_field" data-priority="' . esc_attr( ( $args['priority'] ?? 10 ) ) . '">';
					$field .= '<label for="' . esc_attr( $key ) . '" class="' . esc_attr( implode( ' ', $args['label_class'] ) ) . '">' . wp_kses_post( $args['label'] ) . $required . '</label>';
					foreach ( $args['options'] as $option_value => $option_label ) {
						$field .= '<label for="' . esc_attr( sanitize_title( $key . $option_value ) ) . '" ><input id="' . esc_attr( sanitize_title( $key . $option_value ) ) . '" type="checkbox" name="' . esc_attr( $key ) . '[]" value="' . esc_attr( $option_value ) . '"' . checked( in_array( $option_value, $value ), 1, false ) . ' /> ' . esc_html( $option_label ) . '</label>';
					}
					$field .= '</span>';

				}

				break;
		}
		return $field;
	}
	/**
	 * @param $custom_fields
	 * @return array
	 */
	private function process_before_save_custom_data( $custom_fields, $isDataDave = true ) {
		if ( empty( $custom_fields ) ) {
			return [];
		}
		$process_fields = [];
		foreach ( $custom_fields as $key => $field ) {
			if ( ! isset( $_POST[ $key ] ) ) {
				continue;
			}
			$val = [
				'label' => $field['label'],
				'key'   => $key,
			];

			switch ( $field['type'] ) {
				case 'multiselect':
				case 'checkboxgroup':
					$optionval = [];
					if ( $isDataDave && is_array( $_POST[ $key ] ) ) {
						foreach ( $_POST[ $key ] as $value ) {
							$optionval[ sanitize_text_field( $value ) ] = sanitize_text_field( $field['options'][ $value ] );
						}
					}
					$val['value'] = $optionval;
					break;
				case 'select':
				case 'radio':
					if ( $isDataDave ) {
						$radio_key                  = sanitize_text_field( $_POST[ $key ] );
						$radio_label                = $field['options'][ $radio_key ] ?? '';
						$val['value'][ $radio_key ] = sanitize_text_field( $radio_label );
					} else {
						$val['value'] = [];
					}
					break;
				default:
					$val['value'] = $isDataDave ? sanitize_text_field( $_POST[ $key ] ) : '';
			}

			$process_fields[ $key ] = $val;
		}
		return $process_fields;
	}

	/**
	 * @param integer $order_id Order id.
	 * @return void
	 */
	public function checkout_field_update_order_meta( $order_id ) {
		if ( empty( $this->checkout_custom_fields ) ) {
			return;
		}
		$order = wc_get_order( $order_id );
		if ( ! empty( $this->billing_custom_field ) ) {
			 $order->update_meta_data( 'rtsb_billing_custom_fields', $this->process_before_save_custom_data( $this->billing_custom_field ) );
		}
		if ( ! empty( $this->shipping_custom_field ) ) {
			$isDifferentAddress = is_checkout() ? ! empty( $_POST['ship_to_different_address'] ) : true;
			$order->update_meta_data( 'rtsb_shipping_custom_fields', $this->process_before_save_custom_data( $this->shipping_custom_field, $isDifferentAddress ) );
		}
		if ( ! empty( $this->additional_custom_field ) ) {
			$order->update_meta_data( 'rtsb_additional_custom_fields', $this->process_before_save_custom_data( $this->additional_custom_field ) );
		}
		 $order->save_meta_data();
	}

	/**
	 * @param $order
	 * @return void
	 */
	public function custom_billing_address_field_display( $order ) {
		$fields = $order->get_meta( 'rtsb_billing_custom_fields', true );
		if ( empty( $fields ) ) {
			return;
		}
		$title = ! empty( $this->options['checkout_billing_title'] ) ? esc_html( $this->options['checkout_billing_title'] ) : esc_html__( 'Billing Information', 'shopbuilder-pro' );
		?>
		<br class="clear" />

		<h4><?php echo esc_html( $title ); ?></h4>
		<?php
		$this->order_details( 'billing', $fields );
	}

	/**
	 * @param $order
	 * @return void
	 */
	public function custom_shipping_address_field_display( $order ) {
		$fields = $order->get_meta( 'rtsb_shipping_custom_fields', true );
		if ( ! empty( $fields ) ) {
			$title = ! empty( $this->options['checkout_shipping_title'] ) ? esc_html( $this->options['checkout_shipping_title'] ) : esc_html__( 'Shipping Information', 'shopbuilder-pro' );
			?>
			<br/>
			<h4 class="form-field form-field-wide"><?php echo esc_html( $title ); ?></h4>
			<?php
			$this->order_details( 'shipping', $fields );
		}
		$fields = $order->get_meta( 'rtsb_additional_custom_fields', true );
		if ( ! empty( $fields ) ) {
			$title = ! empty( $this->options['checkout_additional_title'] ) ? esc_html( $this->options['checkout_additional_title'] ) : esc_html__( 'Additional Information', 'shopbuilder-pro' );
			?>
			<br class="clear" />
			<h4><?php echo esc_html( $title ); ?></h4>
			<?php
			$this->order_details( 'additional', $fields );
		}
	}

	/**
	 * @param string $type fields type.
	 * @param array  $fields custom fields.
	 * @return void
	 */
	private function order_details( $type, $fields ) {
		$custom_fields = [];
		if ( 'billing' === $type ) {
			$custom_fields = $this->billing_custom_field;
		} elseif ( 'shipping' === $type ) {
			$custom_fields = $this->shipping_custom_field;
		} elseif ( 'additional' === $type ) {
			$custom_fields = $this->additional_custom_field;
		}
		if ( empty( $custom_fields ) ) {
			return;
		}
		?>
		<div class="address">
			<?php
			$has_fields_value = false;
			foreach ( $fields as $field ) {
				if ( empty( $field['value'] ) ) {
					continue;
				}
				$has_fields_value = true;
				?>
				<p>
					<strong><?php echo esc_html( $field['label'] ); ?>:</strong>
					<?php
					if ( is_array( $field['value'] ) ) {
						echo esc_html( implode( ', ', $field['value'] ) );
					} else {
						echo esc_html( $field['value'] );
					}
					?>
				</p>
				<?php
			}
			if ( ! $has_fields_value ) {
				?>
				<p> <?php echo esc_html__( 'N/A', 'shopbuilder-pro' ); ?> </p>
				<?php
			}
			?>
		</div>
		<div class="edit_address">
			<?php
			foreach ( $fields as $field ) {
				$args       = $custom_fields[ $field['key'] ] ?? [];
				$args       = wp_parse_args( $args, $field );
				$args       = wp_parse_args(
					$args,
					[
						'id'            => '',
						'label'         => '',
						'value'         => '',
						'placeholder'   => '',
						'wrapper_class' => 'form-field-wide',
						'type'          => '',
					]
				);
				$args['id'] = $args['key'] ?? '';
				if ( 'checkbox' === $args['type'] ) {
					$args['style'] = 'width:0';
				}
				$args['class'] = implode( ' ', array_map( 'sanitize_html_class', ( $args['class'] ?? [] ) ) );
				if ( in_array( $args['type'], [ 'radio', 'select' ], true ) ) {
					if ( is_array( $args['value'] ) && ! empty( $args['value'] ) ) {
						$first_value = key( $args['value'] );
					} else {
						$first_value = $args['value'] ?: '';
					}
					$args['value'] = $first_value;
				}
				switch ( $args['type'] ) {
					case 'radio':
						$args['class'] = $args['class'] . ' checkbox ';
						woocommerce_wp_radio( $args );
						break;
					case 'select':
						woocommerce_wp_select( $args );
						break;
					case 'checkbox':
						woocommerce_wp_checkbox( $args );
						break;
					case 'textarea':
						woocommerce_wp_textarea_input( $args );
						break;
					case 'multiselect':
						$args['custom_attributes']['multiple'] = 'multiple';
						$args['value']                         = is_array( $args['value'] ) ? array_keys( $args['value'] ) : [];
						$args['class']                         = $args['class'] . ' rtsb_custom_field ';
						$args['name']                          = $args['id'] . '[]';
						woocommerce_wp_select( $args );
						break;
					case 'checkboxgroup':
						if ( ! empty( $args['value'] ) ) {
							$value = is_array( $args['value'] ) ? $args['value'] : array_map( 'trim', (array) explode( ',', $args['value'] ) );
							$value = array_keys( $value );
						} else {
							$value = [];
						}

						$args['class'] = $args['class'] . ' checkbox ';
						if ( ! empty( $args['options'] ) ) {
							echo '<p class="form-field ' . esc_attr( $args['wrapper_class'] ) . '" >';
							?>
							<label><?php echo esc_html( $field['label'] ); ?></label>
							<?php
							foreach ( $args['options'] as $option_value => $option_label ) {
								?>
								<label>
									<input
										type="checkbox"
										name="<?php echo esc_attr( $args['id'] ); ?>[]"
										value="<?php echo esc_attr( $option_value ); ?>"
										class="<?php echo esc_attr( $args['class'] ); ?>"
										<?php echo checked( in_array( $option_value, $value, true ), 1, false ); ?>
									/>
									<?php echo esc_html( $option_label ); ?>
								</label>
								<?php
							}
							echo '</p>';
						}
						break;
					default:
						 woocommerce_wp_text_input( $args );
						break;
				}
			}
			?>
		</div>
		<script>
			jQuery(document).ready(function ($) {
				$('select.rtsb_custom_field').selectWoo();
			});
		</script>
		<?php
	}
}
