<?php
/**
 * Main FilterHooks class.
 *
 * @package RadiusTheme\SB
 */

namespace RadiusTheme\SBPRO\Modules\CheckoutEditor;

use RadiusTheme\SB\Helpers\Fns;
use RadiusTheme\SB\Modules\CheckoutEditor\CheckoutEditorBase;
use RadiusTheme\SBPRO\Traits\SingletonTrait;

defined( 'ABSPATH' ) || exit();

/**
 * Main FilterHooks class.
 */
class CheckoutEditorCustomData extends CheckoutEditorBase {

	/**
	 * @return array[]
	 */
	public function get_custom_data_for_email( $order ) {
		return [
			[
				'title'  => ! empty( $this->options['checkout_billing_title'] ) ? esc_html( $this->options['checkout_billing_title'] ) : esc_html__( 'Billing Information', 'shopbuilder-pro' ),
				'fields' => $this->fields_for( 'billing', $order, 'isShowinEmail' ),
			],
			[
				'title'  => ! empty( $this->options['checkout_shipping_title'] ) ? esc_html( $this->options['checkout_shipping_title'] ) : esc_html__( 'Shipping Information', 'shopbuilder-pro' ),
				'fields' => $this->fields_for( 'shipping', $order, 'isShowinEmail' ),
			],
			[
				'title'  => ! empty( $this->options['checkout_additional_title'] ) ? esc_html( $this->options['checkout_additional_title'] ) : esc_html__( 'Additional Information', 'shopbuilder-pro' ),
				'fields' => $this->fields_for( 'additional', $order, 'isShowinEmail' ),
			],
		];
	}
	/**
	 * @return array[]
	 */
	public function get_custom_data_for_thankyou_page( $order ) {
		return [
			'billing'    => [
				'title'  => ! empty( $this->options['checkout_billing_title'] ) ? esc_html( $this->options['checkout_billing_title'] ) : esc_html__( 'Billing Information', 'shopbuilder-pro' ),
				'fields' => $this->fields_for( 'billing', $order ),
			],
			'shipping'   => [
				'title'  => ! empty( $this->options['checkout_shipping_title'] ) ? esc_html( $this->options['checkout_shipping_title'] ) : esc_html__( 'Shipping Information', 'shopbuilder-pro' ),
				'fields' => $this->fields_for( 'shipping', $order ),
			],
			'additional' => [
				'title'  => ! empty( $this->options['checkout_additional_title'] ) ? esc_html( $this->options['checkout_additional_title'] ) : esc_html__( 'Order Note Information', 'shopbuilder-pro' ),
				'fields' => $this->fields_for( 'additional', $order ),
			],
		];
	}
	/**
	 * Retrieves custom fields based on the specified type for a given order.
	 *
	 * @param string    $type  The type of fields to retrieve ('billing', 'shipping', or 'additional').
	 * @param \WC_Order $order  The order object to retrieve the fields from.
	 *
	 * @return array  Returns an array of custom fields for the specified type, or an empty array if no fields are found.
	 */
	protected function fields_for( $type, $order, $for = 'isShowinOrderDetails' ) {
		$thankYouData = [];
		if ( 'billing' === $type ) {
			$fields       = $order->get_meta( 'rtsb_billing_custom_fields', true );
			$thankYouData = $this->fields_filtering( $fields, $this->billing_custom_field, $for );
		} elseif ( 'shipping' === $type ) {
			$fields       = $order->get_meta( 'rtsb_shipping_custom_fields', true );
			$thankYouData = $this->fields_filtering( $fields, $this->shipping_custom_field, $for );
		} elseif ( 'additional' === $type ) {
			$fields       = $order->get_meta( 'rtsb_additional_custom_fields', true );
			$thankYouData = $this->fields_filtering( $fields, $this->additional_custom_field, $for );
		}
		return $thankYouData;
	}

	/**
	 * @param $input
	 * @return string
	 */
	private function getWeekDateRangeFromInput( $input ) {
		if ( empty( $input ) ) {
			return '';
		}
		// Split the input string "2024-W44" to get the year and week number.
		list($year, $week) = explode( '-W', $input );

		$startOfWeek = new \DateTime();
		// Set the start date to the first day of the specified week.
		$startOfWeek->setISODate( $year, $week );

		// Clone the start date and add 6 days to get the end of the week.
		$endOfWeek = clone $startOfWeek;
		$endOfWeek->modify( '+6 days' );
		return $startOfWeek->format( 'Y-m-d' ) . ' TO ' . $endOfWeek->format( 'Y-m-d' );
	}

	/**
	 * Filters meta fields based on a specified condition and custom fields.
	 *
	 * @param array  $meta_fields  An array of meta fields to be filtered.
	 * @param array  $custom_fields  An array of custom fields with additional settings.
	 * @param string $for The key to check in the custom fields, default is 'isShowinOrderDetails'.
	 *
	 * @return array  Returns an array of filtered meta fields where the corresponding custom field meets the specified condition.
	 */
	private function fields_filtering( $meta_fields, $custom_fields, $for = 'isShowinOrderDetails' ) {
		$thankYouData = [];
		if ( empty( $meta_fields ) || empty( $custom_fields ) ) {
			return $thankYouData;
		}
		foreach ( $meta_fields as $key => $field ) {
			if ( empty( $custom_fields[ $key ] ) ) {
				continue;
			}
			if ( ! empty( $custom_fields[ $key ][ $for ] ) ) {
				if ( 'week' === ( $custom_fields[ $key ]['type'] ?? '' ) && ! empty( $field['value'] ) ) {
					$field['value'] = $field['value'] . '<br/><span style="font-size: 13px;">( ' . $this->getWeekDateRangeFromInput( $field['value'] ) . ' )</span>';
				}
				if ( 'time' === ( $custom_fields[ $key ]['type'] ?? '' ) && ! empty( $field['value'] ) ) {
					$field['value'] = $field['value'] . '<br/><span style="font-size: 13px;">( ' . esc_html__( '24-Hour Time Format', 'shopbuilder-pro' ) . ' )</span>';
				}
				$thankYouData[ $key ] = $field;
			}
		}
		return $thankYouData;
	}
}
