<?php
/**
 * Customize My Account Module Class.
 *
 * @package RadiusTheme\SB
 */

namespace RadiusTheme\SBPRO\Modules\BulkDiscount;

use RadiusTheme\SBPRO\Traits\SingletonTrait;
use RadiusTheme\SB\Helpers\Fns;

defined( 'ABSPATH' ) || exit();

/**
 * Back-order Module Class.
 */
class BulkDiscountApply {
	/**
	 * Singleton Trait.
	 */
	use SingletonTrait;

	/**
	 * @var array
	 */
	protected $calculate_fees = [];
	/**
	 * @var array
	 */
	protected $cart_discount_available = [];
	/**
	 * Constructor method to initialize the class and hook into WooCommerce.
	 */
	private function __construct() {
		add_action( 'woocommerce_before_calculate_totals', [ $this, 'custom_discounted_cart_item_price' ], 99, 1 );
		add_filter( 'woocommerce_cart_item_name', [ $this, 'append_percentage_to_item_name' ], 20, 2 );
		add_action( 'woocommerce_cart_calculate_fees', [ $this, 'cart_calculate_fees' ] );
		add_action( 'woocommerce_before_cart', [ $this, 'add_cart_notice' ] );
		// Product Price Html.
		add_filter( 'woocommerce_loop_add_to_cart_link', [ $this, 'bulk_discount_note_in_product_loop' ], 11, 2 );
		add_action( 'woocommerce_before_add_to_cart_form', [ $this, 'single_bulk_discount_note' ], 11 );
		// Custom Widget Support.
		add_action( 'rtsb/after/shop/product/title', [ $this, 'custom_layout_bulk_discount_note' ], 11 );
		// Calculate Count Down Campaign.
		add_filter( 'rtsb/calculate/countdown/campaign', [ $this, 'is_calculate_price_for_module' ], 11, 2 );
		add_filter( 'rtsb/calculate/pre_order/price', [ $this, 'is_calculate_price_for_module' ], 11, 2 );
		// Bulk Discount Order Page.
		add_action( 'woocommerce_checkout_create_order_line_item', [ $this, 'add_volume_discount_to_order_items' ], 10, 3 );
		add_filter( 'woocommerce_order_item_get_formatted_meta_data', [ $this, 'format_volume_discount_meta' ], 10, 2 );
		add_filter( 'woocommerce_hidden_order_itemmeta', [ $this, 'hide_admin_meta' ], 10, 1 );
	}
	/**
	 * Hides the volume discount metadata from the admin order details view.
	 *
	 * This prevents the `_rtsb_bulk_discount_volume` meta key from being displayed
	 * in the order details section for better UI clarity.
	 *
	 * @param array $hidden_meta_data List of meta keys to be hidden.
	 * @return array Updated list of hidden meta keys.
	 */
	public function hide_admin_meta( $hidden_meta_data ) {
		$hidden_meta_data[] = '_rtsb_bulk_discount_volume';
		return $hidden_meta_data;
	}

	/**
	 * Formats and adds the volume discount metadata to order line items.
	 *
	 * Retrieves the discount applied per item and adds it to the order metadata
	 * in a formatted structure for display.
	 *
	 * @param array  $meta_data Existing meta data for the line item.
	 * @param object $item      The order line item object.
	 * @return array Updated meta data with volume discount information.
	 */
	public function format_volume_discount_meta( $meta_data, $item ) {
		$volume_discount = $item->get_meta( '_rtsb_bulk_discount_volume' );
		if ( $volume_discount ) {
			$meta_data[] = (object) [
				'key'           => esc_html__( 'Discount Applied', 'shopbuilder-pro' ),
				'value'         => '',
				'display_key'   => esc_html__( 'Bulk Discount', 'shopbuilder-pro' ),
				'display_value' => wp_strip_all_tags( $volume_discount ),
			];
		}
		return $meta_data;
	}

	/**
	 * Adds the volume discount metadata to order line items.
	 *
	 * Calculates the discount for the cart item and stores it as meta data
	 * in the order line item to track applied bulk discounts.
	 *
	 * @param object $item          The order line item object.
	 * @param string $cart_item_key The unique cart item key.
	 * @param array  $values        The cart item data containing pricing and discount details.
	 * @return void
	 */
	public function add_volume_discount_to_order_items( $item, $cart_item_key, $values ) {
		$discount = new CartDiscountCalculator( $values );
		$amount   = $discount->calculate_discount();
		if ( ! empty( $amount['discount'] ) ) {
			$item->add_meta_data( '_rtsb_bulk_discount_volume', $amount['discount'], true );
		}
	}

	/**
	 * Determines if a countdown campaign should be applied to a product.
	 *
	 * @param mixed      $is      The current state or flag determining campaign applicability.
	 * @param WC_Product $product The WooCommerce product object.
	 *
	 * @return mixed Returns false if the product has a discount price; otherwise, returns the original state.
	 */
	public function is_calculate_price_for_module( $is, $product ) {
		$discount = BulkDiscountFns::product_has_bulk_discounts( $product );
		if ( $discount ) {
			return false;
		}
		return $is;
	}

	/**
	 * Displays the bulk discount notice on the single product page.
	 *
	 * This function retrieves the global WooCommerce product and calls
	 * `bulk_discount_note()` to generate and echo the discount message.
	 *
	 * @return string
	 */
	public function single_bulk_discount_note() {
		$bulkDiscount         = BulkDiscountFns::get_options();
		$showDiscount_message = 'on' === ( $bulkDiscount['discount_message_in_product_details'] ?? '' );
		if ( ! $showDiscount_message ) {
			return '';
		}
		global $product;
		$discount   = new ProductsDiscount( $product );
		$price_html = $discount->get_price_html();
		if ( $price_html ) {
			$note = '<div class="rtsb-bulk-availability-notice rtsb-product-details">' . $price_html . '</div>';
			Fns::print_html( $note, true );
		}
	}
	/**
	 * Displays the bulk discount notice on the single product page.
	 *
	 * This function retrieves the global WooCommerce product and calls
	 * `bulk_discount_note()` to generate and echo the discount message.
	 *
	 * @return void
	 */
	public function custom_layout_bulk_discount_note() {
		global $product;
		$note = $this->bulk_discount_note_in_product_loop( '', $product );
		Fns::print_html( $note, true );
	}
	/**
	 * Adds a bulk discount notice to the product button in the loop.
	 *
	 * This function checks if the bulk discount message should be displayed
	 * in the product loop. If enabled, it calculates the discount and appends
	 * the discount notice before returning the button.
	 *
	 * @param string     $button  The original product button HTML.
	 * @param WC_Product $product The WooCommerce product object.
	 * @return string  The modified button HTML with the bulk discount notice (if applicable).
	 */
	public function bulk_discount_note_in_product_loop( $button, $product ) {
		$bulkDiscount         = BulkDiscountFns::get_options();
		$showDiscount_message = 'on' === ( $bulkDiscount['discount_message_in_product_loop'] ?? '' );
		if ( ! $showDiscount_message ) {
			return $button;
		}
		$discount   = new ProductsDiscount( $product );
		$price_html = $discount->get_price_html();
		if ( $price_html ) {
			return '<div class="rtsb-bulk-availability-notice rtsb-product-showcase">' . $price_html . '</div>' . $button;
		}
		return $button;
	}
	/**
	 * @param object $cart cart item.
	 * @return void
	 */
	public function custom_discounted_cart_item_price( $cart ) {
		if ( is_admin() && ! defined( 'DOING_AJAX' ) ) {
			return;
		}
		if ( did_action( 'woocommerce_before_calculate_totals' ) >= 2 ) {
			return;
		}
		$discount_applied = false;
		// Loop through cart items.
		foreach ( $cart->get_cart() as $cart_item ) {
			$discount = new CartDiscountCalculator( $cart_item );
			$calc     = $discount->calculate_discount();
			if ( empty( $calc['price'] ) || $calc['price'] <= 0 ) {
				continue;
			}
			$this->cart_discount_available = [ $cart_item['data']->get_id() ];
			// Set cart item discounted price.
			if ( 'fixed-amount' === $calc['type'] ) {
				$this->calculate_fees[] = [
					'label' => $calc['discountLabel'],
					'price' => -$calc['price'],
					'is'    => false,
				];
			} else {
				$regular_price = $cart_item['data']->get_regular_price();
				$cart_item['data']->set_price( $calc['price'] );
				$cart_item['data']->set_regular_price( $regular_price );
			}
			$discount_applied = true;
		}
		if ( $discount_applied ) {
			WC()->session->set( 'bulk_discount_applied_notice', true );
		} else {
			WC()->session->__unset( 'bulk_discount_applied_notice' );
		}
	}
	/**
	 * @param object $cart cart item.
	 * @return void
	 */
	public function cart_calculate_fees( $cart ) {
		if ( empty( $this->cart_discount_available ) ) {
			return;
		}
		$bulkDiscount     = BulkDiscountFns::get_options();
		$isDisabledCoupon = 'on' === ( $bulkDiscount['disable_coupons'] ?? '' );
		if ( ! empty( $this->calculate_fees ) ) {
			foreach ( $this->calculate_fees as $calc ) {
				$cart->add_fee( $calc['label'], $calc['price'], $calc['is'] );
			}
		}
		if ( $isDisabledCoupon ) {
			$cart->remove_coupons();
		}
	}
	/**
	 * @param $product_name
	 * @param $cart_item
	 * @return string
	 */
	public function append_percentage_to_item_name( $product_name, $cart_item ) {
		$discount = new CartDiscountCalculator( $cart_item );
		$amount   = $discount->calculate_discount();
		if ( ! empty( $amount['discount'] ) ) {
			$product_name .= ' <div class="rtsb-dicount-summery"> ' . $amount['discount'] . '</div>';
		}
		return $product_name;
	}

	/**
	 * @return void
	 */
	public function add_cart_notice() {
		$bulkDiscount      = BulkDiscountFns::get_options();
		$enableCartMessage = 'on' === ( $bulkDiscount['enable_cart_message'] ?? '' );
		$show_notice       = WC()->session->get( 'bulk_discount_applied_notice' );
		if ( $enableCartMessage && $show_notice ) {
			$message = $bulkDiscount['cart_message'] ?? __( 'Discount Applied', 'shopbuilder-pro' );
			wc_print_notice(
				sprintf(
					'<span class="discount-notice">%s</span>',
					$message
				),
				'success'
			);
			WC()->session->__unset( 'bulk_discount_applied_notice' );
		}
	}
}
