<?php
/**
 * Customize My Account Module Class.
 *
 * @package RadiusTheme\SB
 */

namespace RadiusTheme\SBPRO\Modules\BulkDiscount;

use RadiusTheme\SB\Helpers\Fns;
use RadiusTheme\SBPRO\Helpers\AdminInputsFns;
use RadiusTheme\SBPRO\Traits\SingletonTrait;

defined( 'ABSPATH' ) || exit();

/**
 * Back-order Module Class.
 */
class BulkDiscountAdmin {
	/**
	 * Singleton Trait.
	 */
	use SingletonTrait;

	/**
	 * Module Class Constructor.
	 */
	private function __construct() {
		add_filter( 'woocommerce_product_data_tabs', [ $this, 'add_product_bulk_discount_tab' ] );
		add_action( 'woocommerce_product_data_panels', [ $this, 'render_meta_boxes' ] );
		add_action( 'woocommerce_process_product_meta', [ $this, 'save_bulk_discount_meta' ], 10, 2 );
	}
	/**
	 * Sanitize an array of strings.
	 *
	 * @param string $meta_key Meta key.
	 *
	 * @return array
	 */
	public function sanitize_array( $meta_key ) {
		return isset( $_REQUEST[ $meta_key ] ) && is_array( $_REQUEST[ $meta_key ] ) ? array_map( 'sanitize_text_field', wp_unslash( $_REQUEST[ $meta_key ] ) ) : []; // phpcs:ignore WordPress.Security.NonceVerification.Recommended
	}
	/**
	 * Saves bulk discount meta fields for a product.
	 *
	 * This function handles saving bulk discount enable/disable options
	 * when a product is updated in the admin panel.
	 *
	 * @param int      $post_id The ID of the product being saved.
	 * @param \WP_Post $post    The post object for the product.
	 * @return void
	 */
	public function save_bulk_discount_meta( $post_id, $post ) {
		// Verify nonce.
		check_admin_referer( rtsb()->nonceText, rtsb()->nonceId );

		$disable_discount = isset( $_REQUEST['_rtsb_disable_bulk_discount'] ) ? sanitize_text_field( wp_unslash( $_REQUEST['_rtsb_disable_bulk_discount'] ) ) : null;
		$enable_discount  = isset( $_REQUEST['_rtsb_enable_bulk_discount'] ) ? sanitize_text_field( wp_unslash( $_REQUEST['_rtsb_enable_bulk_discount'] ) ) : null;

		$discount_type = isset( $_REQUEST['_rtsb_enable_bulk_discount_type'] ) ? sanitize_text_field( wp_unslash( $_REQUEST['_rtsb_enable_bulk_discount_type'] ) ) : null;
		$apply_on      = isset( $_REQUEST['_discount_apply_on'] ) ? sanitize_text_field( wp_unslash( $_REQUEST['_discount_apply_on'] ) ) : null;
		$condition     = isset( $_REQUEST['_rtsb_bulk_discount_condition'] ) ? sanitize_text_field( wp_unslash( $_REQUEST['_rtsb_bulk_discount_condition'] ) ) : null;
		$roles         = $this->sanitize_array( '_rtsb_user_roles' );
		$start_date    = isset( $_REQUEST['_rtsb_discount_start_date_and_time'] ) ? sanitize_text_field( wp_unslash( $_REQUEST['_rtsb_discount_start_date_and_time'] ) ) : null;
		$end_date      = isset( $_REQUEST['_rtsb_discount_end_date_and_time'] ) ? sanitize_text_field( wp_unslash( $_REQUEST['_rtsb_discount_end_date_and_time'] ) ) : null;

		// Disable Balk Discount.
		if ( ! empty( $disable_discount ) && 'on' === $disable_discount ) {
			update_post_meta( $post_id, '_rtsb_disable_bulk_discount', $disable_discount );
		} else {
			delete_post_meta( $post_id, '_rtsb_disable_bulk_discount' );
		}
		// Enable Balk Discount.
		if ( ! empty( $enable_discount ) && 'on' === $enable_discount ) {
			update_post_meta( $post_id, '_rtsb_enable_bulk_discount', $enable_discount );
		} else {
			delete_post_meta( $post_id, '_rtsb_enable_bulk_discount' );
		}

		// Discount Type.
		update_post_meta( $post_id, '_rtsb_enable_bulk_discount_type', $discount_type );
		update_post_meta( $post_id, '_rtsb_bulk_discount_condition', $condition );
		update_post_meta( $post_id, '_rtsb_user_roles', $roles );
		update_post_meta( $post_id, '_rtsb_discount_start_date_and_time', $start_date );
		update_post_meta( $post_id, '_rtsb_discount_end_date_and_time', $end_date );
		update_post_meta( $post_id, '_discount_apply_on', $apply_on );
	}

	/**
	 * Adds the Product Bulk Discount tab to the product data tabs.
	 *
	 * @param array $tabs Existing tabs.
	 *
	 * @return array Modified tabs.
	 */
	public function add_product_bulk_discount_tab( $tabs ) {
		$tabs['rtsb_product_bulk_discount'] = [
			'label'    => esc_html__( 'Bulk Discount', 'shopbuilder-pro' ),
			'target'   => 'rtsb_product_bulk_discount',
			'class'    => [ 'show_if_simple', 'show_if_variable' ],
			'priority' => 65,
		];

		return $tabs;
	}

	/**
	 * Renders meta-boxes to the WooCommerce product data panels.
	 *
	 * @return void
	 */
	public function render_meta_boxes() {
		global $post;
		$disable_bulk_discount = get_post_meta( $post->ID, '_rtsb_disable_bulk_discount', true );
		$enable_bulk_discount  = get_post_meta( $post->ID, '_rtsb_enable_bulk_discount', true );
		$type                  = get_post_meta( $post->ID, '_rtsb_enable_bulk_discount_type', true );
		$condition             = get_post_meta( $post->ID, '_rtsb_bulk_discount_condition', true );
		$user_roles            = get_post_meta( $post->ID, '_rtsb_user_roles', true );
		$start_date            = get_post_meta( $post->ID, '_rtsb_discount_start_date_and_time', true );
		$end_date              = get_post_meta( $post->ID, '_rtsb_discount_end_date_and_time', true );
		$apply_on              = get_post_meta( $post->ID, '_discount_apply_on', true );
		$disable_checked       = 'on' === $disable_bulk_discount ? 'on' : 'off';
		$enable_checked        = 'on' === $enable_bulk_discount ? 'on' : 'off';
		$enable_visibility     = 'on' === $enable_bulk_discount ? ' hidden' : '';
		?>
		<div id='rtsb_product_bulk_discount' class='panel woocommerce_options_panel rtsb_panel hidden'>
			<div class="rtsb-group-heading">
				<h2><?php esc_html_e( 'Bulk Discount Settings', 'shopbuilder-pro' ); ?></h2>
			</div>
			<div id="rtsb_product_bulk_discount_options_group" class="options_group">
				<?php wp_nonce_field( rtsb()->nonceText, rtsb()->nonceId ); ?>
				<div class="bulk-discount-group-control">
					<div id="rtsb_disable_bulk_discount_group">
						<?php
						Fns::print_html(
							AdminInputsFns::generate_switch_input(
								[
									'id'            => '_rtsb_disable_bulk_discount',
									'label'         => esc_html__( 'Disable Global Rules?', 'shopbuilder-pro' ),
									'desc_tip'      => true,
									'aria_label'    => esc_html__( 'Disable Global Product Bulk discount', 'shopbuilder-pro' ),
									'description'   => esc_html__( 'Switch on to disable the ShopBuilder global Bulk Discount rules for this product.', 'shopbuilder-pro' ),
									'checked'       => esc_attr( $disable_checked ),
									'value'         => 'on',
									'input_class'   => 'switch-input',
									'wrapper_class' => 'form-field disable-rules',
								]
							),
							true
						);
						?>
					</div>
					<div id="rtsb_enable_bulk_discount_group" class="<?php echo esc_attr( $enable_visibility ); ?>">
						<?php
						Fns::print_html(
							AdminInputsFns::generate_switch_input(
								[
									'id'            => '_rtsb_enable_bulk_discount',
									'label'         => esc_html__( 'Override Global Rules?', 'shopbuilder-pro' ),
									'desc_tip'      => true,
									'aria_label'    => esc_html__( 'Override Global Product Bulk Discount', 'shopbuilder-pro' ),
									'description'   => esc_html__( 'Switch on to override the ShopBuilder global bulk discount rules.', 'shopbuilder-pro' ),
									'checked'       => esc_attr( $enable_checked ),
									'value'         => 'on',
									'input_class'   => 'switch-input',
									'wrapper_class' => 'form-field ',
								]
							),
							true
						);
						?>
					</div>
				</div>
				<div id="rtsb_bulk_discount_groups" class="rtsb-bulk-discount-fields<?php echo esc_attr( $enable_visibility ); ?>">
					<?php
					$fields  = '';
					$fields .= AdminInputsFns::generate_select_input(
						[
							'id'            => '_rtsb_enable_bulk_discount_type',
							'label'         => esc_html__( 'Discount Type', 'shopbuilder-pro' ),
							'desc_tip'      => true,
							'aria_label'    => esc_html__( 'Bulk discount type', 'shopbuilder-pro' ),
							'description'   => esc_html__( 'Select your desired discount type.', 'shopbuilder-pro' ),
							'value'         => $type ?? '',
							'input_class'   => 'short',
							'label_class'   => 'field-label',
							'options'       => [
								'percentage'   => esc_html__( 'Percentage (%)', 'shopbuilder-pro' ),
								'fixed'        => esc_html__( 'Fixed Discount - Per Item', 'shopbuilder-pro' ),
								'fixed-amount' => esc_html__( 'Fixed Amount Discount on Cart', 'shopbuilder-pro' ),
							],
							'wrapper_class' => 'form-field input-type ',
						]
					);
					$fields .= AdminInputsFns::generate_select_input(
						[
							'id'            => '_discount_apply_on',
							'label'         => esc_html__( 'Discount Applies On', 'shopbuilder-pro' ),
							'desc_tip'      => true,
							'aria_label'    => esc_html__( 'Bulk discount apply on', 'shopbuilder-pro' ),
							'description'   => esc_html__( 'Bulk discount will apply base on selected price.', 'shopbuilder-pro' ),
							'value'         => $apply_on ?? 'regular-price',
							'input_class'   => 'short',
							'label_class'   => 'field-label',
							'options'       => [
								'regular-price' => esc_html__( 'Regular Price', 'shopbuilder-pro' ),
								'sale-price'    => esc_html__( 'Sale Price ( If Available )', 'shopbuilder-pro' ),
							],
							'wrapper_class' => 'form-field input-type ',
						]
					);
					$fields .= AdminInputsFns::generate_textarea_input(
						[
							'id'            => '_rtsb_bulk_discount_condition',
							'label'         => esc_html__( 'Bulk Discount Condition', 'shopbuilder-pro' ),
							'desc_tip'      => true,
							'aria_label'    => esc_html__( 'Product Bulk Discount field description', 'shopbuilder-pro' ),
							'description'   => esc_html__( 'Enter \'Quantity:Discount\' pairs separated by | (pipe), e.g., 10:5 for 5 (Percentage,Fixed base on discount type) off above 10 items.', 'shopbuilder-pro' ),
							'placeholder'   => esc_html__( 'Example: 10:5|20:10|30:15 (Quantity:Discount).', 'shopbuilder-pro' ),
							'value'         => $condition ?? '',
							'input_class'   => 'short',
							'label_class'   => 'field-label',
							'wrapper_class' => 'form-field input-description ',
						]
					);
					$fields .= AdminInputsFns::generate_select_input(
						[
							'id'            => '_rtsb_user_roles',
							'label'         => esc_html__( 'User Roles', 'shopbuilder-pro' ),
							'desc_tip'      => true,
							'aria_label'    => esc_html__( 'User Roles', 'shopbuilder-pro' ),
							'description'   => esc_html__( 'Select desired user roles type.', 'shopbuilder-pro' ),
							'value'         => ! empty( $user_roles ) ? $user_roles : [],
							'input_class'   => 'rtsb-select2-input wc-enhanced-select',
							'label_class'   => 'field-label',
							'options'       => Fns::get_all_user_roles(),
							'wrapper_class' => 'form-field input-type ',
							'multiple'      => true,
						]
					);
					$fields .= AdminInputsFns::generate_date_picker_input(
						$post->ID,
						[
							'id'            => '_rtsb_discount_start_date_and_time',
							'label'         => esc_html__( 'Discount Start Date', 'shopbuilder-pro' ),
							'desc_tip'      => true,
							'value'         => $start_date ?? '',
							'class'         => 'short',
							'description'   => esc_html__( 'Enter the estimated date when bulk discount will be available. ', 'shopbuilder-pro' ),
							'wrapper_class' => '',
						],
						true,
						false
					);
					$fields .= AdminInputsFns::generate_date_picker_input(
						$post->ID,
						[
							'id'            => '_rtsb_discount_end_date_and_time',
							'label'         => esc_html__( 'Discount End Date', 'shopbuilder-pro' ),
							'desc_tip'      => true,
							'value'         => $end_date ?? '',
							'class'         => 'short',
							'description'   => esc_html__( 'Enter the estimated date when bulk discount will end. ', 'shopbuilder-pro' ),
							'wrapper_class' => '',
						],
						true,
						false
					);
					Fns::print_html( $fields, true )
					?>
				</div>

			</div>
		</div>
		<?php
	}
}
