<?php
/**
 * Customize My Account Module Class.
 *
 * @package RadiusTheme\SB
 */

namespace RadiusTheme\SBPRO\Modules\BogoBxGy\Discount;

use RadiusTheme\SBPRO\Models\DiscountCalculatorBase;
use RadiusTheme\SBPRO\Modules\BogoBxGy\BogoGiftBxGyFns;

defined( 'ABSPATH' ) || exit();

/**
 * Back-order Module Class.
 */
class BxGyDiscount extends DiscountCalculatorBase {

	/**
	 * Constructor for the class.
	 *
	 * @param \WC_Product $product   The product object.
	 * @param array       $rule      The rule data (optional).
	 * @param array       $cart_item The cart item data (optional).
	 */
	public function __construct( $product, $rule = [], $cart_item = [] ) {
		parent::__construct( $product, $rule, $cart_item );
	}
	/**
	 * Retrieves the discount message.
	 *
	 * @return array
	 */
	public function discount_message() {
		if ( $this->is_discount_applicable() ) {
			$type                 = $this->discountRule['option']['discount_type'] ?? 'bogo';
			$qty                  = $this->discountRule['required_quantity'] ?? 1;
			$product              = $this->discountRule['discount_product'] ?? [];
			$dis_quantity         = $this->discountRule['discount_product_quantity'] ?? 1;
			$discount_calculation = $this->discountRule['discount_calculation'] ?? 'fixed';
			$amount               = $this->discountRule['bxgy_amount'] ?? '';
			$discount_product     = reset( $product );
			$product_name         = sprintf(
				'<a href="%s">%s</a>',
				get_permalink( $discount_product ),
				get_the_title( $discount_product )
			);
			 $message             = '';
			if ( 'fixed' === $discount_calculation ) {
				$amount  = wp_strip_all_tags( wc_price( $amount ) );
				$message = sprintf(
				/* translators: %1$s is the quantity to buy, %2$s is the discounted quantity, %3$s is the product name, %4$s is the discount percentage */
					__( 'Buy %1$s, Get %2$s %3$s at %4$s off!', 'shopbuilder-pro' ),
					'<b>' . $qty . '</b>',
					'<b>' . $dis_quantity . '</b>',
					'<b>' . $product_name . '</b>',
					'<b>' . $amount . '</b>',
				);
			} elseif ( 'percentage' === $discount_calculation ) {
				$amount  = $amount . '%';
				$message = sprintf(
				/* translators: %1$s is the quantity to buy, %2$s is the discounted quantity, %3$s is the product name, %4$s is the discount percentage */
					__( 'Buy %1$s, Get %2$s %3$s at %4$s discount!', 'shopbuilder-pro' ),
					'<b>' . $qty . '</b>',
					'<b>' . $dis_quantity . '</b>',
					'<b>' . $product_name . '</b>',
					'<b>' . $amount . '</b>',
				);
			} elseif ( 'full' === $discount_calculation ) {
				$amount  = '100%';
				$message = sprintf(
				/* translators: %1$s is the quantity to buy, %2$s is the discounted quantity, %3$s is the product name, %4$s is the discount percentage */
					__( 'Buy %1$s, Get %2$s %3$s Free!', 'shopbuilder-pro' ),
					'<b>' . $qty . '</b>',
					'<b>' . $dis_quantity . '</b>',
					'<b>' . $product_name . '</b>',
					'<b>' . $amount . '</b>',
				);
			}
			$note = sprintf(
			/* translators: %s is the quantity */
				_n(
					'Buy %s item and get a discount', // Singular.
					'Buy %s items and get a discount', // Plural.
					$qty,
					'shopbuilder-pro'
				),
				$qty
			);
			return [
				'note'             => apply_filters( 'rtsb/bxgy/discount/note', $note ),
				'discount_type'    => $type,
				'discount_message' => apply_filters( 'rtsb/bxgy/discount/message', $message, $this->discountRule ),
			];
		}
		return [];
	}
	/**
	 * Get Product Id
	 */
	public function apply_discount() {
		$discounted_product_ids = [];
		if ( ! empty( $this->discountRule['discount_product'] ) ) {
			$discounted_product_ids = $this->discountRule['discount_product'];
		}
		$this->add_free_gifts_to_cart( $discounted_product_ids );
	}
	/**
	 * Add free products to the cart when a coupon is applied.
	 *
	 * @param array $free_product_ids The coupon code being applied.
	 *
	 * @return void
	 */
	public function add_free_gifts_to_cart( $discounted_product_ids ) {
		$discounted_product_qty = $this->discountRule['discount_product_quantity'] ?? 1;
		if ( ! empty( $this->discountRule['bxgy_multiple_apply'] ) ) {
			$miltiapply              = absint( $this->cart_item['quantity'] / $this->discountRule['required_quantity'] );
			$discounted_product_qty *= $miltiapply;
		}
		$cart = WC()->cart;
		foreach ( $discounted_product_ids as $product_id ) {
			$product                = wc_get_product( $product_id );
			$available_qty          = $product->get_stock_quantity();
			$discounted_product_qty = $available_qty ? min( $discounted_product_qty, $available_qty ) : $discounted_product_qty;
			$cart_item_data         = [
				BogoGiftBxGyFns::DISCOUNTED_CART_KEY => $this->product->get_id(),
				'rule'                               => 'bxgy',
				'discount_type'                      => $this->discountRule['discount_calculation'] ?? 'full',
				'amount'                             => $this->discountRule['bxgy_amount'] ?? 0,
				'required_minimum'                   => $this->discountRule['required_quantity'],
			];
			$product_cart_id        = $cart->generate_cart_id( $product_id, 0, [], $cart_item_data );
			// Check if the free/gift is already in the cart.
			$cart_item_key = $cart->find_product_in_cart( $product_cart_id );

			$discount_applied = WC()->session->get( 'rtsb_discount_applied_session' );
			if ( empty( $discount_applied ) ) {
				$discount_applied = [];
			}
			$discount_applied[] = $product_id;
			WC()->session->set( 'rtsb_discount_applied_session', array_unique( $discount_applied ) );

			if ( $cart_item_key ) {
				// If already in the cart, update the quantity.
				if ( $this->cart_item['key'] !== $cart_item_key ) {
					$cart->set_quantity( $cart_item_key, $discounted_product_qty );
				}
			} else {
				// Otherwise, add it as a new item.
				$cart->add_to_cart( $product_id, $discounted_product_qty, 0, [], $cart_item_data );
			}
		}
	}
}
