<?php
/**
 * Product AddOns Single Product Page Class.
 *
 * @package RadiusTheme\SBPRO
 */

namespace RadiusTheme\SBPRO\Modules\AddOns\Frontend;

use Exception;
use RadiusTheme\SBPRO\Modules\PartialPay\PartialPayFns;
use WC_Product;
use WC_Product_Variation;
use RadiusTheme\SB\Helpers\Fns;
use RadiusTheme\SB\Helpers\Cache;
use RadiusTheme\SBPRO\Helpers\FnsPro;
use RadiusTheme\SBPRO\Traits\SingletonTrait;
use RadiusTheme\SBPRO\Modules\AddOns\AddOnsFns;
use RadiusTheme\SB\Elementor\Helper\RenderHelpers;
use RadiusTheme\SBPRO\Modules\PreOrder\PreOrderFns;
use RadiusTheme\SBPRO\Modules\FlashSaleCountdown\CountdownFns;

defined( 'ABSPATH' ) || exit();

/**
 * Product AddOns Single Product Page Class.
 */
class Product {
	/**
	 * Singleton Trait.
	 */
	use SingletonTrait;

	/**
	 * Options.
	 *
	 * @var array|mixed
	 */
	private array $options;

	/**
	 * User restriction.
	 *
	 * @var bool
	 */
	private bool $restriction = false;

	/**
	 * Module Class Constructor.
	 */
	private function __construct() {
		/**
		 * Get the options.
		 */
		$this->options = AddOnsFns::get_add_ons_settings_data();

		$has_user_restriction = 'specific' === ( $this->options['user_role_management'] ?? null );
		$user_roles           = $this->options['roles'];

		if ( $has_user_restriction && ( ! FnsPro::is_visible_for_user( $user_roles ) ) ) {
			$this->restriction = true;
		}

		/**
		 * Hooks.
		 */
		$this->init_hooks();
	}

	/**
	 * Init hooks.
	 *
	 * @return void
	 */
	private function init_hooks() {
		if ( $this->restriction ) {
			return;
		}

		$position = $this->options['position'] ?? 'before';
		$priority = $this->options['hook_priority'] ?? 10;

		/**
		 * Actions.
		 */
		add_action( "woocommerce_{$position}_add_to_cart_button", [ $this, 'render_product_addons' ], $priority );

		// Product AddOns image upload.
		add_action( 'wp_ajax_rtsb_addons_upload_image', [ $this, 'upload_image' ] );
		add_action( 'wp_ajax_nopriv_rtsb_addons_upload_image', [ $this, 'upload_image' ] );
		add_action( 'wp_ajax_rtsb_addons_delete_image', [ $this, 'delete_image' ] );
		add_action( 'wp_ajax_nopriv_rtsb_addons_delete_image', [ $this, 'delete_image' ] );

		/**
		 * Filters.
		 */
		add_filter( 'woocommerce_product_add_to_cart_text', [ $this, 'shop_button_label' ], 99, 2 );
		add_filter( 'rtsb/elementor/render/add_to_cart_text', [ $this, 'shop_button_label' ], 99, 2 );
		add_filter( 'rtsb/elementor/render/cart_attributes', [ $this, 'change_add_to_cart_args' ], 10, 2 );
		add_filter( 'woocommerce_loop_add_to_cart_args', [ $this, 'change_add_to_cart_args' ], 10, 2 );
		add_filter( 'woocommerce_product_add_to_cart_url', [ $this, 'modify_add_to_cart_url' ], 10, 2 );
		add_filter( 'woocommerce_available_variation', [ $this, 'addons_variation_data' ] );
		add_filter( 'woocommerce_post_class', [ $this, 'add_custom_class' ], 10, 2 );
		add_filter( 'woocommerce_get_price_html', [ $this, 'custom_product_price_display' ], 10, 2 );
		add_filter( 'woocommerce_available_variation', [ $this, 'custom_variation_price_display' ], 10, 3 );

		// Disable features.
		add_filter( 'rtsb/module/bulk_discounts/global/disable', [ $this, 'disable_feature' ], 99, 2 );
		add_filter( 'rtsb/module/partial_pay/global/disable', [ $this, 'disable_feature' ], 99, 2 );
	}

	/**
	 * Checks if bulk discounts apply to the product.
	 *
	 * @param mixed      $is        The current state or flag determining discount applicability.
	 * @param WC_Product $product   The WooCommerce product object.
	 *
	 * @return bool
	 */
	public function disable_feature( $is, $product ) {
		if ( ! is_a( $product, 'WC_Product' ) ) {
			return $is;
		}

		if ( is_a( $product, 'WC_Product_Variation' ) ) {
			$parent_id = $product->get_parent_id();

			if ( $parent_id ) {
				$product = wc_get_product( $parent_id );
			}
		}

		if ( AddOnsFns::product_has_addons( $product, 'variation' === $product->get_type() ) ) {
			return true;
		}

		return $is;
	}

	/**
	 * Display Product addon in product details page.
	 *
	 * @return void
	 */
	public function render_product_addons() {
		$product    = self::get_product()['product'];
		$product_id = self::get_product()['product_id'];

		if ( ! $product instanceof WC_Product ) {
			return;
		}

		if ( ! $this->is_supported_product_type( $product ) ) {
			return;
		}

		$cache_key     = 'rtsb_product_addons_' . $product_id;
		$cached_addons = wp_cache_get( $cache_key, 'shopbuilder' );

		if ( false === $cached_addons ) {
			$product_addons    = AddOnsFns::get_addon_groups( $product_id );
			$update_base_price = 'on' === AddOnsFns::get_add_ons_settings_data()['update_base_price'];

			$cached_addons = [
				'product_addons'    => $product_addons,
				'update_base_price' => $update_base_price,
			];

			wp_cache_set( $cache_key, $cached_addons, 'shopbuilder', 12 * HOUR_IN_SECONDS );
			Cache::set_data_cache_key( $cache_key );
		} else {
			$product_addons    = $cached_addons['product_addons'];
			$update_base_price = $cached_addons['update_base_price'];
		}

		if ( $this->has_addons( $product_addons ) ) {
			?>
			<div class="rtsb-addon-groups<?php echo esc_attr( $update_base_price ? ' need-update-base-price' : '' ); ?>">
				<?php
				$counter = 0;
				$this->render_addons_wrapper_start( $product );

				foreach ( $product_addons as $addon ) {
					$this->render_addon_group( $addon, $counter );

					$counter++;
				}

				$this->render_addons_wrapper_end( $product );
				$this->enqueue_addons_scripts();
				?>
			</div>
			<?php
			$this->render_price_box( $product, $product_id );
		}
	}

	/**
	 * Uploads an image.
	 *
	 * @return void
	 */
	public function upload_image() {
		if ( ! Fns::verify_nonce() ) {
			wp_send_json_error( [ 'message' => esc_html__( 'Security check failed.', 'shopbuilder-pro' ) ] );
		}

		// phpcs:ignore WordPress.Security.NonceVerification.Missing
		if ( empty( $_FILES['file'] ) ) {
			wp_send_json_error( [ 'message' => esc_html__( 'No file uploaded.', 'shopbuilder-pro' ) ] );
		}

		$allowed_extensions = AddOnsFns::get_add_ons_settings_data()['allowed_extensions'];
		$uploaded_file      = $_FILES['file'] ?? null; // phpcs:ignore WordPress.Security.NonceVerification.Missing,WordPress.Security.ValidatedSanitizedInput.InputNotSanitized
		$file_name          = $uploaded_file['name'] ?? '';
		$file_ext           = strtolower( pathinfo( $file_name, PATHINFO_EXTENSION ) );

		if ( ! in_array( $file_ext, $allowed_extensions, true ) ) {
			wp_send_json_error(
				[
					'message' => esc_html__( 'Invalid file type. You can only upload files with the following extensions: ', 'shopbuilder-pro' ) . implode( ', ', $allowed_extensions ) . esc_html__( '.', 'shopbuilder-pro' ),
				]
			);
		}

		if ( ! function_exists( 'wp_handle_upload' ) ) {
			require_once ABSPATH . 'wp-admin/includes/file.php';
		}

		add_filter( 'upload_dir', [ $this, 'change_upload_dir' ] );

		$upload_overrides = [ 'test_form' => false ];
		$move_file        = wp_handle_upload( $uploaded_file, $upload_overrides );

		remove_filter( 'upload_dir', [ $this, 'change_upload_dir' ] );

		if ( $move_file && ! isset( $move_file['error'] ) ) {
			wp_send_json_success(
				[
					'name' => $file_name ,
					'url'  => $move_file['url'],
				]
			);
		} else {
			wp_send_json_error( [ 'message' => $move_file['error'] ] );
		}
	}

	/**
	 * Deletes an image.
	 *
	 * @return void
	 */
	public function delete_image() {
		if ( ! Fns::verify_nonce() ) {
			wp_send_json_error();
		}

		// phpcs:ignore WordPress.Security.NonceVerification.Missing
		$file_url = isset( $_POST['file_url'] ) ? esc_url_raw( wp_unslash( $_POST['file_url'] ) ) : '';

		if ( ! $file_url ) {
			wp_send_json_error();
		}

		$upload_dir      = wp_upload_dir();
		$upload_base_url = trailingslashit( $upload_dir['baseurl'] ) . 'shopbuilder_uploads';
		$upload_base_dir = trailingslashit( $upload_dir['basedir'] ) . 'shopbuilder_uploads';

		if ( strpos( $file_url, $upload_base_url ) === false ) {
			wp_send_json_error();
		}

		$relative_path = str_replace( $upload_base_url, '', $file_url );
		$file_path     = $upload_base_dir . $relative_path;

		if ( file_exists( $file_path ) ) {
			wp_delete_file( $file_path );
		}

		wp_send_json_success();
	}

	/**
	 * Override the default upload-path.
	 *
	 * @param   array $dir The default upload-path.
	 * @return  array
	 */
	public function change_upload_dir( $dir ) {
        // phpcs:ignore WordPress.Security.NonceVerification.Missing
		$action = ! empty( $_POST['action'] ) ? sanitize_text_field( wp_unslash( $_POST['action'] ) ) : '';

		if ( 'rtsb_addons_upload_image' !== $action ) {
			return $dir;
		}

		$dir['path'] = trailingslashit( $dir['basedir'] ) . 'shopbuilder_uploads';
		$dir['url']  = trailingslashit( $dir['baseurl'] ) . 'shopbuilder_uploads';

		return $dir;
	}

	/**
	 * Modify the "Add to Cart" button arguments for addon products in the product loop.
	 *
	 * @param array      $args The original arguments for the "Add to Cart" button.
	 * @param WC_Product $product The product object.
	 *
	 * @return array
	 * @throws Exception If there is an error in determining the pre-order status.
	 */
	public function change_add_to_cart_args( $args, $product ) {
		if ( ! AddOnsFns::product_has_addons( $product ) ) {
			return $args;
		}

		$args['href']  = esc_url( get_permalink( $product->get_id() ) );
		$args['class'] = str_replace(
			[ 'ajax_add_to_cart', 'rtsb-mini-cart', 'rtsb-add-to-cart-btn', 'variable-product' ],
			[ '', '', 'rtsb-pre-order-btn', 'variable-product rtsb-pre-order-btn' ],
			$args['class']
		);

		if ( ! empty( $args['title'] ) ) {
			$args['title'] = $this->get_custom_label( $args['title'], $product );
		}

		return $args;
	}

	/**
	 * Modify the shop button text for pre-order products.
	 *
	 * @param string     $button_label The default button label.
	 * @param WC_Product $product The product object.
	 *
	 * @return string
	 * @throws Exception If there is an error in determining the pre-order status.
	 */
	public function shop_button_label( $button_label, $product ) {
		return $this->get_custom_label( $button_label, $product );
	}

	/**
	 * Modify the button text for pre-order products.
	 *
	 * @param string     $button_label The default button label.
	 * @param WC_Product $product The product object.
	 *
	 * @return string
	 * @throws Exception If there is an error in determining the pre-order status.
	 */
	public function get_custom_label( $button_label, $product ) {
		if ( AddOnsFns::product_has_addons( $product ) ) {
			$button_label = esc_html( $this->options['cart_button_label'] );
		}

		return $button_label;
	}

	/**
	 * Modify the "Add to Cart" URL for pre-order products.
	 *
	 * @param string     $permalink The original permalink URL.
	 * @param WC_Product $product The product object.
	 *
	 * @return string
	 */
	public function modify_add_to_cart_url( $permalink, $product ) {
		if ( AddOnsFns::product_has_addons( $product ) ) {
			$permalink = esc_url( get_permalink( $product->get_id() ) );
		}

		return $permalink;
	}

	/**
	 * Adds pre-order data to product variations.
	 *
	 * @param array $variations The variation data.
	 *
	 * @return array
	 */
	public function addons_variation_data( $variations ) {
		if ( AddOnsFns::product_has_addons( wc_get_product( $variations['variation_id'] ), true ) ) {
			$variations['rtsb_product_has_addons'] = 'yes';
			$variations['rtsb_addons_label']       = $this->options['cart_button_label'];
		}

		return $variations;
	}

	/**
	 * Add custom class.
	 *
	 * @param array  $classes Classes.
	 * @param object $product The product object.
	 *
	 * @return array
	 */
	public function add_custom_class( $classes, $product ) {
		if ( AddOnsFns::product_has_addons( $product ) ) {
			$classes[] = 'rtsb-product-has-addons';

			if ( 'on' === AddOnsFns::get_add_ons_settings_data()['show_add_to_cart'] ) {
				$classes[] = 'rtsb-hide-cart-btn';
			}
		}

		return $classes;
	}

	/**
	 * Modify the displayed product price.
	 *
	 * @param string     $price_html The current price HTML.
	 * @param WC_Product $product The WooCommerce product object.
	 *
	 * @return string
	 */
	public function custom_product_price_display( $price_html, $product ) {
		if ( 'variable' === $product->get_type() || ! AddOnsFns::product_has_addons( $product ) ) {
			return $price_html;
		}

		$campaign      = CountdownFns::get_campaign_for_current_product( $product );
		$price         = floatval( $product->get_price() );
		$regular_price = floatval( $product->get_regular_price() );
		$sale_price    = floatval( $product->get_sale_price() );
		$prices        = compact( 'price', 'regular_price', 'sale_price' );

		/**
		 * Pre-Order compatibility.
		 */
		if ( FnsPro::is_module_active( 'pre_order' ) && PreOrderFns::is_on_pre_order( $product ) ) {
			$prices               = AddOnsFns::apply_preorder_price( $prices, $product );
			$prices['sale_price'] = self::convert_currency( $prices['sale_price'] );

			return wc_format_sale_price( $prices['regular_price'], $prices['sale_price'] );
		}

		/**
		 * Flash sale compatibility.
		 */
		if ( FnsPro::is_module_active( 'flash_sale_countdown' ) ) {
			$prices = AddOnsFns::apply_campaign_discount( $prices, $campaign );
			if ( ! empty( $prices['sale_price'] ) ) {
				return wc_format_sale_price( $prices['regular_price'], $prices['sale_price'] );
			}
		}

		return $price_html;
	}

	/**
	 * Modify the variation price HTML for product variations.
	 *
	 * @param array                $variation_data The variation data array.
	 * @param WC_Product_Variation $variation The product variation object.
	 * @param WC_Product           $product The parent product object.
	 *
	 * @return array
	 */
	public function custom_variation_price_display( $variation_data, $variation, $product ) {
		if ( ! AddOnsFns::product_has_addons( $variation ) || empty( $variation_data['price_html'] ) ) {
			return $variation_data;
		}

		$campaign      = CountdownFns::get_campaign_for_current_product( $product );
		$price         = floatval( $product->get_price() );
		$regular_price = floatval( $product->get_regular_price() );
		$sale_price    = floatval( $product->get_sale_price() );
		$prices        = compact( 'price', 'regular_price', 'sale_price' );

		/**
		 * Flash sale compatibility.
		 */
		if ( FnsPro::is_module_active( 'flash_sale_countdown' ) && ! empty( $campaign ) ) {
			$prices                       = AddOnsFns::apply_campaign_discount( $prices, $campaign );
			$variation_data['price_html'] = '<span class="price">' . wc_format_sale_price( $prices['regular_price'], $prices['sale_price'] ) . '</span>';
		}

		/**
		 * Pre-Order compatibility.
		 */
		if ( FnsPro::is_module_active( 'pre_order' ) && ( PreOrderFns::is_on_pre_order( $product ) || PreOrderFns::variation_is_on_pre_order( $product ) ) ) {
			$prices               = AddOnsFns::apply_preorder_price( $prices, $product );
			$prices['sale_price'] = self::convert_currency( $prices['sale_price'] );

			$variation_data['price_html'] = '<span class="price">' . wc_format_sale_price( $prices['regular_price'], $prices['sale_price'] ) . '</span>';
		}

		return $variation_data;
	}

	/**
	 * Render an addon group.
	 *
	 * @param array $addon The addon data.
	 * @param int   $counter The data index.
	 *
	 * @return void
	 */
	private function render_addon_group( $addon, $counter ) {
		$addon = $this->initialize_addon_data( $addon, $counter );
		?>
		<div class="rtsb-addon-group" id="rtsb_addon_group_<?php echo absint( $counter ); ?>">
			<?php
			// Nonce verification.
			wp_nonce_field( rtsb()->nonceText, rtsb()->nonceId );

			// Group header template.
			Fns::load_template( 'add-ons/group-header', $addon, false, '', $this->get_template_path() );

			if ( ! empty( $addon['inputs'] ) ) {
				$this->render_addon_fields( $addon['inputs'], $counter );
			}
			?>
		</div>
		<?php
	}

	/**
	 * Render addon fields.
	 *
	 * @param array $fields The array of field data.
	 * @param int   $group_id The group index.
	 *
	 * @return void
	 */
	private function render_addon_fields( $fields, $group_id ) {
		?>
		<div class="rtsb-addon-fields">
			<?php
			$counter = 0;

			foreach ( $fields as $field ) {
				$field    = $this->initialize_field_data( $field, $counter, $group_id );
				$required = 'on' === $field['required'] ? ' rtsb-required-field' : '';
				?>
				<div class="rtsb-addon-field<?php echo esc_attr( $required ); ?>">
					<?php
					// Field header template.
					Fns::load_template( 'add-ons/field-header', $field, false, '', $this->get_template_path() );

					// Field type template.
					$this->render_field_type_template( $field );
					?>
				</div>
				<?php
				$counter++;
			}
			?>
		</div>
		<?php
	}

	/**
	 * Render the template for the field type.
	 *
	 * @param array $field The field data.
	 *
	 * @return void
	 */
	private function render_field_type_template( $field ) {
		$field_types = AddOnsFns::field_types();

		if ( array_key_exists( $field['type'], $field_types ) ) {
			$template_path = 'add-ons/fields/' . $field['type'];

			// Field type template.
			Fns::load_template( $template_path, $field, false, '', $this->get_template_path() );
		} else {
			echo esc_html__( 'Field type not supported.', 'shopbuilder-pro' );
		}
	}

	/**
	 * Renders the product add-ons price box.
	 *
	 * @param WC_Product $product    The WooCommerce product object.
	 * @param int        $product_id The ID of the product.
	 *
	 * @return void
	 */
	private function render_price_box( $product, $product_id ) {
		$currency_position = get_option( 'woocommerce_currency_pos' );
		$visibility        = AddonsFns::get_add_ons_settings_data()['price_total_box'] ?? 'all';
		$total_label       = AddonsFns::get_add_ons_settings_data()['price_box_text'] ?? esc_html__( 'Grand Total', 'shopbuilder-pro' );
		$product_price     = wc_get_price_to_display( $product );
		$prices            = [
			'price'         => $product_price,
			'regular_price' => $product->get_regular_price(),
			'sale_price'    => $product->get_sale_price(),
		];
		$variation_data    = [];

		if ( 'variable' === $product->get_type() ) {
			$min_price         = $product->get_variation_price( 'min', true );
			$min_regular_price = $product->get_variation_regular_price( 'min', true );
			$variation_ids     = $product->get_children();

			foreach ( $variation_ids as $variation_id ) {
				$variation                       = wc_get_product( $variation_id );
				$variation_data[ $variation_id ] = [
					'price'         => floatval( $variation->get_price() ),
					'regular_price' => floatval( $variation->get_regular_price() ),
					'sale_price'    => floatval( $variation->get_sale_price() ),
				];
			}

			$prices = [
				'price'         => $min_price,
				'regular_price' => $min_regular_price,
				'sale_price'    => $product->get_variation_sale_price( 'min', true ),
			];
		}

		/**
		 * Flash sale compatibility.
		 */
		if ( FnsPro::is_module_active( 'flash_sale_countdown' ) ) {
			$campaign       = CountdownFns::get_campaign_for_current_product( $product );
			$prices         = AddOnsFns::apply_campaign_discount( $prices, $campaign );
			$campaign_price = $prices['sale_price'];

			if ( 'variable' === $product->get_type() && ! empty( $campaign ) && ! empty( $variation_data ) ) {
				foreach ( $variation_data as $variation_product_id => $variation_product_data ) {
					$regular_price  = $variation_product_data['regular_price'];
					$discount_price = floatval( CountdownFns::campaign_discount_price( $regular_price, $campaign ) );

					$variation_data[ $variation_product_id ]['sale_price'] = floatval( $regular_price - $discount_price );
				}
			}
		}

		/**
		 * Pre-Order compatibility.
		 */
		if ( FnsPro::is_module_active( 'pre_order' ) && ( PreOrderFns::is_on_pre_order( $product ) || PreOrderFns::variation_is_on_pre_order( $product ) ) ) {
			$prices         = AddOnsFns::apply_preorder_price_with_variations( $prices, $product, $variation_data );
			$preorder_price = self::convert_currency( $prices['preorder_price'] );

			if ( 'variable' === $product->get_type() && ! empty( $variation_data ) ) {
				$variation_data = $prices['variation_data'];
			}
		}

		$template_args = [
			'product_id'        => $product_id,
			'product_price'     => ! empty( $prices['sale_price'] ) ? $prices['sale_price'] : $prices['regular_price'],
			'preorder_price'    => $preorder_price ?? 0,
			'campaign_price'    => $campaign_price ?? 0,
			'product_type'      => $product->get_type(),
			'currency_position' => $currency_position,
			'symbol_html'       => get_woocommerce_currency_symbol(),
			'visibility'        => $visibility,
			'total_label'       => $total_label,
			'variation_data'    => ! empty( $variation_data ) ? wp_json_encode( $variation_data, JSON_UNESCAPED_UNICODE ) : [],
			'classes'           => ! empty( $visibility ) ? ' visibility-' . esc_attr( $visibility ) : '',
		];

		Fns::load_template( 'add-ons/price-box', $template_args, false, '', $this->get_template_path() );
	}

	/**
	 * Check if the product type is supported.
	 *
	 * @param object $product The product object.
	 *
	 * @return bool
	 */
	private function is_supported_product_type( $product ) {
		$supported_types = [ 'simple', 'variable' ];

		return in_array( $product->get_type(), $supported_types, true );
	}

	/**
	 * Check if there are any addons.
	 *
	 * @param array $product_addons The array of product addons.
	 *
	 * @return bool
	 */
	private function has_addons( $product_addons ) {
		return is_array( $product_addons ) && count( $product_addons ) > 0;
	}

	/**
	 * Render the start of the addons wrapper.
	 *
	 * @param object $product The product object.
	 *
	 * @return void
	 */
	private function render_addons_wrapper_start( $product ) {
		do_action( 'rtsb/module/product_add_ons/wrapper_start', $product, $this );
	}

	/**
	 * Render the end of the addons wrapper.
	 *
	 * @param object $product The product object.
	 *
	 * @return void
	 */
	private function render_addons_wrapper_end( $product ) {
		do_action( 'rtsb/module/product_add_ons/wrapper_end', $product, $this );
	}

	/**
	 * Enqueue the necessary scripts for addons.
	 *
	 * @return void
	 */
	private function enqueue_addons_scripts() {
		do_action( 'rtsb/module/product_add_ons/scripts' );
	}

	/**
	 * Initialize addon data with defaults.
	 *
	 * @param array $addon The addon data.
	 * @param int   $counter The data index.
	 *
	 * @return array
	 */
	private function initialize_addon_data( $addon, $counter ) {
		return [
			'name'        => RenderHelpers::get_data( $addon, 'name', '' ),
			'description' => RenderHelpers::get_data( $addon, 'description', '' ),
			'show_name'   => RenderHelpers::get_data( $addon, 'show_name', 'off' ),
			'show_desc'   => RenderHelpers::get_data( $addon, 'show_desc', 'off' ),
			'inputs'      => RenderHelpers::get_data( $addon, 'inputs', [] ),
			'heading_tag' => AddOnsFns::group_heading_tag(),
			'group_id'    => $counter,
		];
	}

	/**
	 * Initialize field data with defaults.
	 *
	 * @param array $field The field data.
	 * @param int   $counter The data index.
	 * @param int   $group_id The group index.
	 *
	 * @return array
	 */
	private function initialize_field_data( $field, $counter, $group_id ) {
		$price           = RenderHelpers::get_data( $field, 'price', 0 );
		$converted_price = self::convert_currency( $price );
		$price_display   = $converted_price > 0 ? '(+ ' . wc_price( $converted_price ) . ')' : '';
		$tooltip         = RenderHelpers::get_data( $field, 'tooltip', 'off' );
		$multiple        = RenderHelpers::get_data( $field, 'multiple', 'off' );
		$class           = 'rtsb-addon-field-header';
		$class          .= 'on' === $tooltip ? ' has-tooltip' : '';
		$class          .= 'on' === $multiple ? ' has-multiple' : '';
		$multiple        = RenderHelpers::get_data( $field, 'multiple', 'off' );
		$field_id        = 'rtsb_addon_field_' . self::get_product()['product_id'] . '_' . $group_id . '_' . $counter ?? 0;
		$required        = RenderHelpers::get_data( $field, 'required', 'off' );
		$type            = RenderHelpers::get_data( $field, 'type', 'text' );

		if ( 'on' === $multiple && ! ( 'text' === $type || 'textarea' === $type || 'number' === $type ) ) {
			// phpcs:ignore WordPress.Security.NonceVerification.Recommended
			$field_value = ! empty( $_REQUEST[ $field_id ] ) && is_array( $_REQUEST[ $field_id ] ) ? array_map( 'sanitize_text_field', wp_unslash( $_REQUEST[ $field_id ] ) ) : [];
		} else {
			// phpcs:ignore WordPress.Security.NonceVerification.Recommended
			$field_value = ! empty( $_REQUEST[ $field_id ] ) && is_string( $_REQUEST[ $field_id ] ) ? sanitize_text_field( wp_unslash( $_REQUEST[ $field_id ] ) ) : '';
		}

		return [
			'p_id'           => self::get_product()['product_id'],
			'field_id'       => $field_id,
			'type'           => $type,
			'label'          => RenderHelpers::get_data( $field, 'label', '' ),
			'price'          => $converted_price,
			'description'    => RenderHelpers::get_data( $field, 'description', '' ),
			'placeholder'    => RenderHelpers::get_data( $field, 'placeholder', '' ),
			'required'       => $required,
			'tooltip'        => $tooltip,
			'multiple'       => RenderHelpers::get_data( $field, 'multiple', 'off' ),
			'options'        => RenderHelpers::get_data( $field, 'options', [] ),
			'price_display'  => $price_display,
			'counter'        => $counter ?? 0,
			'group_id'       => $group_id ?? 0,
			'class'          => $class,
			'heading_tag'    => AddOnsFns::field_heading_tag(),
			'field_value'    => $field_value,
			'field_required' => ! empty( $required ) && 'on' === $required ? ' required' : '',
			'required_html'  => 'on' === RenderHelpers::get_data( $field, 'required', 'off' ) ? '<span class="required" title="' . esc_attr__( 'Required Fields', 'shopbuilder-pro' ) . '">&nbsp;*</span>' : '',
		];
	}

	/**
	 * Convert the given price to the active currency.
	 *
	 * @param float $price The original price.
	 *
	 * @return float
	 */
	public static function convert_currency( $price ) {
		return apply_filters( 'rtsb/module/product_add_ons/price', $price, self::get_product()['product'] ?? null );
	}

	/**
	 * Get product.
	 *
	 * @return array
	 */
	public static function get_product() {
		global $product;

		if ( ! $product instanceof WC_Product ) {
			return [];
		}

		return [
			'product'    => $product,
			'product_id' => $product->get_id(),
		];
	}

	/**
	 * Get the template path.
	 *
	 * @return string
	 */
	private function get_template_path() {
		return rtsbpro()->get_plugin_template_path();
	}
}
