<?php
/**
 * Product AddOns Order Class.
 *
 * @package RadiusTheme\SBPRO
 */

namespace RadiusTheme\SBPRO\Modules\AddOns\Frontend;

use WC_Order_Item_Product;
use RadiusTheme\SBPRO\Traits\SingletonTrait;
use RadiusTheme\SBPRO\Modules\AddOns\AddOnsFns;

defined( 'ABSPATH' ) || exit();

/**
 * Product AddOns Order Class.
 */
class Order {
	/**
	 * Singleton Trait.
	 */
	use SingletonTrait;

	/**
	 * Module Class Constructor.
	 */
	private function __construct() {
		/**
		 * Actions.
		 */
		add_action( 'woocommerce_checkout_create_order_line_item', [ $this, 'add_addons_meta' ], 10, 3 );
		add_action( 'woocommerce_delete_order_items', [ $this, 'delete_addon_images' ] );

		/**
		 * Filters.
		 */
		add_filter( 'woocommerce_hidden_order_itemmeta', [ $this, 'hide_admin_meta' ], 10, 1 );
		add_filter( 'woocommerce_order_item_get_formatted_meta_data', [ $this, 'hide_addons_in_order_email' ], 10, 2 );
	}

	/**
	 * Adds product addons meta-data to the cart item.
	 *
	 * @param object $item      The cart item object to which meta-data is being added.
	 * @param string $cart_key  The unique identifier for the cart item.
	 * @param array  $values    The cart item values, including the product addons.
	 *
	 * @return void
	 */
	public function add_addons_meta( $item, $cart_key, $values ) {
		if ( empty( $values['rtsb_product_addons'] ) ) {
			return;
		}

		$total_price = 0;

		foreach ( $values['rtsb_product_addons'] as $addon_group ) {
			if ( empty( $addon_group['fields'] ) ) {
				continue;
			}

			foreach ( $addon_group['fields'] as $addon ) {
				$addon_price = 0;

				if ( is_array( $addon['value'] ) ) {
					$value_array = [];

					foreach ( $addon['value'] as $index => $single_value ) {
						$price         = is_array( $addon['price'] ) ? ( $addon['price'][ $index ] ?? 0 ) : $addon['price'];
						$addon_price  += floatval( $price );
						$value_array[] = $single_value . ( $price > 0 ? ' (' . wc_price( Product::convert_currency( $price ) ) . ')' : '' );
					}

					$value = implode( ', ', $value_array );
				} else {
					$addon_price = $addon['price'] ?? 0;
					$value       = $addon['value'];

					if ( 'image_upload' === $addon['type'] ) {
						$image_data = json_decode( $addon['value'], true );

						if ( is_array( $image_data ) && isset( $image_data['name'], $image_data['url'] ) ) {
							$value = sprintf(
								'<a href="%1$s" target="_blank">%2$s</a> <a href="%1$s" download class="rtsb-download-button" style="margin-left: 10px; padding: 5px 10px; background: #0073aa; color: #fff; text-decoration: none; border-radius: 3px;">%3$s</a>',
								esc_url( $image_data['url'] ),
								esc_html( $image_data['name'] ),
								esc_html__( 'Download', 'shopbuilder-pro' ),
							);

							$item->add_meta_data( '_rtsb_addons_image_url', esc_url( $image_data['url'] ) );
						}
					}

					if ( $addon_price ) {
						$value .= ' (' . wc_price( Product::convert_currency( $addon_price ) ) . ')';
					}
				}

				$item->add_meta_data( $addon['label'], $value );

				$total_price += floatval( $addon_price );
			}
		}

		if ( $total_price > 0 ) {
			$item->add_meta_data( '_rtsb_addons_price', wc_price( floatval( $total_price ) ) );
		}
	}

	/**
	 * Delete uploaded images when an order is deleted.
	 *
	 * @param int $order_id The ID of the order being deleted.
	 *
	 * @return void
	 */
	public function delete_addon_images( $order_id ) {
		global $wpdb;

		$results = $wpdb->get_results(
			$wpdb->prepare(
				"SELECT oim.meta_value 
		        FROM {$wpdb->prefix}woocommerce_order_itemmeta AS oim
		        INNER JOIN {$wpdb->prefix}woocommerce_order_items AS oi 
		        ON oim.order_item_id = oi.order_item_id
		        WHERE oim.meta_key = '_rtsb_addons_image_url' 
		        AND oi.order_id = %d",
				absint( $order_id )
			),
			ARRAY_A
		);

		if ( empty( $results ) ) {
			return;
		}

		foreach ( $results as $result ) {
			$this->safe_delete_file( $result['meta_value'] ?? '' );
		}
	}

	/**
	 * Hides the addon price meta-data from the admin order details view.
	 *
	 * @param array $hidden_meta_data Array of meta-data keys that should be hidden.
	 *
	 * @return array
	 */
	public function hide_admin_meta( $hidden_meta_data ) {
		$hidden_meta_data[] = '_rtsb_addons_price';
		$hidden_meta_data[] = '_rtsb_addons_image_url';

		return $hidden_meta_data;
	}

	/**
	 * Hide addons in order email metadata.
	 *
	 * @param array  $formatted_meta The formatted metadata array.
	 * @param object $order_item The order item object.
	 *
	 * @return array
	 */
	public function hide_addons_in_order_email( $formatted_meta, $order_item ) {
		if ( 'on' === AddOnsFns::get_add_ons_settings_data()['show_in_order_email'] ) {
			return $formatted_meta;
		}

		if ( did_action( 'woocommerce_email_order_details' ) ) {
			if ( ! $order_item instanceof WC_Order_Item_Product ) {
				return $formatted_meta;
			}

			$product_id     = $order_item->get_product_id();
			$product_addons = AddOnsFns::get_addon_groups( $product_id );
			$addon_labels   = [];

			foreach ( $product_addons as $addon_group ) {
				foreach ( $addon_group['inputs'] as $input ) {
					$addon_labels[] = $input['label'];
				}
			}

			if ( empty( $addon_labels ) ) {
				return $formatted_meta;
			}

			foreach ( $formatted_meta as $meta_id => $meta ) {
				if ( in_array( $meta->key, $addon_labels, true ) ) {
					unset( $formatted_meta[ $meta_id ] );
				}
			}
		}

		return $formatted_meta;
	}

	/**
	 * Safely delete a file from the uploads' directory.
	 *
	 * @param string $file_url The URL of the file to delete.
	 *
	 * @return void
	 */
	private function safe_delete_file( $file_url ) {
		$upload_dir      = wp_upload_dir();
		$upload_base_url = trailingslashit( $upload_dir['baseurl'] ) . 'shopbuilder_uploads';
		$upload_base_dir = trailingslashit( $upload_dir['basedir'] ) . 'shopbuilder_uploads';

		if ( strpos( $file_url, $upload_base_url ) === false ) {
			return;
		}

		$relative_path = str_replace( $upload_base_url, '', $file_url );
		$file_path     = $upload_base_dir . $relative_path;

		if ( file_exists( $file_path ) ) {
			wp_delete_file( $file_path );
		}
	}
}
