<?php
/**
 * Product AddOns Cart Class.
 *
 * @package RadiusTheme\SBPRO
 */

namespace RadiusTheme\SBPRO\Modules\AddOns\Frontend;

use RadiusTheme\SBPRO\Helpers\FnsPro;
use RadiusTheme\SBPRO\Modules\PreOrder\PreOrderFns;
use RadiusTheme\SBPRO\Traits\SingletonTrait;
use RadiusTheme\SBPRO\Modules\AddOns\AddOnsFns;
use RadiusTheme\SBPRO\Modules\FlashSaleCountdown\CountdownFns;

defined( 'ABSPATH' ) || exit();

/**
 * Product AddOns Cart Class.
 */
class Cart {
	/**
	 * Singleton Trait.
	 */
	use SingletonTrait;

	/**
	 * Module Class Constructor.
	 */
	private function __construct() {
		/**
		 * Filters.
		 */
		add_filter( 'woocommerce_add_cart_item_data', [ $this, 'add_addons_data_to_cart' ], 20, 2 );
		add_filter( 'woocommerce_get_item_data', [ $this, 'display_addon_data' ], 10, 2 );
		add_filter( 'woocommerce_add_cart_item', [ $this, 'update_cart_price' ], 20 );
		add_filter( 'woocommerce_get_cart_item_from_session', [ $this, 'get_cart_item_from_session' ], 20, 2 );
	}

	/**
	 * Adds add-on data to the cart item data for a given product.
	 *
	 * @param array $cart_item_data The current cart item data to which add-on data will be added.
	 * @param int   $product_id The ID of the product for which add-on data is being processed.
	 *
	 * @return array
	 */
	public function add_addons_data_to_cart( $cart_item_data, $product_id ) {
		// Verify nonce and request data.
		if ( ! AddOnsFns::verify_add_to_cart_request( $product_id ) ) {
			return $cart_item_data;
		}

		$addon_processor = new AddonProcessor( $product_id, $cart_item_data );

		return $addon_processor->process();
	}

	/**
	 * Display addon data in the WooCommerce cart.
	 *
	 * @param array $show_data The current array of cart item data to display.
	 * @param array $cart_item The cart item array containing product data.
	 *
	 * @return array
	 */
	public function display_addon_data( $show_data, $cart_item ) {
		if ( 'on' !== AddOnsFns::get_add_ons_settings_data()['show_in_cart'] ) {
			return $show_data;
		}

		if ( ! empty( $cart_item['rtsb_product_addons'] ) ) {
			foreach ( $cart_item['rtsb_product_addons'] as $addon_group ) {
				$show_data = $this->process_addon_group( $addon_group, $show_data );
			}
		}

		return $show_data;
	}

	/**
	 * Process a single addon group and add its data to the cart display.
	 *
	 * @param array $addon_group The addon group data.
	 * @param array $show_data The current array of cart item data to display.
	 *
	 * @return array
	 */
	private function process_addon_group( $addon_group, $show_data ) {
		if ( ! empty( $addon_group['fields'] ) ) {
			foreach ( $addon_group['fields'] as $field ) {
				if ( ! empty( $field['value'] ) ) {
					if ( 'image_upload' === $field['type'] ) {
						$image_data = json_decode( $field['value'], true );

						if ( is_array( $image_data ) && isset( $image_data['name'], $image_data['url'] ) ) {
							$field['value'] = sprintf(
								'<a href="%s" target="_blank">%s</a>',
								esc_url( $image_data['url'] ),
								esc_html( $image_data['name'] )
							);
						}
					}

					$field_output = $this->format_field_output( $field );

					$show_data[] = [
						'name'  => esc_html( $field['label'] ),
						'value' => implode( ', ', $field_output ),
					];
				}
			}
		}

		return $show_data;
	}

	/**
	 * Format the output for a single field within an addon group.
	 *
	 * @param array $field The field data.
	 *
	 * @return array
	 */
	private function format_field_output( $field ) {
		$field_output = [];

		if ( is_array( $field['value'] ) ) {
			foreach ( $field['value'] as $index => $value ) {
				$price          = esc_html( Product::convert_currency( $this->get_field_price( $field, $index ) ) );
				$field_output[] = $this->format_single_field_output( $value, $price );
			}
		} else {
			$price          = is_array( $field['price'] ) ? implode( ', ', array_map( 'esc_html', $field['price'] ) ) : esc_html( $field['price'] );
			$field_output[] = $this->format_single_field_output( $field['value'], esc_html( Product::convert_currency( $price ) ) );
		}

		return $field_output;
	}

	/**
	 * Get the price for a specific field within an addon group.
	 *
	 * @param array $field The field data.
	 * @param int   $index The index for the field value.
	 *
	 * @return float
	 */
	private function get_field_price( $field, $index ) {
		if ( is_array( $field['price'] ) ) {
			return $field['price'][ $index ] ?? 0;
		}

		return $field['price'];
	}

	/**
	 * Format the output for a single field value and its price.
	 *
	 * @param string $value The field value.
	 * @param float  $price The price associated with the field value.
	 *
	 * @return string
	 */
	private function format_single_field_output( $value, $price ) {
		$allowed_tags = [
			'a'      => [
				'href'   => [],
				'title'  => [],
				'target' => [],
			],
			'span'   => [ 'class' => [] ],
			'strong' => [],
			'em'     => [],
			'br'     => [],
		];

		$output = '<span class="addon-field">' . wp_kses( $value, $allowed_tags ) . '</span>';

		if ( $price > 0 ) {
			$output .= ' <span class="addon-price">(+ ' . wc_price( floatval( $price ) ) . ')</span>';
		}

		return $output;
	}

	/**
	 * Update the product price in the cart item based on addons.
	 *
	 * @param array $cart_item The cart item data.
	 *
	 * @return array
	 */
	public function update_cart_price( $cart_item ) {
		$product  = $cart_item['data'];
		$campaign = CountdownFns::get_campaign_for_current_product( $product );

		$price         = floatval( $product->get_price() );
		$regular_price = floatval( $product->get_regular_price() );
		$sale_price    = floatval( $product->get_sale_price() );
		$prices        = compact( 'price', 'regular_price', 'sale_price' );

		if ( FnsPro::is_module_active( 'flash_sale_countdown' ) ) {
			$prices = AddOnsFns::apply_campaign_discount( $prices, $campaign );
		}

		return $this->update_product_price( $cart_item, $prices );
	}

	/**
	 * Get the cart item data from the session and update it with addon data and prices.
	 *
	 * @param array $cart_item The current cart item data.
	 * @param array $values The session data for the cart item.
	 *
	 * @return array
	 */
	public function get_cart_item_from_session( $cart_item, $values ) {
		$product       = $cart_item['data'];
		$campaign      = CountdownFns::get_campaign_for_current_product( $product );
		$price         = floatval( $product->get_price() );
		$regular_price = floatval( $product->get_regular_price() );
		$sale_price    = floatval( $product->get_sale_price() );
		$prices        = compact( 'price', 'regular_price', 'sale_price' );

		if ( FnsPro::is_module_active( 'flash_sale_countdown' ) ) {
			$prices = AddOnsFns::apply_campaign_discount( $prices, $campaign );
		}

		if ( isset( $values['rtsb_product_addons'] ) ) {
			$cart_item['rtsb_product_addons'] = $values['rtsb_product_addons'];
		}

		return $this->update_product_price( $cart_item, $prices );
	}

	/**
	 * Update the product price in the cart item data based on the given prices.
	 *
	 * @param array $cart_item The cart item data.
	 * @param array $prices The prices (regular, sale, and current) to update.
	 *
	 * @return array
	 */
	public function update_product_price( $cart_item, $prices ) {
		$result_data = $this->price_calculate( $cart_item, $prices );
		$cart_item['data']->set_price( $result_data['price'] );
		$cart_item['data']->set_regular_price( $result_data['regular_price'] );
		$cart_item['data']->set_sale_price( $result_data['sale_price'] );
		return $cart_item;
	}

	/**
	 * Calculate the total price of the product in the cart, including addons.
	 *
	 * @param array $cart_data The cart item data.
	 * @param array $prices The base prices (regular, sale, and current).
	 *
	 * @return array
	 */
	public function price_calculate( $cart_data, $prices ) {
		$product_id = 'variation' === $cart_data['data']->get_type() ? $cart_data['variation_id'] : $cart_data['product_id'];
		$product    = wc_get_product( $product_id );
		$campaign   = CountdownFns::get_campaign_for_current_product( $product );

		/**
		 * Addons price.
		 */
		$addons_price = $this->calculate_addons_price( $cart_data );

		/**
		 * Flash sale compatibility.
		 */
		if ( FnsPro::is_module_active( 'flash_sale_countdown' ) && ! empty( $campaign ) ) {
			$prices = AddOnsFns::apply_campaign_discount( $prices, $campaign );
		}

		/**
		 * Pre-Order compatibility.
		 */
		if ( FnsPro::is_module_active( 'pre_order' ) && PreOrderFns::is_on_pre_order( $product ) ) {
			$prices = AddOnsFns::apply_preorder_price( $prices, $cart_data['data'] );
		}

		return $this->add_addons_to_prices( $prices, $addons_price );
	}

	/**
	 * Calculate the total price of the add-ons.
	 *
	 * @param array $cart_data The cart item data.
	 *
	 * @return float
	 */
	private function calculate_addons_price( $cart_data ) {
		$addons_price = 0;

		if ( ! empty( $cart_data['rtsb_product_addons'] ) ) {
			foreach ( $cart_data['rtsb_product_addons'] as $addon_group ) {
				if ( ! empty( $addon_group['fields'] ) ) {
					foreach ( $addon_group['fields'] as $field ) {
						$field_price   = is_array( $field['price'] )
							? array_sum( array_map( 'floatval', $field['price'] ) )
							: floatval( $field['price'] );
						$addons_price += $field_price;
					}
				}
			}
		}

		return $addons_price;
	}

	/**
	 * Add the addons price to all base prices (price, regular, sale).
	 *
	 * @param array $prices The base prices.
	 * @param float $addons_price The calculated add-ons price.
	 *
	 * @return array
	 */
	private function add_addons_to_prices( $prices, $addons_price ) {
		$prices['price']         += $addons_price;
		$prices['regular_price'] += $addons_price;
		$prices['sale_price']    += $addons_price;
		$prices['addons_price']   = $addons_price;

		return $prices;
	}
}
