<?php
/**
 * Product Add-Ons Module Custom Fields Class.
 *
 * @package RadiusTheme\SB
 */

namespace RadiusTheme\SBPRO\Modules\AddOns\Admin;

use RadiusTheme\SB\Helpers\Fns;
use RadiusTheme\SBPRO\Traits\SingletonTrait;
use RadiusTheme\SBPRO\Helpers\AdminInputsFns;
use RadiusTheme\SBPRO\Modules\AddOns\AddOnsFns;

defined( 'ABSPATH' ) || exit();

/**
 * Product Add-Ons Module Admin Class.
 */
class CustomFields {
	/**
	 * Singleton Trait.
	 */
	use SingletonTrait;

	/**
	 * Class Constructor.
	 */
	private function __construct() {
		$this->add_custom_data_tab();
		$this->add_tab_options();
		$this->save_tab_options();
	}

	/**
	 * Adds a custom tab to the WooCommerce product data tabs.
	 *
	 * @return void
	 */
	private function add_custom_data_tab() {
		add_filter( 'woocommerce_product_data_tabs', [ $this, 'add_product_add_ons_tab' ] );
	}


	/**
	 * Adds meta-boxes to the WooCommerce custom data tab.
	 *
	 * @return void
	 */
	public function add_tab_options() {
		add_action( 'woocommerce_product_data_panels', [ $this, 'render_meta_boxes' ] );
	}

	/**
	 * Saves the Product Add-Ons data when the product is saved.
	 *
	 * @return void
	 */
	public function save_tab_options() {
		add_action( 'woocommerce_process_product_meta', [ AddOnsFns::class, 'save_product_add_ons' ] );
	}

	/**
	 * Adds the Product Add-Ons tab to the product data tabs.
	 *
	 * @param array $tabs Existing tabs.
	 *
	 * @return array Modified tabs.
	 */
	public function add_product_add_ons_tab( $tabs ) {
		$tabs['rtsb_product_add_ons'] = [
			'label'    => esc_html__( 'Product Add-Ons', 'shopbuilder-pro' ),
			'target'   => 'rtsb_product_add_ons',
			'class'    => [ 'show_if_simple', 'show_if_variable' ],
			'priority' => 61,
		];

		return $tabs;
	}

	/**
	 * Renders meta-boxes to the WooCommerce product data panels.
	 *
	 * @return void
	 */
	public function render_meta_boxes() {
		global $post;

		$disable_add_ons    = get_post_meta( $post->ID, '_rtsb_disable_addons', true );
		$enable_add_ons     = get_post_meta( $post->ID, '_rtsb_enable_addons', true );
		$add_ons            = get_post_meta( $post->ID, '_rtsb_product_addons', true );
		$disable_checked    = ! empty( $disable_add_ons ) && 'on' === $disable_add_ons ? 'on' : 'off';
		$disable_visibility = 'on' === $disable_checked ? ' hidden' : '';
		$enable_checked     = ! empty( $enable_add_ons ) && 'on' === $enable_add_ons ? 'on' : 'off';
		$enable_visibility  = 'on' === $enable_checked ? '' : ' hidden';

		?>
		<div id='rtsb_product_add_ons' class='panel woocommerce_options_panel rtsb_panel hidden'>
			<div class="rtsb-group-heading">
				<h2><?php esc_html_e( 'Product Add-On Settings', 'shopbuilder-pro' ); ?></h2>
			</div>

			<div id="rtsb_product_add_ons_options_group" class="options_group">
				<?php wp_nonce_field( rtsb()->nonceText, rtsb()->nonceId ); ?>
				<div class="addons-group-control">
					<div id="rtsb_disable_addons_group">
						<?php
						Fns::print_html(
							AdminInputsFns::generate_switch_input(
								[
									'id'            => '_rtsb_disable_addons',
									'label'         => esc_html__( 'Disable Global Rules?', 'shopbuilder-pro' ),
									'desc_tip'      => true,
									'aria_label'    => esc_html__( 'Disable Global Product Add-ons', 'shopbuilder-pro' ),
									'description'   => esc_html__( 'Switch on to disable the ShopBuilder global add-on rules for this product.', 'shopbuilder-pro' ),
									'checked'       => esc_attr( $disable_checked ),
									'value'         => 'yes',
									'input_class'   => 'switch-input',
									'wrapper_class' => 'form-field disable-rules',
								]
							),
							true
						);
						?>
					</div>
					<div id="rtsb_enable_addons_group">
						<?php
						Fns::print_html(
							AdminInputsFns::generate_switch_input(
								[
									'id'            => '_rtsb_enable_addons',
									'label'         => esc_html__( 'Override Global Rules?', 'shopbuilder-pro' ),
									'desc_tip'      => true,
									'aria_label'    => esc_html__( 'Override Global Product Add-ons', 'shopbuilder-pro' ),
									'description'   => esc_html__( 'Switch on to override the ShopBuilder global add-on rules.', 'shopbuilder-pro' ),
									'checked'       => esc_attr( $enable_checked ),
									'value'         => 'yes',
									'input_class'   => 'switch-input',
									'wrapper_class' => 'form-field' . esc_attr( $disable_visibility ),
								]
							),
							true
						);
						?>
					</div>
				</div>
				<div id="rtsb_addon_groups" class="rtsb-repeater<?php echo esc_attr( $enable_visibility ); ?>">
					<?php
					if ( ! empty( $add_ons ) ) {
						foreach ( $add_ons as $group_index => $group ) {
							Fns::print_html( AddOnsFns::render_admin_group_template( $group, $group_index ), true );
						}
					}
					?>
				</div>

				<div class="repeater-footer">
					<button type="button" class="button button-primary" id="save_add_ons_button" disabled="disabled"><?php esc_html_e( 'Save Changes', 'shopbuilder-pro' ); ?></button>
					<button type="button" class="button<?php echo esc_attr( $enable_visibility ); ?>" id="add_group_button"><?php esc_html_e( 'Add New Group', 'shopbuilder-pro' ); ?></button>
					<div class="save-status"></div>
				</div>
			</div>
		</div>
		<?php
	}
}
