<?php
/**
 * Product Add-Ons Module Front End Class.
 *
 * @package RadiusTheme\SB
 */

namespace RadiusTheme\SBPRO\Modules\AddOns;

use RadiusTheme\SB\Helpers\Fns;
use RadiusTheme\SB\Helpers\Cache;
use RadiusTheme\SBPRO\Helpers\FnsPro;
use RadiusTheme\SBPRO\Traits\SingletonTrait;
use RadiusTheme\SBPRO\Modules\AddOns\Frontend\Cart;
use RadiusTheme\SBPRO\Modules\AddOns\Frontend\Order;
use RadiusTheme\SBPRO\Modules\AddOns\Frontend\Product;

defined( 'ABSPATH' ) || exit();

/**
 * Product Add-Ons Module Front End Class.
 */
class AddOnsFrontEnd {
	/**
	 * Singleton Trait.
	 */
	use SingletonTrait;

	/**
	 * Options.
	 *
	 * @var array|mixed
	 */
	private array $options;

	/**
	 * Asset Handle
	 *
	 * @var string
	 */
	private $handle = 'rtsb-product-addons';

	/**
	 * Module Class Constructor.
	 */
	private function __construct() {
		/**
		 * Get the options.
		 */
		$this->options = Fns::get_options( 'modules', 'product_add_ons' );

		/**
		 * Product modifications.
		 */
		Product::instance();

		/**
		 * Cart modifications.
		 */
		Cart::instance();

		/**
		 * Order data modifications.
		 */
		Order::instance();

		/**
		 * Global Styles.
		 */
		$this->dynamic_styles();
	}

	/**
	 * Dynamic Styles.
	 *
	 * @return void
	 */
	public function dynamic_styles() {
		add_action( 'wp_enqueue_scripts', [ $this, 'enqueue_dynamic_styles' ], 99 );
		add_action( 'rtsb/module/product_add_ons/scripts', [ $this, 'tooltip_script' ] );
	}

	/**
	 * Dynamic styles.
	 *
	 * @return void
	 */
	public function enqueue_dynamic_styles() {
		// Enqueue assets.
		$this->handle = Fns::enqueue_module_assets(
			$this->handle,
			'product-add-ons',
			[
				'context' => rtsbpro(),
				'version' => RTSBPRO_VERSION,
			]
		);

		$cache_key         = 'rtsb_addons_dynamic_styles_' . md5( wp_json_encode( $this->options ) );
		$addons_parent     = '.rtsb-addon-groups';
		$addons_header     = $addons_parent . ' .rtsb-addons-header';
		$addons_title      = $addons_header . ' .rtsb-addons-title';
		$addons_group      = $addons_parent . ' .rtsb-addon-fields';
		$addons_label      = $addons_parent . ' .rtsb-addon-title';
		$addons_desc       = $addons_parent . ' .rtsb-addon-field-description';
		$addons_text       = "$addons_group input, $addons_group textarea, $addons_group .select2-selection--single .select2-selection__rendered";
		$checkbox_label    = "$addons_group .rtsb-addon-field-content.rtsb-addon-field-checkbox label";
		$radio_label       = "$addons_group .rtsb-addon-field-content.rtsb-addon-field-radio label";
		$text_select_label = "$addons_group .rtsb-addon-field-content.rtsb-addon-field-text-select .rtsb-checkbox-group .rtsb-checkbox-field > label";
		$switch_label      = "$addons_group .rtsb-addon-field-content.rtsb-addon-field-switch .rtsb-checkbox-field > label";
		$upload_button     = "$addons_group .rtsb-addon-field-content.rtsb-addon-field-image .rtsb-choose-file";
		$form_inputs       = "$addons_parent .rtsb-addon-field-content.rtsb-addon-field-text input[type=text], $addons_parent .rtsb-addon-field-content.rtsb-addon-field-number input[type=number], $addons_parent .rtsb-addon-field-content.rtsb-addon-field-textarea textarea";
		$checkbox          = "$addons_parent .rtsb-addon-field-content.rtsb-addon-field-checkbox input[type=checkbox]:checked";
		$radio             = "$addons_parent .rtsb-addon-field-content.rtsb-addon-field-radio input[type=radio]";
		$text_select       = "$addons_parent .rtsb-addon-field-content.rtsb-addon-field-text-select .rtsb-checkbox-group input[type=checkbox] + .rtsb-option-label";
		$switch            = "$addons_parent .rtsb-addon-field-content.rtsb-addon-field-switch .switch-field input[type=checkbox]:checked + .slider";
		$price_box         = '.rtsb-product-addons-total';
		$primary_color     = "$addons_parent, .rtsb-select-dropdown";

		// phpcs:disable
		// Disabled phpcs for one-liner display.
		$css_properties = [
			'group_header_font_size'        => [ 'selector' => "$addons_header .rtsb-addons-title", 'property' => 'font-size' ],
			'group_desc_font_size'          => [ 'selector' => "$addons_header .rtsb-addons-description", 'property' => 'font-size' ],
			'group_header_background_color' => [ 'selector' => $addons_header, 'property' => 'background-color' ],
			'group_header_text_color'       => [ 'selector' => "$addons_header, $addons_title", 'property' => 'color' ],
			'group_header_padding'          => [ 'selector' => $addons_header, 'property' => 'padding' ],
			'group_header_margin'           => [ 'selector' => $addons_header, 'property' => 'margin' ],
			'field_label_font_size'         => [ 'selector' => $addons_label, 'property' => 'font-size' ],
			'field_desc_font_size'          => [ 'selector' => $addons_desc, 'property' => 'font-size' ],
			'field_text_color'              => [ 'selector' => $addons_text, 'property' => 'color' ],
			'checkbox_label_color'          => [ 'selector' => $checkbox_label, 'property' => 'color' ],
			'radio_label_color'             => [ 'selector' => $radio_label, 'property' => 'color' ],
			'text_select_label_color'       => [ 'selector' => $text_select_label, 'property' => 'color' ],
			'switch_label_color'            => [ 'selector' => $switch_label, 'property' => 'color' ],
			'image_button_color'            => [ 'selector' => $upload_button, 'property' => 'color' ],
			'image_button_bg_color'         => [ 'selector' => $upload_button, 'property' => [ 'background-color', 'border-color' ] ],
			'image_button_hover_color'      => [ 'selector' => $upload_button . ':hover', 'property' => 'color' ],
			'image_button_hover_bg_color'   => [ 'selector' => $upload_button . ':hover', 'property' => [ 'background-color', 'border-color' ] ],
			'field_label_color'             => [ 'selector' => $addons_label, 'property' => 'color' ],
			'group_background_color'        => [ 'selector' => $addons_group, 'property' => 'background-color' ],
			'group_border_color'            => [ 'selector' => $addons_group, 'property' => 'border-color' ],
			'group_radius'                  => [ 'selector' => $addons_group, 'property' => 'border-radius' ],
			'group_padding'                 => [ 'selector' => $addons_group, 'property' => 'padding' ],
			'group_margin'                  => [ 'selector' => $addons_group, 'property' => 'margin' ],
			'form_primary_color'            => [ 'selector' => $primary_color, 'property' => '--rtsb-color-primary' ],
			'form_background_color'         => [ 'selector' => "$form_inputs, $addons_group .rtsb-addon-field-select .select2-selection", 'property' => 'background-color' ],
			'form_border_color'             => [ 'selector' => $form_inputs, 'property' => 'border-color' ],
			'checkbox_background_color'     => [ 'selector' => $checkbox, 'property' => 'background-color' ],
			'checkbox_border_color'         => [ 'selector' => $checkbox, 'property' => 'border-color' ],
			'radio_background_color'        => [ 'selector' => $radio, 'property' => 'background-color' ],
			'radio_border_color'            => [ 'selector' => $radio, 'property' => 'border-color' ],
			'text_select_background_color'  => [ 'selector' => $text_select, 'property' => 'background-color' ],
			'switch_background_color'       => [ 'selector' => $switch, 'property' => 'background-color' ],
			'price_box_text_color'          => [ 'selector' => $price_box, 'property' => 'color' ],
			'price_box_background_color'    => [ 'selector' => $price_box, 'property' => 'background-color' ],
			'price_box_border_color'        => [ 'selector' => $price_box, 'property' => 'border-color' ],
			'price_box_radius'              => [ 'selector' => $price_box, 'property' => 'border-radius' ],
			'price_box_padding'             => [ 'selector' => $price_box, 'property' => 'padding' ],
			'price_box_margin'              => [ 'selector' => $price_box, 'property' => 'margin' ],
		];
		// phpcs:enable

		// Generate dynamic CSS.
		FnsPro::dynamic_styles( $this->options, $cache_key, $css_properties, $this->handle );
	}


	/**
	 * Tooltip script.
	 *
	 * @return void
	 */
	public function tooltip_script() {
		$cache_key     = 'rtsb_addons_tooltip_' . md5( $this->options['group_tooltip_position'] ?? 'default' );
		$cached_script = wp_cache_get( $cache_key, 'shopbuilder' );

		wp_enqueue_style( 'select2' );
		wp_enqueue_script( 'rtsb-tipsy' );
		wp_enqueue_script( 'select2' );

		if ( false !== $cached_script ) {
			wp_add_inline_script( 'rtsb-tipsy', $cached_script );

			return;
		}

		$tooltip_position = $this->options['group_tooltip_position'] ?? 'right';

		switch ( $tooltip_position ) {
			case 'bottom':
				$tooltip_position = 'n';
				break;

			case 'top':
				$tooltip_position = 's';
				break;

			case 'left':
				$tooltip_position = 'e';
				break;

			default:
				$tooltip_position = 'w';
				break;
		}

		$tooltip_script = "
            (function($) {
                $(window).on('load', function() {
	                if ($('.tipsy-tooltip').length > 0) {
	                    $('.tipsy-tooltip').each(function() {
	                        $(this).tipsy({
	                            gravity: '$tooltip_position',
	                            fade: true,
	                            html: true,
	                            offset: 5,
	                            opacity: 1,
	                            className: 'rtsb-addons-tooltip',
	                        });
	                    });
	                }
                });
            })(jQuery);
        ";

		wp_cache_set( $cache_key, $tooltip_script, 'shopbuilder', 12 * HOUR_IN_SECONDS );
		Cache::set_data_cache_key( $cache_key );

		wp_add_inline_script( 'rtsb-tipsy', $tooltip_script );
	}
}
