<?php
/**
 * Installation
 *
 * @package RadiusTheme\SB
 */

namespace RadiusTheme\SBPRO\Helpers;

use RadiusTheme\SB\Helpers\Cache;
use RadiusTheme\SB\Models\ExtraSettings;
use RadiusTheme\SB\Helpers\Installation as InstallationMain;

// Do not allow directly accessing this file.
if ( ! defined( 'ABSPATH' ) ) {
	exit( 'This script cannot be accessed directly.' );
}

/**
 * Installation
 */
class Installation {
	/**
	 * Init
	 *
	 * @return void
	 */
	public static function init() {
		add_action( 'admin_init', [ __CLASS__, 'check_version' ], 6 );
		add_action( 'upgrader_process_complete', [ __CLASS__, 'maybe_run_after_update' ], 10, 2 );
	}

	/**
	 * Check Version
	 *
	 * @return void
	 */
	public static function check_version() {
		$version = ExtraSettings::instance()->get_option( 'rtsbpro_version' ) ?? get_option( 'rtsbpro_version' );

		if ( ! $version || version_compare( $version, RTSBPRO_VERSION, '<' ) ) {
			self::save_default_pro_settings();
		}
	}

	/**
	 * Maybe Run After Update
	 *
	 * @param object $upgrader_object Upgrader Object.
	 * @param array  $options Options.
	 *
	 * @return void
	 */
	public static function maybe_run_after_update( $upgrader_object, $options ) {
		if ( 'plugin' !== $options['type'] || ! in_array( $options['action'], [ 'install', 'update' ], true ) ) {
			return;
		}

		// For normal update flow.
		if ( ! empty( $options['plugins'] ) ) {
			foreach ( $options['plugins'] as $plugin ) {
				if ( strpos( $plugin, 'shopbuilder-pro' ) !== false ) {
					if ( class_exists( '\RadiusTheme\SB\Helpers\Cache' ) ) {
						Cache::clear_all_cache();
					}
					break;
				}
			}
			return;
		}

		// For upload-based installation.
		if (
			! empty( $upgrader_object->result['destination_name'] ) &&
			strpos( $upgrader_object->result['destination_name'], 'shopbuilder-pro' ) !== false
		) {
			if ( class_exists( '\RadiusTheme\SB\Helpers\Cache' ) ) {
				Cache::clear_all_cache();
			}
		}
	}

	/**
	 * Save Default Pro Settings
	 *
	 * @return void
	 */
	public static function save_default_pro_settings() {
		// If we made it till here nothing is running yet, lets set the transient now.
		set_transient( 'rtsb_installing', 'yes', MINUTE_IN_SECONDS * 10 );

		InstallationMain::set_default_options();

		self::update_rtsb_version();

		delete_transient( 'rtsb_installing' );
	}

	/**
	 * Update RTSB Version
	 *
	 * @return void
	 */
	private static function update_rtsb_version() {
		ExtraSettings::instance()->set_option( 'rtsbpro_version', RTSB_VERSION );
	}

	/**
	 * Activate Function
	 *
	 * @return void
	 */
	public static function activate() {
		if ( class_exists( '\RadiusTheme\SB\Helpers\Cache' ) ) {
			Cache::clear_all_cache();
		}
	}

	/**
	 * Deactivate Function
	 *
	 * @return void
	 */
	public static function deactivation() {
		self::clear_scheduled_events();
	}

	/**
	 * Clear Scheduled Events
	 *
	 * @return void
	 */
	public static function clear_scheduled_events() {
		$schedule = get_option( 'rtsb_cron_schedule', [] );
		if ( empty( $schedule ) ) {
			return;
		}
		foreach ( $schedule as $v ) {
			wp_clear_scheduled_hook( $v );
		}
	}
}
