<?php
/**
 * FnsPro Helpers class
 *
 * @package  RadiusTheme\SB
 */

namespace RadiusTheme\SBPRO\Helpers;

use DateTime;
use WP_Term;
use WC_Product_Query;
use RadiusTheme\SB\Helpers\Fns;
use RadiusTheme\SB\Helpers\Cache;
use RadiusTheme\SB\Models\DataModel;
use RadiusTheme\SB\Helpers\BuilderFns;
use RadiusTheme\SB\Helpers\ElementorDataMap;
use RadiusTheme\SBPRO\Modules\AddOns\AddOnsFns;
use RadiusTheme\SBPRO\Modules\AddOns\Frontend\Cart;
use RadiusTheme\SBPRO\Modules\PreOrder\PreOrderFns;
use RadiusTheme\SBPRO\Modules\PreOrder\Frontend\Product;
use RadiusTheme\SBPRO\Modules\PartialPay\Frontend\CartPage;
use RadiusTheme\SBPRO\Modules\FlashSaleCountdown\CountdownFns;
use RadiusTheme\SBPRO\Modules\AddOns\Frontend\Product as AddOnsProduct;

// Do not allow directly accessing this file.
if ( ! defined( 'ABSPATH' ) ) {
	exit( 'This script cannot be accessed directly.' );
}

/**
 * FnsPro class
 */
class FnsPro {
	/**
	 * @var array
	 */
	private static $cache = [];

	/**
	 * Check License
	 *
	 * @return mixed|null
	 */
	public static function check_license() {
		return apply_filters( 'rtsbpro_check_license', true );
	}

	/**
	 * Gets the current timestamp in WordPress timezone.
	 *
	 * @return int Current timestamp.
	 */
	public static function currentTimestamp() {
		$wp_timezone = wp_timezone();
		$now         = new DateTime( 'now', $wp_timezone ); // Current time in WP timezone.
		return $now->getTimestamp();
	}

	/**
	 * Filter layouts.
	 *
	 * @return mixed|null
	 */
	public static function filter_layouts() {
		return apply_filters(
			'rtsb/elements/elementor/filter_layouts',
			[
				'vertical'   => [
					'title' => esc_html__( 'Vertical', 'shopbuilder-pro' ),
					'url'   => esc_url( rtsb()->get_assets_uri( 'images/layout/filter-layout-1.png' ) ),
				],
				'horizontal' => [
					'title' => esc_html__( 'Horizontal', 'shopbuilder-pro' ),
					'url'   => esc_url( rtsb()->get_assets_uri( 'images/layout/filter-layout-2.png' ) ),
				],
				// 'off-canvas' => [
				// 'title' => esc_html__( 'Off Canvas', 'shopbuilder-pro' ),
				// 'url'   => esc_url( rtsb()->get_assets_uri( 'images/layout/grid-style-02.png' ) ),
				// ],
			]
		);
	}
	/**
	 * Product Stock Count layouts.
	 *
	 * @return mixed|null
	 */
	public static function product_stock_count_layouts() {
		return apply_filters(
			'rtsb/elements/elementor/product_stock_count_layouts',
			[
				'layout1' => [
					'title' => esc_html__( 'Layout 1', 'shopbuilder-pro' ),
					'url'   => esc_url( rtsb()->get_assets_uri( 'images/layout/filter-layout-1.png' ) ),
				],
			]
		);
	}
	/**
	 * Highlighted Product Widget layouts.
	 *
	 * @return mixed|null
	 */
	public static function highlighted_product_layouts() {
		return apply_filters(
			'rtsb/elements/elementor/highlighted_product_layouts',
			[
				'layout1' => [
					'title' => esc_html__( 'Layout 1', 'shopbuilder-pro' ),
					'url'   => esc_url( rtsb()->get_assets_uri( 'images/layout/filter-layout-1.png' ) ),
				],
			]
		);
	}

	/**
	 * Order message
	 *
	 * @return void
	 */
	public static function order_message() {
		printf( '<div class="rtsb-order-info">%s</div>', esc_html__( 'There is no order, please create at least one demo order for preview', 'shopbuilder-pro' ) );
	}

	/**
	 * Retrieve product categories (including hierarchical support)
	 *
	 * @param string $taxonomy Taxonomy name.
	 * @return int[]|string|string[]|\WP_Error|\WP_Term[]
	 */
	public static function get_products( $taxonomy ) {
		$args = [
			'taxonomy'     => esc_html( $taxonomy ),
			'hide_empty'   => false,
			'pad_counts'   => true,
			'hierarchical' => true,
		];

		$cache_key      = 'rtsb_get_terms_' . $taxonomy;
		$taxonomy_terms = wp_cache_get( $cache_key, 'shopbuilder' );

		if ( false === $taxonomy_terms ) {
			$queried_object = get_queried_object();

			if ( 'archive' === apply_filters( 'rtsb/builder/set/current/page/type', '' ) && $queried_object instanceof WP_Term && $queried_object->taxonomy === $taxonomy ) {
				$term_id           = $queried_object->term_id;
				$taxonomy          = $queried_object->taxonomy;
				$archive_cache_key = 'rtsb_get_terms_' . $taxonomy . $term_id;
				$taxonomy_terms    = wp_cache_get( $archive_cache_key, 'shopbuilder' );

				if ( false === $taxonomy_terms ) {
					$taxonomy_terms = [ $queried_object ];
					$child_args     = [
						'taxonomy'   => $taxonomy,
						'child_of'   => $term_id,
						'hide_empty' => false,
					];

					$child_terms = get_terms( $child_args );

					if ( ! empty( $child_terms ) ) {
						$taxonomy_terms = array_merge( $taxonomy_terms, $child_terms );
					}

					wp_cache_set( $archive_cache_key, $taxonomy_terms, 'shopbuilder' );
					Cache::set_data_cache_key( $archive_cache_key );
				}
			} else {
				$taxonomy_terms = get_terms( $args );
			}

			wp_cache_set( $cache_key, $taxonomy_terms, 'shopbuilder' );
			Cache::set_data_cache_key( $cache_key );
		}

		return $taxonomy_terms;
	}


	/**
	 * Get filter title.
	 *
	 * @param array $title Filter title.
	 * @return string
	 */
	public static function get_filter_title( $title ) {
		$html = '';
		$icon = '';

		if ( empty( $title['title'] ) ) {
			return $html;
		}

		$html .= '<div class="filter-title-wrapper ' . $title['icon_position'] . '  preset-' . esc_html( $title['preset'] ) . '">';

		if ( $title['show_icon'] ) {
			$icon .= '<span class="filter-title-icon">';
			$icon .= Fns::icons_manager( $title['icon'], 'title-icon' );
			$icon .= '</span>';
		}

		$html .= '<h3 class="widget-title rtsb-d-flex rtsb-align-items-center">' . $icon . '<span class="title">' . $title['title'] . '</span></h3>';

		$html .= '</div>';

		return apply_filters( 'rtsb/elementor/filter/title', $html, $title );
	}

	/**
	 * Get base URL.
	 *
	 * @return string|null
	 */
	public static function get_base_url() {
		global $wp;
		return home_url( add_query_arg( [], $wp->request ) );
	}

	/**
	 * Get taxonomy type.
	 *
	 * @param string $tax_type Taxonomy type.
	 * @param string $attr_type Attribute type.
	 *
	 * @return string
	 */
	public static function get_tax_type( $tax_type, $attr_type = '' ) {
		if ( 'product_attr' !== $tax_type ) {
			return $tax_type;
		}

		if ( empty( $attr_type ) ) {
			return $tax_type;
		}

		return $attr_type;
	}

	/**
	 * Get attribute terms.
	 *
	 * @param string $tax Taxonomy type.
	 * @return int[]|string|string[]|\WP_Error|\WP_Term[]
	 */
	public static function get_attribute_terms( $tax ) {
		$cache_key = 'get_attribute_terms_' . $tax;
		if ( isset( self::$cache[ $cache_key ] ) ) {
			return self::$cache[ $cache_key ];
		}

		$attributes = wc_get_attribute_taxonomies();
		$att_name   = str_replace( 'pa_', '', $tax );
		$exist      = array_search( $att_name, array_column( $attributes, 'attribute_name' ), true );
		if ( false === $exist ) {
			self::$cache[ $cache_key ] = [];
			return self::$cache[ $cache_key ];
		}
		$attribute = $tax;

		if ( ! is_shop() && is_product_taxonomy() ) {
			$term = get_queried_object();

			if ( ! $term instanceof WP_Term ) {
				return [];
			}

			$args = [
				'status' => 'publish',
				'limit'  => -1,
				'return' => 'ids',
			];

			if ( 'product_cat' === $term->taxonomy ) {
				$args['category'] = [ $term->slug ];
			}

			if ( 'product_tag' === $term->taxonomy ) {
				$args['tag'] = [ $term->slug ];
			}

			if ( strpos( $term->taxonomy, 'pa_' ) !== false ) {
				$args['tax_query'] = [
					[
						'taxonomy' => $term->taxonomy,
						'field'    => 'term_id',
						'terms'    => $term->term_id,
						'operator' => 'IN',
					],
				];
			}

			$query      = new WC_Product_Query( $args );
			$products   = $query->get_products();
			$attr_terms = [];

			foreach ( $products as $product_id ) {
				$product_attr = wp_get_post_terms( $product_id, $attribute );

				foreach ( $product_attr as $attr ) {
					$attr_terms[] = $attr;
				}
			}
			self::$cache[ $cache_key ] = self::array_unique( $attr_terms );
			return self::$cache[ $cache_key ];
		} else {
			self::$cache[ $cache_key ] = get_terms(
				[
					'taxonomy'   => $attribute,
					'hide_empty' => false,
				]
			);
			return self::$cache[ $cache_key ];
		}
	}

	/**
	 * Remove duplicate values from an array.
	 *
	 * @param array $array          The input array.
	 * @param bool  $keep_key_assoc Whether to keep the key associations after removing duplicates.
	 * @return array The array with duplicates removed.
	 */
	public static function array_unique( $array, $keep_key_assoc = false ) {
		$duplicate_keys = [];
		$tmp            = [];

		foreach ( $array as $key => $val ) {
			// convert objects to arrays, in_array() does not support objects.
			if ( is_object( $val ) ) {
				$val = (array) $val;
			}

			if ( ! in_array( $val, $tmp, true ) ) {
				$tmp[] = $val;
			} else {
				$duplicate_keys[] = $key;
			}
		}

		foreach ( $duplicate_keys as $key ) {
			unset( $array[ $key ] );
		}

		return $keep_key_assoc ? $array : array_values( $array );
	}

	/**
	 * Get attribute term info.
	 *
	 * @param string $tax Taxonomy type.
	 * @return array
	 */
	public static function get_attribute_term_info( $tax ) {
		$result     = [];
		$attributes = wc_get_attribute_taxonomies();

		foreach ( $attributes as $attr ) {
			if ( str_replace( 'pa_', '', $tax ) === $attr->attribute_name ) {
				$term_type      = $attr->attribute_type;
				$attribute_slug = $attr->attribute_name;
				$attribute      = wc_attribute_taxonomy_name( $attr->attribute_name );
				break;
			}
		}

		$filter_name = 'filter_' . wc_attribute_taxonomy_slug( $attribute_slug );

		$result['attribute']      = $attribute ?? '';
		$result['term_type']      = $term_type ?? '';
		$result['attribute_slug'] = $attribute_slug ?? '';
		$result['filter_name']    = $filter_name;
		$result['base_link']      = self::get_base_url();

		return $result;
	}

	/**
	 * Count the number of occurrences for a specific term in a taxonomy for the current queried object.
	 *
	 * @param object|array $tax   The term object for the taxonomy.
	 * @param int          $count The initial count value.
	 *
	 * @return int The updated count value.
	 */
	public static function count_tax_data( $tax, $count ) {
		$page = get_queried_object();

		if ( is_array( $tax ) && strpos( $tax['taxonomy'], 'attribute_' ) !== false ) {
			$taxonomy = str_replace( 'attribute_', '', $tax['taxonomy'] );
			$term_id  = $tax['term_id'];
		} else {
			$taxonomy = $tax->taxonomy;
			$term_id  = $tax->term_id;
		}

		if ( strpos( $page->taxonomy, 'pa_' ) !== false ) {
			$args['status']    = 'publish';
			$args['limit']     = -1;
			$args['return']    = 'ids';
			$args['tax_query'] = [
				'relation' => 'AND',
				[
					'taxonomy' => $page->taxonomy,
					'field'    => 'term_id',
					'terms'    => $page->term_id,
					'operator' => 'IN',
				],
				[
					'taxonomy' => $tax->taxonomy,
					'field'    => 'term_id',
					'terms'    => $tax->term_id,
					'operator' => 'IN',
				],
			];

			$query    = new WC_Product_Query( $args );
			$products = $query->get_products();

			return count( $products );
		}

		if ( $taxonomy !== $page->taxonomy ) {
			$args = [
				'limit'     => -1,
				'return'    => 'ids',
				'status'    => 'publish',
				'tax_query' => [
					[
						'taxonomy' => $taxonomy,
						'field'    => 'term_id',
						'terms'    => $term_id,
					],
				],
			];

			if ( 'product_cat' === $page->taxonomy ) {
				$args['category'] = [ $page->slug ];
			}

			if ( 'product_tag' === $page->taxonomy ) {
				$args['tag'] = [ $page->slug ];
			}

			$query    = new WC_Product_Query( $args );
			$products = $query->get_products();

			$count = count( $products );
		}

		return $count;
	}

	/**
	 * Get taxonomy data.
	 *
	 * @param array $tax_data Taxonomy data.
	 * @return array
	 */
	public static function get_ajax_tax_data( $tax_data ) {
		$ajax_tax_data = [];

		if ( empty( $tax_data ) && ! is_array( $tax_data ) ) {
			return $ajax_tax_data;
		}

		foreach ( $tax_data as $tax_datum ) {
			if ( ! empty( $tax_datum->slug ) && ! empty( $tax_datum->name ) ) {
				$ajax_tax_data[ $tax_datum->slug ] = str_replace( '"', '\'', $tax_datum->name );
			}
		}

		return $ajax_tax_data;
	}

	/**
	 * Recursive function to generate the filter list with hierarchy.
	 *
	 * @param array  $data Array of required data.
	 * @param string $input Type of input field for the filter (e.g., checkbox).
	 * @param array  $ajax_data Data for AJAX filtering.
	 * @param int    $parent ID of the parent term (default: 0).
	 *
	 * @return string The generated HTML for the product filter list.
	 */
	public static function get_product_filter_list_html( $data, $input = 'checkbox', $ajax_data = [], $parent = 0 ) {
		$html               = '';
		$default_tax        = ! empty( $_GET[ $ajax_data['taxonomy'] ] ) ? explode( ',', sanitize_text_field( wp_unslash( $_GET[ $ajax_data['taxonomy'] ] ) ) ) : []; // phpcs:ignore WordPress.Security.NonceVerification.Recommended
		$is_attribute       = ! empty( $data['is_attribute'] );
		$exclude_categories = ! empty( $data['exclude_categories'] ) ? $data['exclude_categories'] : '';
		$counter            = 0;

		if ( $is_attribute ) {
			list(
				'filter_name' => $filter_name,
				'base_link' => $base_link
				) = self::get_attribute_term_info( $ajax_data['taxonomy'] );
		}

		$tax_name = wp_json_encode(
			[
				$ajax_data['taxonomy'] => get_taxonomy( $ajax_data['taxonomy'] )->label,
			],
			JSON_UNESCAPED_UNICODE
		);

		$html .= '<ul class="product-filters input-type-' . esc_attr( $input ) . '" data-taxonomy-name="' . esc_attr( esc_js( $tax_name ) ) . '" data-rtsb-filter=\'' . esc_attr( esc_js( wp_json_encode( $ajax_data, JSON_UNESCAPED_UNICODE ) ) ) . '\'>';

		foreach ( $data['taxonomy'] as $tax ) {
			if ( $tax->parent !== $parent ) {
				continue;
			}

			if ( $is_attribute ) {
				$tax_link    = remove_query_arg( $filter_name, $base_link );
				$tax_link    = add_query_arg( $filter_name, $tax->slug, $tax_link );
				$default_tax = isset( $_GET[ $filter_name ] ) ? explode( ',', sanitize_text_field( wp_unslash( $_GET[ $filter_name ] ) ) ) : []; // phpcs:ignore WordPress.Security.NonceVerification.Recommended
			} else {
				$tax_link = get_term_link( $tax );
			}

			$unique_id         = substr( uniqid(), -4 );
			$active_tax        = in_array( $tax->slug, $default_tax, true ) ? ' checked' : '';
			$active_tax_parent = in_array( $tax->slug, $default_tax, true ) ? ' active' : '';
			$term_children     = in_array( $tax->taxonomy, [ 'product_cat','product_brand' ], true ) ? get_term_children( $tax->term_id, $tax->taxonomy ) : [];

			// Exclude Categories.
			$exclude_categories  = str_replace( ' ', '', esc_html( $exclude_categories ) );
			$excluded_categories = explode( ',', $exclude_categories );

			if ( ! empty( $exclude_categories ) && in_array( (string) $tax->term_id, $excluded_categories, true ) ) {
				continue;
			}

			$product_count = $tax->count;

			// Show count.
			if ( 'archive' === apply_filters( 'rtsb/builder/set/current/page/type', '' ) && 'block' === $data['count_display'] ) {
				$product_count = self::count_tax_data( $tax, $tax->count );
			}

			// Show Empty Categories.
			if ( empty( $data['show_empty_terms'] ) && 0 === $product_count ) {
				continue;
			}

			$html .= '<li class="rtsb-term-item term-item-' . absint( $tax->term_id ) . esc_attr( $term_children ? ' term-has-children' : '' ) . ( $counter >= absint( $data['additional_items'] ) ? esc_attr( ' additional-option' ) : '' ) . ( ! $data['show_child'] ? ' child-terms-hidden' : '' ) . '">
                <div class="rtsb-filter-group' . esc_attr( $active_tax_parent ) . '">
                    <input type="' . esc_attr( $input ) . '" class="rtsb-ajax-filter-trigger rtsb-' . esc_attr( $input . '-filter rtsb-term-' . $tax->slug . $active_tax ) . '" data-href="' . esc_url_raw( $tax_link ) . '" name="rtsb-' . esc_attr( $input ) . '[]" id="rtsb-term-filter-' . $unique_id . absint( $tax->term_id ) . '" value="' . esc_attr( $tax->slug ) . '">
                    <label class="rtsb-' . esc_attr( $input ) . '-filter-label" for="rtsb-term-filter-' . $unique_id . absint( $tax->term_id ) . '">
                        <span class="name">' . esc_html( $tax->name ) . '</span>
                    </label>
                    <div class="rtsb-count">(' . esc_html( $product_count ) . ')</div>
                    ' . ( $term_children ? '<i class="plus-icon"></i>' : '' ) . '
                </div>';

			// Show Sub-categories.
			if ( $data['show_child'] ) {
				$children = self::get_product_filter_list_html( $data, $input, $ajax_data, $tax->term_id );

				if ( ! empty( $children ) ) {
					$html .= '<div class="filter-child">' . $children . '</div>';
				}
			}

			$counter++;

			$html .= '</li>';
		}

		$html .= '</ul>';

		// Check if the output contains any child elements.
		$has_children = strpos( $html, '<li class="rtsb-term-item' ) !== false;

		if ( ! $has_children ) {
			// Remove the outer <ul> if there are no child elements.
			$html = '';
		}

		return $html;
	}

	/**
	 * Get attribute filter HTML.
	 *
	 * @param array  $data Array of required data.
	 * @param string $input Type of input field for the filter (e.g., color).
	 * @param array  $ajax_data Data for AJAX filtering.
	 *
	 * @return string The generated HTML for the attribute filter list.
	 */
	public static function get_attribute_filter_list_html( $data, $input = 'color', $ajax_data = [] ) {
		if ( ! function_exists( 'rtwpvs' ) ) {
			return '';
		}

		$terms            = $data['terms'];
		$term_info        = $data['term_info'];
		$exclude_terms    = ! empty( $data['exclude_terms'] ) ? $data['exclude_terms'] : '';
		$show_tooltips    = $data['show_tooltips'];
		$show_empty_terms = $data['show_empty_terms'];
		$additional_items = $data['additional_items'];
		$counter          = 0;

		$excluded_term_ids = ! empty( $exclude_terms ) ? array_map( 'intval', explode( ',', $exclude_terms ) ) : [];

		$tax_name = wp_json_encode(
			[
				$ajax_data['taxonomy'] => get_taxonomy( $ajax_data['taxonomy'] )->label,
			],
			JSON_UNESCAPED_UNICODE
		);

		$html = '<ul class="product-filters rtsb-terms-wrapper ' . esc_attr( $term_info['type'] ) . '-variable-wrapper' . esc_attr( $data['show_label'] ) . '" data-taxonomy-name="' . esc_attr( esc_js( $tax_name ) ) . '" data-rtsb-filter=\'' . esc_attr( esc_js( wp_json_encode( $ajax_data, JSON_UNESCAPED_UNICODE ) ) ) . '\'>';

		foreach ( $terms as $attr_term ) {
			$count          = $attr_term->count;
			$term_link      = remove_query_arg( $term_info['filter_name'], $term_info['base_link'] );
			$term_link      = add_query_arg( $term_info['filter_name'], $attr_term->slug, $term_link );
			$unique_id      = substr( uniqid(), -6 );
			$name           = 'term-' . wc_variation_attribute_name( $term_info['attribute'] ) . '-' . $unique_id;
			$selected_class = in_array( $attr_term->slug, $term_info['current_filter'], true ) ? ' selected' : '';
			$active_class   = in_array( $attr_term->slug, $term_info['current_filter'], true ) ? ' active' : '';

			$args = [
				'id'        => $name,
				'name'      => $attr_term->name,
				'link'      => $term_link,
				'type'      => $term_info['type'],
				'term_id'   => $attr_term->term_id,
				'term_slug' => $attr_term->slug,
				'taxonomy'  => wc_variation_attribute_name( $term_info['attribute'] ),
				'tooltips'  => $show_tooltips,
			];

			if ( 'archive' === apply_filters( 'rtsb/builder/set/current/page/type', '' ) && 'block' === $data['count_display'] ) {
				$count = self::count_tax_data( $args, $attr_term->count );
			}

			$args['count'] = $count;

			if ( ! $show_empty_terms && 0 === $count ) {
				continue;
			}

			if ( in_array( $attr_term->term_id, $excluded_term_ids, true ) ) {
				continue;
			}

			$html .= '<li class="rtsb-term-item term-item-' . esc_attr( $attr_term->term_id ) . ' rtsb-term rtsb-' . esc_attr( $input ) . '-term ' . esc_attr( $term_info['type'] ) . '-variable-term-' . esc_attr( $attr_term->slug ) . esc_attr( $selected_class ) . ( $counter >= absint( $additional_items ) ? esc_attr( ' additional-option' ) : '' ) . '" data-term="' . esc_attr( $attr_term->slug ) . '">';
			$html .= '<div class="rtsb-filter-group' . esc_attr( $active_class ) . '">';
			$html .= self::get_input_type_html( $input, $args );

			$counter++;

			$html .= '</div>';
			$html .= '</li>';
		}

		$html .= '</ul>';

		return $html;
	}

	/**
	 * Get filter HTML.
	 *
	 * @param string $filter_type Filter type.
	 * @param array  $options Options.
	 * @param array  $ajax_data Data for AJAX filtering.
	 * @param string $input Input type.
	 *
	 * @return string The generated HTML for the attribute filter list.
	 */
	public static function get_filter_html( $filter_type, $options, $ajax_data = [], $input = 'checkbox' ) {
		$html = '<ul class="product-filters input-type-' . esc_attr( $input ) . '" data-taxonomy-name="' . esc_attr( $ajax_data['taxonomy'] ) . '" data-rtsb-filter=\'' . esc_js( wp_json_encode( $ajax_data ) ) . '\'>';

		$active_option = isset( $_GET[ $filter_type ] ) ? explode( ',', wc_clean( wp_unslash( $_GET[ $filter_type ] ) ) ) : [];
		$option_link   = self::get_base_url() . '/' . $filter_type . '/';

		foreach ( $options as $option => $option_name ) {
			$unique_id    = substr( uniqid(), -6 );
			$active_tax   = in_array( $option, $active_option, true ) ? ' checked' : '';
			$active_group = in_array( $option, $active_option, true ) ? ' active' : '';

			$html .= '
					<li class="rtsb-term-item">
						<div class="rtsb-filter-group' . esc_attr( $active_group ) . '">
							<input type="' . esc_attr( $input ) . '" class="rtsb-ajax-filter-trigger rtsb-' . esc_attr( $input ) . '-filter rtsb-term-' . esc_attr( $option ) . esc_attr( $active_tax ) . '" data-href="' . esc_url( $option_link ) . esc_attr( $option ) . '/" name="rtsb-' . esc_attr( $input ) . '[]" id="rtsb-term-filter-' . $unique_id . '" value="' . esc_attr( $option ) . '">
							<label class="rtsb-' . esc_attr( $input ) . '-filter-label" for="rtsb-term-filter-' . $unique_id . '">
								<span class="name">' . esc_html( $option_name ) . '</span>
							</label>
						</div>
					</li>';
		}

		$html .= '</ul>';

		return $html;
	}

	/**
	 * Get input type HTML.
	 *
	 * @param string $input Input type.
	 * @param array  $args Args.
	 * @return mixed|string|null
	 */
	public static function get_input_type_html( $input, $args ) {
		$html  = '';
		$count = $args['count'];

		switch ( $input ) {
			case 'color':
				$color = sanitize_hex_color( get_term_meta( $args['term_id'], 'product_attribute_color', true ) );

				$html .= sprintf(
					'<a id="%s" href="%s" rel="nofollow" class="rtsb-ajax-filter-trigger rtsb-attr-filter rtsb-color-filter rtsb-term-span rtsb-term-span-%s%s" title="' . esc_attr( $args['name'] ) . '"><span class="filter-attr-color" style="background-color:%s;"></span><span class="filter-attr-name">%s</span></a><div class="rtsb-count">(%s)</div>',
					esc_attr( $args['id'] ),
					$args['link'],
					esc_attr( $args['type'] ),
					$args['tooltips'] ? esc_attr( ' tipsy' ) : '',
					esc_attr( $color ),
					esc_html( $args['name'] ),
					absint( $count )
				);
				break;

			case 'image':
				if ( ! function_exists( 'rtwpvs' ) ) {
					return $html;
				}

				$attachment_id = absint( get_term_meta( $args['term_id'], 'product_attribute_image', true ) );
				$image_size    = rtwpvs()->get_option( 'attribute_image_size' );
				$image_url     = wp_get_attachment_image_url( $attachment_id, apply_filters( 'rtwpvs_product_attribute_image_size', $image_size ) );

				$html .= sprintf(
					'<a id="%s" href="%s" rel="nofollow" class="rtsb-ajax-filter-trigger rtsb-image-filter rtsb-term-span rtsb-term-span-%s%s" title="' . esc_attr( $args['name'] ) . '"><img class="rtsb-attr-filter" alt="%s" src="%s" /></a>',
					esc_attr( $args['id'] ),
					$args['link'],
					esc_attr( $args['type'] ),
					$args['tooltips'] ? esc_attr( ' tipsy' ) : '',
					esc_attr( $args['name'] ),
					esc_url( $image_url )
				);
				break;

			case 'button':
				$html .= sprintf(
					'<a id="%s" href="%s" rel="nofollow" class="rtsb-ajax-filter-trigger rtsb-attr-filter rtsb-button-filter rtsb-term-span rtsb-term-span-%s%s" title="' . esc_attr( $args['name'] ) . '"><span class="filter-attr-name">%s</span><div class="rtsb-count">(%s)</div></a>',
					esc_attr( $args['id'] ),
					$args['link'],
					esc_attr( $args['type'] ),
					$args['tooltips'] ? esc_attr( ' tipsy' ) : '',
					esc_html( $args['name'] ),
					absint( $count )
				);
				break;

			default:
		}

		return apply_filters( 'rtsb/elementor/filter/get_input_type_html', $html );
	}

	/**
	 * Get filtered product price.
	 *
	 * @return array
	 */
	public static function get_filtered_price() {
		global $wpdb;

		$args = wc()->query->get_main_query();

		if ( empty( $args ) ) {
			return [
				'min' => 0,
				'max' => 1000,
			];
		}

		$tax_query  = isset( $args->tax_query->queries ) ? $args->tax_query->queries : [];
		$meta_query = isset( $args->query_vars['meta_query'] ) ? $args->query_vars['meta_query'] : [];

		foreach ( $meta_query + $tax_query as $key => $query ) {
			if ( ! empty( $query['price_filter'] ) || ! empty( $query['rating_filter'] ) ) {
				unset( $meta_query[ $key ] );
			}
		}

		$meta_query = new \WP_Meta_Query( $meta_query );
		$tax_query  = new \WP_Tax_Query( $tax_query );

		$meta_query_sql = $meta_query->get_sql( 'post', $wpdb->posts, 'ID' );
		$tax_query_sql  = $tax_query->get_sql( $wpdb->posts, 'ID' );

		$sql  = "SELECT min( FLOOR( price_meta.meta_value ) ) as min_price, max( CEILING( price_meta.meta_value ) ) as max_price FROM {$wpdb->posts} ";
		$sql .= " LEFT JOIN {$wpdb->postmeta} as price_meta ON {$wpdb->posts}.ID = price_meta.post_id " . $tax_query_sql['join'] . $meta_query_sql['join'];
		$sql .= " 	WHERE {$wpdb->posts}.post_type IN ('product')
			AND {$wpdb->posts}.post_status = 'publish'
			AND price_meta.meta_key IN ('_price')
			AND price_meta.meta_value > '' ";
		$sql .= $tax_query_sql['where'] . $meta_query_sql['where'];

		$search = \WC_Query::get_main_search_query_sql();
		if ( $search ) {
			$sql .= ' AND ' . $search;
		}

		$prices = $wpdb->get_row( $sql );

		return [
			'min' => ! empty( $prices->min_price ) ? floor( $prices->min_price ) : 0,
			'max' => ! empty( $prices->max_price ) ? ceil( $prices->max_price ) : 0,
		];
	}

	/**
	 * Get Custom query product price.
	 *
	 * @param object $query Query object.
	 *
	 * @return array
	 */
	public static function get_custom_filtered_price( $query ) {
		global $wpdb;

		$tax_query  = isset( $query->tax_query->queries ) ? $query->tax_query->queries : [];
		$meta_query = isset( $query->query_vars['meta_query'] ) ? $query->query_vars['meta_query'] : [];

		foreach ( $meta_query + $tax_query as $key => $query ) {
			if ( ! empty( $query['price_filter'] ) || ! empty( $query['rating_filter'] ) ) {
				unset( $meta_query[ $key ] );
			}
		}

		$meta_query = new \WP_Meta_Query( $meta_query );
		$tax_query  = new \WP_Tax_Query( $tax_query );

		$meta_query_sql = $meta_query->get_sql( 'post', $wpdb->posts, 'ID' );
		$tax_query_sql  = $tax_query->get_sql( $wpdb->posts, 'ID' );

		$sql = "SELECT min( FLOOR( price_meta.meta_value ) ) as min_price, max( CEILING( price_meta.meta_value ) ) as max_price
                FROM {$wpdb->posts}
                LEFT JOIN {$wpdb->postmeta} as price_meta ON {$wpdb->posts}.ID = price_meta.post_id
                " . $tax_query_sql['join'] . $meta_query_sql['join'] . "
                WHERE {$wpdb->posts}.post_type = 'product'
                AND {$wpdb->posts}.post_status = 'publish'
                AND price_meta.meta_key = '_price'
                AND price_meta.meta_value > ''
                " . $tax_query_sql['where'] . $meta_query_sql['where'];

		$search = '';

		if ( isset( $query->query_vars['s'] ) ) {
			$search = $wpdb->prepare( "(({$wpdb->posts}.post_title LIKE '%%%s%%') OR ({$wpdb->posts}.post_content LIKE '%%%s%%'))", $query->query_vars['s'], $query->query_vars['s'] );
		}

		if ( $search ) {
			$sql .= ' AND ' . $search;
		}

		$prices = $wpdb->get_row( $sql );

		return [
			'min' => ! empty( $prices->min_price ) ? floor( $prices->min_price ) : 0,
			'max' => ! empty( $prices->max_price ) ? ceil( $prices->max_price ) : 0,
		];
	}

	/**
	 * Get product min/max price.
	 *
	 * @return array|false
	 */
	public static function get_product_min_max_prices() {
		global $wpdb;

		$cache_key = 'product_min_max_prices';
		$prices    = wp_cache_get( $cache_key, 'shopbuilder' );

		$table_name = $wpdb->prefix . 'postmeta';
		$meta_key   = '_price';

		if ( false === $prices ) {
			$table_name = esc_sql( $table_name );
			$meta_key   = esc_sql( $meta_key );

			$prices = $wpdb->get_row(
				$wpdb->prepare(
					"SELECT MIN(meta_value+0) as min_price, MAX(meta_value+0) as max_price FROM $table_name WHERE meta_key = %s",
					$meta_key
				)
			);

			wp_cache_set( $cache_key, $prices, 'shopbuilder', 24 * HOUR_IN_SECONDS );
			Cache::set_data_cache_key( $cache_key );
		}

		if ( ! empty( $prices ) ) {
			return [
				'min_price' => floatval( $prices->min_price ),
				'max_price' => floatval( $prices->max_price ),
			];
		}

		return [
			'min_price' => 0,
			'max_price' => 1000,
		];
	}





	/**
	 * Get reset button.
	 *
	 * @param string $btn_text Button text.
	 * @param array  $settings Settings array.
	 *
	 * @return string
	 */
	public static function get_filter_reset_button( $btn_text, $settings ) {
		$html = '';

		if ( empty( $settings['apply_filters_btn'] ) && empty( $settings['reset_btn'] ) ) {
			return $html;
		}

		ob_start();

		$ajax_data = [
			'filter_reset' => true,
		];

		$active        = ! empty( $_SERVER['QUERY_STRING'] ) ? ' active' : '';
		$apply_filters = ! empty( $settings['apply_filters_btn'] );
		$reset         = ! empty( $settings['reset_btn'] );
		$apply_text    = ! empty( $settings['apply_filters_btn_text'] ) ? $settings['apply_filters_btn_text'] : esc_html__( 'Apply Filters', 'shopbuilder-pro' );
		$apply_icon    = ! empty( $settings['apply_filters_btn_icon'] ) ? $settings['apply_filters_btn_icon'] : [];

		echo '<div class="filter-btn-wrapper">';

		if ( $apply_filters ) {
			?>
			<div class="rtsb-product-filters rtsb-apply-filters-btn">
				<div class="reset-btn-item">
					<button class="rtsb-apply-filters init">
						<span class="icon"><?php Fns::print_html( Fns::icons_manager( $apply_icon, 'icon-default' ) ); ?></span>
						<span class="text reset-text"><?php echo esc_html( $apply_text ); ?></span>
						<span></span>
					</button>
				</div>
			</div>
			<?php
		}

		if ( $reset ) {
			?>
			<div class="rtsb-product-filters rtsb-reset<?php echo esc_attr( $active ); ?>">
				<div class="reset-btn-item" data-rtsb-filter="<?php echo esc_js( wp_json_encode( $ajax_data ) ); ?>">
					<button class="product-filter-reset">
						<span class="icon"><i class="fas fa-undo"></i></span>
						<span class="text reset-text"><?php echo esc_html( $btn_text ); ?></span>
						<span></span>
					</button>
				</div>
			</div>
			<?php
		}
		echo '</div>';

		$html .= ob_get_clean();

		return $html;
	}

	/**
	 * Get the more/less button.
	 *
	 * @param string $tax Tax type.
	 * @param array  $settings Raw settings.
	 * @param array  $args Args.
	 *
	 * @return void
	 */
	public static function get_more_less_button( $tax, $settings, $args = [] ) {
		if ( $args['total_count'] <= $args['additional_count'] ) {
			return;
		}

		if ( ! is_shop() && is_product_taxonomy() && ! empty( $args['tax_data'] ) ) {
			$count = 0;

			foreach ( $args['tax_data'] as $taxo ) {
				if ( self::count_tax_data( $taxo, $args['total_count'] ) >= 0 ) {
					$count++;
				}
			}

			if ( $count <= $args['additional_count'] ) {
				return;
			}
		}

		if ( ! empty( $settings['more_less_btn'] ) ) {
			$moreText = ! empty( $settings['more_text'] ) ? $settings['more_text'] : esc_html__( 'More', 'shopbuilder-pro' );

			switch ( $tax ) {
				case 'product_cat':
				case 'product_tag':
				case 'product_attr':
					echo '<div class="more-less-button" data-additional-count="' . esc_attr( $args['additional_count'] ) . '">' . esc_html( $moreText ) . '</div>';
					break;
				default:
					break;
			}
		}
	}

	/**
	 * Get the count of products with a specific star rating.
	 *
	 * @param int $star_rating The star rating value.
	 *
	 * @return int
	 */
	public static function get_product_rating_count( $star_rating ) {
		$args = [
			'status'         => 'publish',
			'limit'          => -1,
			'product_rating' => sprintf( '%.1f', $star_rating ),
		];

		if ( is_product_taxonomy() ) {
			$term      = get_queried_object();
			$term_type = 'product_cat' === $term->taxonomy ? 'product_category_id' : 'product_tag_id';

			if ( $term instanceof WP_Term ) {
				$args[ $term_type ] = [ $term->term_id ];
			}
		}

		$query    = new WC_Product_Query( $args );
		$products = $query->get_products();

		return count( $products );
	}

	/**
	 * Get the product count for a specific term in a taxonomy.
	 *
	 * @param string $tax_slug The taxonomy slug.
	 * @param string $term_id The term ID.
	 *
	 * @return int
	 */
	public static function get_attribute_product_count( $tax_slug, $term_id ) {
		$term = get_term( $term_id, $tax_slug );

		if ( ! $term || is_wp_error( $term ) ) {
			return 0;
		}

		if ( ! is_shop() && is_product_taxonomy() ) {
			$page = get_queried_object();

			if ( ! $page instanceof WP_Term ) {
				return 0;
			}

			if ( strpos( $page->taxonomy, 'pa_' ) !== false ) {
				return $term->count;
			}

			$args = [];

			if ( 'product_cat' === $page->taxonomy ) {
				$args['category'] = [ $page->slug ];
			}

			if ( 'product_tag' === $page->taxonomy ) {
				$args['tag'] = [ $page->slug ];
			}

			$products   = wc_get_products( $args );
			$attr_terms = [];

			foreach ( $products as $product ) {
				$product_attr = wp_get_post_terms( $product->get_id(), $term->taxonomy );

				foreach ( $product_attr as $attr ) {
					$attr_terms[] = $attr;
				}
			}

			return count( $attr_terms );
		} else {
			return $term->count;
		}
	}


	/**
	 * @param $link
	 * @param $cart_item
	 * @param $cart_item_key
	 * @param $table
	 *
	 * @return mixed|string
	 */
	public static function woocommerce_cart_item_name( $link, $cart_item, $cart_item_key, $table ) {

		$_product = apply_filters( 'woocommerce_cart_item_product', $cart_item['data'], $cart_item, $cart_item_key );
		if ( ! is_object( $_product ) ) {
			return $link; // Return original link if $_product is not an object.
		}

		$product_permalink = apply_filters(
			'woocommerce_cart_item_permalink',
			$_product->is_visible() ? $_product->get_permalink( $cart_item ) : '',
			$cart_item,
			$cart_item_key
		);
		$main_product_id   = $_product->get_parent_id();

		// Get the main product object.
		if ( $main_product_id ) {
			$main_product_name = get_the_title( $main_product_id );
			$link              = '<div class="rtsb-cart-item-link"><a href="' . esc_url( $product_permalink ) . '">' . $main_product_name . '</a></div>';
		}

		$link = apply_filters( 'rtsb/element/cart/table/item_name', $link, $cart_item, $cart_item_key, $table );

		$sku            = $_product->get_sku();
		$show_addons    = Fns::get_options( 'modules', 'product_add_ons' )['add_ons_show_in_cart'] ?? false;
		$addons_active  = self::is_module_active( 'product_add_ons' );
		$partial_active = self::is_module_active( 'partial_pay' );

		$link .= '<div class="product-attributes-wrapper" data-title="' . esc_html__( 'Attributes', 'shopbuilder-pro' ) . '" ><ul class="product-attributes" >';
		if ( ! empty( $sku ) && ! empty( $table['show_sku'] ) ) {
			$link .= '<li>' . esc_html__( 'SKU', 'shopbuilder-pro' ) . ': <span>' . esc_html( $sku ) . '</span></li>';
		}

		if ( $main_product_id && ! empty( $cart_item['variation'] ) && ! empty( $table['show_variation_data'] ) ) {
			// Assuming $cart_item is the cart item object.
			foreach ( $cart_item['variation'] as $attribute_name => $attribute_value ) {
				$updated_attribute_name = str_replace( [ 'attribute_pa_', 'attribute_' ], '', strtolower( urldecode( $attribute_name ) ) );
				$link                  .= '<li>' . ucfirst( $updated_attribute_name ) . ': ' . ucfirst( $attribute_value ) . '</li>';
			}
		}

		if ( $addons_active && $show_addons && ! empty( $cart_item['rtsb_product_addons'] ) ) {
			$addon_data = Cart::instance()->display_addon_data( [], $cart_item );

			foreach ( $addon_data as $addon ) {
				$link .= '<li class="rtsb-addon-data"><span class="addon-name">' . $addon['name'] . ':</span> <span class="addon-value">' . $addon['value'] . '</span></li>';
			}
		}

		if ( $partial_active && ! empty( $cart_item['rtsb_partial_pay_installments'] ) ) {
			$partial_data = CartPage::instance()->display_partial_pay_data( [], $cart_item );

			foreach ( $partial_data as $partial ) {
				$link .= '<li class="rtsb-addon-data partial-data"><span class="addon-name">' . $partial['name'] . ':</span> <span class="addon-value">' . $partial['value'] . '</span></li>';
			}
		}

		$link .= '</ul></div>';

		return $link;
	}

	/**
	 * Highlight the search term in the given content.
	 *
	 * @param string $content     The content to search and highlight.
	 * @param string $search_term The search term to highlight.
	 *
	 * @return string
	 */
	public static function highlight_search_term( $content, $search_term ) {
		$search_term_escaped = preg_quote( $search_term, '/' );

		return preg_replace( '/' . $search_term_escaped . '/i', '<span class="rtsb-search-highlight">$0</span>', $content );
	}

	/**
	 * Get the search term from the request parameters.
	 *
	 * @return string
	 */
	public static function get_search_term() {
		// phpcs:ignore WordPress.Security.NonceVerification.Recommended
		$search_term = ! empty( $_REQUEST['s'] ) ? sanitize_text_field( wp_unslash( $_REQUEST['s'] ) ) : '';
		// phpcs:ignore WordPress.Security.NonceVerification.Recommended
		$search_term = ! empty( $_REQUEST['terms']['search'] ) ? sanitize_text_field( wp_unslash( $_REQUEST['terms']['search'] ) ) : $search_term;

		return $search_term;
	}

	/**
	 * Check if checkout builder page has multistep checkout widget.
	 *
	 * @return bool
	 */
	public static function checkout_page_has_multi_step_widget() {
		$has_widgets = false;
		$id          = BuilderFns::is_builder_preview() ? get_the_ID() : BuilderFns::builder_page_id_by_type( 'checkout' );
		$cache_key   = 'checkout_page_has_multi_step_widget' . $id;
		if ( isset( self::$cache[ $cache_key ] ) ) {
			return self::$cache[ $cache_key ];
		}
		$elmap  = ElementorDataMap::instance();
		$widget = $elmap->get_widget_data( 'rtsb-multi-step-checkout', [], $id );
		if ( is_array( $widget ) && ! empty( $widget ) ) {
			$has_widgets = true;
		}
		self::$cache[ $cache_key ] = $has_widgets;

		return $has_widgets;
	}
	/**
	 * Generate custom order for specified elements.
	 *
	 * @param array $items An array of items with ordering information.
	 *
	 * @return string
	 */
	public static function generate_custom_order( $items ) {
		$order = '';

		if ( empty( $items ) ) {
			return $order;
		}

		foreach ( $items as $key => $item ) {
			$number = $key + 1;

			switch ( $item['ordering_type'] ) {
				case 'swatches':
					$order .= '--rtsb-swatches-order: ' . absint( $number ) . ';';
					break;

				case 'categories':
					$order .= '--rtsb-category-order: ' . absint( $number ) . ';';
					break;

				case 'title':
					$order .= '--rtsb-title-order: ' . absint( $number ) . ';';
					break;

				case 'rating':
					$order .= '--rtsb-rating-order: ' . absint( $number ) . ';';
					break;

				case 'description':
					$order .= '--rtsb-description-order: ' . absint( $number ) . ';';
					break;

				case 'price':
					$order .= '--rtsb-price-order: ' . absint( $number ) . ';';
					break;
				case 'stock':
					$order .= '--rtsb-stock-order: ' . absint( $number ) . ';';
					break;
			}
		}
		return $order;
	}
	/**
	 * Generate custom order for highlighted products.
	 *
	 * @param array $items An array of items with ordering information.
	 *
	 * @return string
	 */
	public static function generate_highlighted_product_custom_order( $items ) {
		$order = '';

		if ( empty( $items ) ) {
			return $order;
		}

		foreach ( $items as $key => $item ) {
			$number = $key + 1;

			switch ( $item['ordering_type'] ) {
				case 'meta':
					$order .= '--rtsb-meta-order: ' . absint( $number ) . ';';
					break;
				case 'sku':
					$order .= '--rtsb-sku-order: ' . absint( $number ) . ';';
					break;
				case 'add_to_cart':
					$order .= '--rtsb-add-to-cart-order: ' . absint( $number ) . ';';
					break;
				case 'title':
					$order .= '--rtsb-title-order: ' . absint( $number ) . ';';
					break;
				case 'description':
					$order .= '--rtsb-description-order: ' . absint( $number ) . ';';
					break;
				case 'price':
					$order .= '--rtsb-price-order: ' . absint( $number ) . ';';
					break;
				case 'stock':
					$order .= '--rtsb-stock-order: ' . absint( $number ) . ';';
					break;
				case 'action_buttons':
					$order .= '--rtsb-action-buttons-order: ' . absint( $number ) . ';';
					break;
			}
		}
		return $order;
	}

	/**
	 * Checks if the current page is one of the My Account pages.
	 *
	 * @return bool
	 */
	public static function is_my_account_page() {
		$custom_endpoint = apply_filters( 'rtsb/editor/enable/custom_endpoint', false );

		return is_user_logged_in() && ( BuilderFunPro::is_account_dashboard() ||
			BuilderFunPro::is_orders_page() ||
			BuilderFunPro::is_orders_details_page() ||
			BuilderFunPro::is_downloads_page() ||
			BuilderFunPro::is_account_edit_billing_address() ||
			BuilderFunPro::is_account_edit_shipping_address() ||
			BuilderFunPro::is_address_page() ||
			BuilderFunPro::is_account_details() ||
			BuilderFunPro::is_account_login_register() ||
			BuilderFunPro::is_lost_password_page() ) ||
			$custom_endpoint;
	}

	/**
	 * Is_wc_endpoint_url - Check if an endpoint is showing.
	 *
	 * @param string|false $endpoint Whether endpoint.
	 * @return bool
	 */
	public static function is_wc_endpoint_url( $endpoint = false ) {
		global $wp;

		$wc_endpoints = WC()->query->get_query_vars();

		if ( false !== $endpoint ) {
			if ( ! isset( $wc_endpoints[ $endpoint ] ) ) {
				return isset( $wp->query_vars[ $endpoint ] );
			} else {
				$endpoint_var = $wc_endpoints[ $endpoint ];
			}

			return isset( $wp->query_vars[ $endpoint_var ] );
		} else {
			foreach ( $wc_endpoints as $key => $value ) {
				if ( isset( $wp->query_vars[ $key ] ) ) {
					return true;
				}
			}

			return false;
		}
	}

	/**
	 * Update repeater options with a new value based on the provided keys.
	 *
	 * @param string $section_id The section ID.
	 * @param string $block_id The block ID.
	 * @param string $option_key The option key.
	 * @param string $compare_key The key where the new value will be checked.
	 * @param string $update_key The key to update.
	 * @param mixed  $update_value The new value to set.
	 * @return void
	 */
	public static function update_repeater_options( $section_id, $block_id, $option_key, $compare_key, $update_key, $update_value ) {
		$modules = DataModel::source()->get_option( $section_id, [], false );

		if ( empty( $modules[ $block_id ] ) ) {
			return;
		}

		$options = $modules[ $block_id ];

		if ( empty( $options[ $option_key ] ) ) {
			return;
		}

		$repeater_options = json_decode( $options[ $option_key ], true );

		if ( ! is_array( $repeater_options ) ) {
			return;
		}

		$repeater_options = array_reduce(
			$repeater_options,
			function ( $carry, $element ) {
				if ( isset( $element['uniqueid'] ) ) {
					$carry[ $element['uniqueid'] ] = $element;
				}
				return $carry;
			},
			[]
		);

		if ( isset( $repeater_options[ $compare_key ] ) ) {
			$repeater_options[ $compare_key ][ $update_key ] = $update_value;
		}

		$repeater_options       = array_values( $repeater_options );
		$options[ $option_key ] = wp_json_encode( $repeater_options );

		Fns::set_options( $section_id, $block_id, $options );
	}

	/**
	 * Get all user roles.
	 *
	 * @return array|false|mixed
	 */
	public static function get_all_user_roles() {
		if ( method_exists( Fns::class, 'get_all_user_roles' ) ) {
			return Fns::get_all_user_roles();
		}
		// TODO: Below Code Will Remove In the future.
		$cache_key = 'rtsb_get_user_roles';
		$roles     = wp_cache_get( $cache_key, 'shopbuilder' );

		if ( empty( $roles ) ) {
			if ( ! function_exists( 'get_editable_roles' ) ) {
				require_once ABSPATH . 'wp-admin/includes/user.php';
			}

			$user_roles = \get_editable_roles();
			$roles      = [];

			foreach ( $user_roles as $key => $role ) {
				if ( empty( $role['capabilities'] ) ) {
					continue;
				}
				$roles[ $key ] = $role['name'];
			}
		}

		wp_cache_set( $cache_key, $roles, 'shopbuilder' );
		Cache::set_data_cache_key( $cache_key );

		return $roles;
	}

	/**
	 * Retrieves all available WooCommerce shipping methods.
	 *
	 * @return array
	 */
	public static function get_all_shipping_methods() {
		$cache_key        = 'rtsb_get_shipping_methods';
		$shipping_methods = wp_cache_get( $cache_key, 'shopbuilder' );

		if ( empty( $shipping_methods ) ) {
			if ( ! function_exists( 'WC' ) ) {
				return [];
			}

			$shipping_methods  = [];
			$available_methods = WC()->shipping()->get_shipping_methods();

			foreach ( $available_methods as $method_id => $method ) {
				$shipping_methods[ $method_id ] = $method->get_method_title();
			}

			wp_cache_set( $cache_key, $shipping_methods, 'shopbuilder' );
			Cache::set_data_cache_key( $cache_key );
		}

		return $shipping_methods;
	}

	/**
	 * Checks if a module is active.
	 *
	 * @param string $module Module name.
	 * @return bool
	 */
	public static function is_module_active( $module ) {
		return 'on' === ( Fns::get_options( 'modules', $module )['active'] ?? '' );
	}

	/**
	 * Check if the current user has visibility based on their role.
	 *
	 * @param array $roles An array of roles to check against.
	 *
	 * @return bool
	 */
	public static function is_visible_for_user( $roles = [ 'administrator' ] ) {
		if ( empty( $roles ) ) {
			return true;
		}

		return ! empty( array_intersect( array_merge( $roles, [ 'administrator' ] ), Fns::get_current_user_roles() ) );
	}

	/**
	 * @param $action
	 * @return void
	 */
	public static function add_to_scheduled_hook_list( $action ) {
		if ( empty( $action ) ) {
			return;
		}
		$schedule   = get_option( 'rtsb_cron_schedule', [] );
		$schedule[] = $action;
		update_option( 'rtsb_cron_schedule', array_unique( $schedule ) );
	}
	/**
	 * Generate and cache dynamic CSS styles.
	 *
	 * @param array  $options      CSS options to apply (e.g. padding, margin).
	 * @param string $cache_key    Cache key for the CSS.
	 * @param array  $css_properties An array of CSS properties with selectors and properties.
	 * @param string $style_handle  The handle for the inline styles.
	 *
	 * @return void
	 */
	public static function dynamic_styles( $options, $cache_key, $css_properties, $style_handle = 'rtsb-frontend' ) {
		$cached_css = wp_cache_get( $cache_key, 'shopbuilder' );

		if ( false !== $cached_css ) {
			wp_add_inline_style( $style_handle, $cached_css );
			return;
		}

		$css_rules         = [];
		$grouped_css_rules = [];

		foreach ( $css_properties as $option => $details ) {
			if ( ! empty( $options[ $option ] ) ) {
				$selector = $details['selector'] ?? '';
				$property = $details['property'] ?? '';
				$value    = $options[ $option ];

				if ( empty( $grouped_css_rules[ $selector ] ) ) {
					$grouped_css_rules[ $selector ] = [];
				}

				if ( is_array( $property ) ) {
					foreach ( $property as $prop ) {
						$grouped_css_rules[ $selector ][] = $prop . ': ' . $value . ' !important';
					}
				} else {
					$grouped_css_rules[ $selector ][] = $property . ': ' . $value . ' !important';
				}
			}
		}

		foreach ( $grouped_css_rules as $selector => $properties ) {
			$css_rules[] = $selector . ' {' . implode( '; ', $properties ) . '}';
		}

		$dynamic_css = implode( ' ', $css_rules );

		wp_cache_set( $cache_key, $dynamic_css, 'shopbuilder', 12 * HOUR_IN_SECONDS );
		Cache::set_data_cache_key( $cache_key );

		if ( ! empty( $dynamic_css ) ) {
			wp_add_inline_style( $style_handle, $dynamic_css );
		}
	}

	/**
	 * Get the module compatible price for a product.
	 *
	 * @param object $product The product object.
	 *
	 * @return float
	 */
	public static function get_module_compatible_price( $product ) {
		$product_id = $product->get_id();
		$full_price = floatval( $product->get_price() );

		// Flash Sale Countdown.
		if ( self::is_module_active( 'flash_sale_countdown' ) ) {
			$campaign = CountdownFns::get_campaign_for_current_product( $product );

			$price         = floatval( $product->get_price() );
			$regular_price = floatval( $product->get_regular_price() );
			$sale_price    = floatval( $product->get_sale_price() );
			$prices        = compact( 'price', 'regular_price', 'sale_price' );

			$discounted_prices = AddOnsFns::apply_campaign_discount( $prices, $campaign );

			if ( ! empty( $discounted_prices['sale_price'] ) ) {
				$full_price = floatval( $discounted_prices['sale_price'] );
			}
		}

		// Pre-Order Support.
		if ( self::is_module_active( 'pre_order' ) ) {
			$is_on_pre_order = 'variation' === $product->get_type()
				? PreOrderFns::variation_id_is_on_pre_order( $product_id )
				: PreOrderFns::is_on_pre_order( $product );

			if ( $is_on_pre_order ) {
				$full_price = AddOnsProduct::convert_currency(
					Product::calculate_preorder_price( $product_id, $full_price )
				);
			}
		}

		return $full_price;
	}
	/**
	 * Output a custom notice with a specific banner class and icon.
	 *
	 * @param string $message       The message to display.
	 * @param string $banner_class The CSS class for the banner.
	 * @param string $icon_svg      The SVG icon to display.
	 *
	 * @return void
	 */
	public static function custom_notice( $message, $banner_class = '', $icon_svg = '' ) {
		$has_icon = ! empty( $icon_svg );
		if ( empty( $icon_svg ) ) {
			$icon_svg = '<svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 24 24" width="24" height="24" aria-hidden="true" focusable="false"><path d="M12 3.2c-4.8 0-8.8 3.9-8.8 8.8 0 4.8 3.9 8.8 8.8 8.8 4.8 0 8.8-3.9 8.8-8.8 0-4.8-4-8.8-8.8-8.8zm0 16c-4 0-7.2-3.3-7.2-7.2C4.8 8 8 4.8 12 4.8s7.2 3.3 7.2 7.2c0 4-3.2 7.2-7.2 7.2zM11 17h2v-6h-2v6zm0-8h2V7h-2v2z"></path></svg>';
		}
		$html = '<div class="wc-block-components-notice-banner is-info ' . esc_attr( $banner_class ) . '" role="alert">'
			. ( $has_icon ? ' <span>' . $icon_svg . '</span> ' : $icon_svg ) .
			'<div class="wc-block-components-notice-banner__content">'
			. $message .
			'</div>
				</div>';

		Fns::print_html( $html, true );
	}
}
