<?php
/**
 * AdminInputsFns Helpers class
 *
 * @package  RadiusTheme\SBPRO
 */

namespace RadiusTheme\SBPRO\Helpers;

use RadiusTheme\SB\Helpers\Fns;

// Do not allow directly accessing this file.
if ( ! defined( 'ABSPATH' ) ) {
	exit( 'This script cannot be accessed directly.' );
}

/**
 * AdminInputsFns Helpers class
 */
class AdminInputsFns {
	/**
	 * Renders the HTML for a section title with actions based on the provided arguments.
	 *
	 * @param array $args Title args.
	 *
	 * @return string The HTML output of the title section.
	 */
	public static function render_title_with_actions( $args = [] ) {
		$defaults = [
			'type'         => 'h3',
			'title_class'  => 'addons-title',
			'title'        => esc_html__( 'Untitled', 'shopbuilder-pro' ),
			'drag_tip'     => esc_html__( 'Drag and drop to rearrange', 'shopbuilder-pro' ),
			'remove_class' => 'remove',
			'has_edit'     => true,
		];

		$args = wp_parse_args( $args, $defaults );

		ob_start();
		?>
		<<?php echo esc_html( $args['type'] ); ?> class="<?php echo esc_attr( $args['title_class'] ); ?>">
		<?php
		if ( ! empty( $args['title'] ) ) {
			?>
			<span><?php echo esc_html( $args['title'] ); ?></span>
			<?php
		}
		?>
		<div class="title-actions">
			<div class="tips sort" data-tip="<?php echo esc_attr( $args['drag_tip'] ); ?>"></div>
			<a href="#" class="remove-button <?php echo esc_attr( $args['remove_class'] ); ?>"><?php esc_html_e( 'Remove', 'shopbuilder-pro' ); ?></a>
			<?php
			if ( $args['has_edit'] ) {
				?>
				<a href="javascript:void(0)" class="edit-group edit"><?php esc_html_e( 'Edit', 'shopbuilder-pro' ); ?></a>
				<?php
			}
			?>
		</div>
		</<?php echo esc_html( $args['type'] ); ?>>
		<?php
		return ob_get_clean();
	}

	/**
	 * Generates HTML for a field description tip.
	 *
	 * @param array $args Arguments.
	 *
	 * @return string
	 */
	private static function render_field_description_tip( $args ) {
		if ( empty( $args['desc_tip'] ) ) {
			return '';
		}

		ob_start();
		?>

		<span class="woocommerce-help-tip" tabindex="0" aria-label="<?php echo esc_attr( $args['aria_label'] ); ?>" data-tip="<?php echo esc_attr( $args['description'] ); ?>"></span>

		<?php
		return ob_get_clean();
	}

	/**
	 * Generates HTML attributes from an associative array.
	 *
	 * @param array $attributes An associative array of attributes.
	 *
	 * @return string
	 */
	private static function generate_custom_attributes( $attributes = [] ) {
		$html = '';

		foreach ( $attributes as $attribute => $value ) {
			if ( 'value' !== $attribute && empty( $value ) ) {
				continue;
			}

			$value = is_array( $value ) ? implode( ',', $value ) : $value;
			$html .= esc_attr( $attribute ) . '="' . esc_attr( $value ) . '" ';
		}

		return trim( $html );
	}

	/**
	 * Provides default arguments for input fields.
	 *
	 * @return array
	 */
	private static function get_default_args() {
		return [
			'tag'               => 'p',
			'id'                => '',
			'label'             => '',
			'desc_tip'          => true,
			'aria_label'        => '',
			'description'       => '',
			'placeholder'       => '',
			'input_class'       => '',
			'value'             => '',
			'wrapper_class'     => 'form-field',
			'custom_attributes' => [],
		];
	}

	/**
	 * Generates common HTML attributes for input fields.
	 *
	 * @param array   $args Arguments for generating the attributes.
	 * @param boolean $is_array If it is an array.
	 *
	 * @return string
	 */
	private static function get_common_input_attributes( $args, $is_array = false ) {
		$attributes = [
			'id'          => $args['id'],
			'name'        => $is_array ? $args['id'] . '[]' : $args['id'],
			'class'       => $args['input_class'],
			'value'       => $args['value'],
			'placeholder' => $args['placeholder'],
		];

		return self::generate_custom_attributes( $attributes );
	}

	/**
	 * Generates the HTML for the label.
	 *
	 * @param array $args Arguments for generating the label.
	 *
	 * @return string
	 */
	private static function render_label( $args ) {
		if ( empty( $args['label'] ) ) {
			return '';
		}

		$class = ! empty( $args['label_class'] ) ? sprintf( ' class="%s"', esc_attr( $args['label_class'] ) ) : '';

		return sprintf(
			'<label for="%s"%s>%s</label>',
			esc_attr( $args['id'] ),
			$class,
			esc_html( $args['label'] )
		);
	}

	/**
	 * Wraps the input field with a specified HTML tag.
	 *
	 * @param string $content The content to wrap.
	 * @param array  $args Arguments for the wrapper tag.
	 *
	 * @return string
	 */
	private static function wrap_with_tag( $content, $args ) {
		$help_text = ! empty( $args['help_text'] ?? '' ) ? sprintf( '<span class="rtsb-input-help">%s</span>', $args['help_text'] ) : '';

		return sprintf(
			'<%s class="%s">%s%s%s%s</%s>',
			esc_html( $args['tag'] ),
			esc_attr( $args['wrapper_class'] ),
			self::render_label( $args ),
			self::render_field_description_tip( $args ),
			$content,
			$help_text,
			esc_html( $args['tag'] )
		);
	}

	/**
	 * Generate the HTML for an admin text input field.
	 *
	 * @param array $args Arguments for generating the text input field.
	 *
	 * @return string
	 */
	public static function generate_text_input( $args = [] ) {
		$args = wp_parse_args( $args, self::get_default_args() );

		ob_start();
		?>
		<input type="text"
			<?php
			Fns::print_html( self::get_common_input_attributes( $args ) );
			Fns::print_html( self::generate_custom_attributes( $args['custom_attributes'] ) );
			?>
		/>
		<?php
		$input_html = ob_get_clean();

		return self::wrap_with_tag( $input_html, $args );
	}

	/**
	 * Generate the HTML for an admin text input field.
	 *
	 * @param array $args Arguments for generating the textarea input field.
	 *
	 * @return string
	 */
	public static function generate_textarea_input( $args = [] ) {
		$args = wp_parse_args( $args, self::get_default_args() );

		ob_start();
		?>
		<textarea
			<?php
			Fns::print_html( self::get_common_input_attributes( $args ) );
			Fns::print_html( self::generate_custom_attributes( $args['custom_attributes'] ) );
			?>
			rows="<?php echo esc_attr( $args['rows'] ?? 4 ); ?>"
			cols="<?php echo esc_attr( $args['cols'] ?? 20 ); ?>"
		><?php echo esc_textarea( $args['value'] ); ?></textarea>
		<?php
		$input_html = ob_get_clean();

		return self::wrap_with_tag( $input_html, $args );
	}

	/**
	 * Generate the HTML for an admin wysiwyg editor input field.
	 *
	 * @param array $args Arguments for generating the textarea input field.
	 *
	 * @return string
	 */
	public static function generate_wysiwyg_editor( $args = [] ) {
		$args = wp_parse_args( $args, self::get_default_args() );

		ob_start();

		$editor_settings = [
			'textarea_name' => esc_attr( $args['id'] ),
			'media_buttons' => false,
		];

		wp_editor( $args['value'], esc_attr( $args['id'] ), $editor_settings );

		$input_html = ob_get_clean();

		return self::wrap_with_tag( $input_html, $args );
	}

	/**
	 * Generate the HTML for an admin switch input field.
	 *
	 * @param array $args The arguments for the input field.
	 *
	 * @return string
	 */
	public static function generate_switch_input( $args = [] ) {
		$args    = wp_parse_args( $args, self::get_default_args() );
		$checked = isset( $args['checked'] ) && 'on' === $args['checked'] ? 'checked' : '';

		$args['wrapper_class'] .= ' switch-field';

		ob_start();
		?>
		<label class="switch-wrapper">
			<span class="switch">
				<input type="checkbox"
					<?php
					Fns::print_html( self::get_common_input_attributes( $args ) );
					echo esc_attr( $checked );
					Fns::print_html( self::generate_custom_attributes( $args['custom_attributes'] ) );
					?>
				/>
				<span class="slider"></span>
			</span>
		</label>
		<?php
		$input_html = ob_get_clean();

		return self::wrap_with_tag( $input_html, $args );
	}

	/**
	 * Generate the HTML for an admin select field.
	 *
	 * @param array $args The arguments for the input field.
	 *
	 * @return string
	 */
	public static function generate_select_input( $args = [] ) {
		$args        = wp_parse_args( $args, self::get_default_args() );
		$options     = $args['options'] ?? [];
		$is_multiple = isset( $args['multiple'] ) && $args['multiple'];
		$multiple    = $is_multiple ? 'multiple' : '';
		$selected    = $args['value'] ?? '';

		if ( $is_multiple ) {
			$selected = ! empty( $args['value'] ) ? $args['value'] : [];
		}

		ob_start();
		?>
		<select
			<?php
			Fns::print_html( self::get_common_input_attributes( $args, $multiple ) );
			Fns::print_html( self::generate_custom_attributes( $args['custom_attributes'] ) );
			echo esc_attr( $multiple );
			echo ! empty( $args['placeholder'] ) ? ' data-placeholder="' . esc_attr( $args['placeholder'] ) . '"' : '';
			?>
		>
			<?php
			foreach ( $options as $value => $label ) {
				$is_selected = $is_multiple ? in_array( $value, $selected, true ) : $selected === $value;
				?>
				<option value="<?php echo esc_attr( $value ); ?>" <?php selected( $is_selected ); ?>>
					<?php echo esc_html( $label ); ?>
				</option>
				<?php
			}
			?>
		</select>
		<?php
		$input_html = ob_get_clean();

		return self::wrap_with_tag( $input_html, $args );
	}

	/**
	 * Generate the HTML for an admin select field.
	 *
	 * @param array $args The arguments for the input field.
	 *
	 * @return string
	 */
	public static function generate_product_search_multiselect_input( $args = [] ) {
		$args        = wp_parse_args( $args, self::get_default_args() );
		$options     = $args['options'] ?? [];
		$is_multiple = isset( $args['multiple'] ) && $args['multiple'];
		$multiple    = $is_multiple ? 'multiple' : '';
		$selected    = $args['value'] ?? '';

		if ( $is_multiple ) {
			if ( ! empty( $args['value'] ) ) {
				if ( is_array( $args['value'] ) ) {
					$selected = array_map( 'strval', $args['value'] );
				} else {
					$selected = [ strval( $args['value'] ) ];
				}
			}
		}

		$args['input_class'] = ( $args['input_class'] ?? '' ) . ' wc-product-search ';
		ob_start();
		?>
		<select
			<?php
			Fns::print_html( self::get_common_input_attributes( $args, $multiple ) );
			Fns::print_html( self::generate_custom_attributes( $args['custom_attributes'] ) );
			echo esc_attr( $multiple );
			echo ! empty( $args['placeholder'] ) ? ' data-placeholder="' . esc_attr( $args['placeholder'] ) . '"' : '';
			?>
			data-placeholder="<?php esc_attr_e( 'Search for products', 'shopbuilder-pro' ); ?>"
			data-action="woocommerce_json_search_products_and_variations"
		>
			<?php
			foreach ( $options as $value => $label ) {
				$value       = strval( $value );
				$is_selected = $is_multiple ? in_array( $value, $selected, true ) : $selected === $value;
				?>
				<option value="<?php echo esc_attr( $value ); ?>" <?php selected( $is_selected ); ?>>
					<?php echo esc_html( $label ); ?>
				</option>
				<?php
			}
			?>
		</select>
		<?php
		$input_html = ob_get_clean();

		return self::wrap_with_tag( $input_html, $args );
	}

	/**
	 * Generate the HTML for an admin number input field.
	 *
	 * @param array $args Arguments for generating the number input field.
	 *
	 * @return string
	 */
	public static function generate_number_input( $args = [] ) {
		$args = wp_parse_args( $args, self::get_default_args() );

		$args = wp_parse_args(
			$args,
			[
				'min'  => '',
				'max'  => '',
				'step' => '',
			]
		);

		ob_start();
		?>
		<input type="number"
			<?php
			Fns::print_html( self::get_common_input_attributes( $args ) );

			if ( isset( $args['min'] ) && '' !== $args['min'] ) {
				echo 'min="' . esc_attr( $args['min'] ) . '" ';
			}

			if ( isset( $args['max'] ) && '' !== $args['max'] ) {
				echo 'max="' . esc_attr( $args['max'] ) . '" ';
			}

			if ( isset( $args['step'] ) && '' !== $args['step'] ) {
				echo 'step="' . esc_attr( $args['step'] ) . '" ';
			}

			Fns::print_html( self::generate_custom_attributes( $args['custom_attributes'] ) );
			?>
		/>
		<?php
		$input_html = ob_get_clean();

		return self::wrap_with_tag( $input_html, $args );
	}

	/**
	 * Generates date picker input for a specified field.
	 *
	 * @param int     $post_id Post ID.
	 * @param array   $field   Field properties.
	 * @param boolean $time    With time field.
	 *
	 * @return void|string
	 */
	public static function generate_date_picker_input( $post_id, $field, $time = false, $echo = true ) {
		// Set default values for the field properties.
		$field = wp_parse_args(
			$field,
			[
				'type'          => $time ? 'datetime-local' : 'date',
				'name'          => $field['name'] ?? $field['id'],
				'wrapper_class' => $field['wrapper_class'] ?? '',
				'class'         => $field['class'] ?? 'short',
				'style'         => $field['style'] ?? '',
				'value'         => $field['value'] ?? get_post_meta( $post_id, $field['id'], true ),
			]
		);
		ob_start();
		?>
		<p class="form-field rtsb-form-field <?php echo esc_attr( $field['id'] ); ?>_field <?php echo esc_attr( $field['wrapper_class'] ); ?>">
			<label for="<?php echo esc_attr( $field['id'] ); ?>" class="<?php echo esc_attr( $field['label_class'] ?? '' ); ?>" ><?php echo wp_kses_post( $field['label'] ); ?></label>

			<?php
			if ( ! empty( $field['description'] ) && false !== $field['desc_tip'] ) {
				echo wp_kses_post( wc_help_tip( $field['description'] ) );
			} elseif ( ! empty( $field['description'] ) && false === $field['desc_tip'] ) {
				echo '<span class="description">' . wp_kses_post( $field['description'] ) . '</span>';
			}
			?>
			<input type="<?php echo esc_attr( $field['type'] ); ?>"
				   name="<?php echo esc_attr( $field['name'] ); ?>"
				   id="<?php echo esc_attr( $field['id'] ); ?>"
				   value="<?php echo esc_attr( $field['value'] ); ?>"
				   class="<?php echo esc_attr( $field['class'] ); ?>"
				   style="<?php echo esc_attr( $field['style'] ); ?>"
				   data-date="<?php echo esc_attr( $field['value'] ); ?>"
			/>
		</p>
		<?php
		if ( $echo ) {
			Fns::print_html( ob_get_clean(), true );
		} else {
			return ob_get_clean();
		}
	}

	/**
	 * Generates a location selection input field.
	 *
	 * @param string $meta_key The key used for the input's name and ID attributes.
	 * @param array  $locations An array of selected location codes.
	 *
	 * @return string
	 */
	public static function generate_location_select_input( $meta_key, $locations = [] ) {
		$locations  = ! empty( $locations ) ? array_map( 'html_entity_decode', $locations ) : [];
		$countries  = WC()->countries->get_countries();
		$continents = WC()->countries->get_continents();
		$all_states = WC()->countries->get_states();

		ob_start();
		?>
		<p class="form-field rtsb-location-input">
			<label for="<?php echo esc_attr( $meta_key ); ?>"><?php echo esc_html__( 'Allowed locations', 'shopbuilder-pro' ); ?></label>
			<select name="<?php echo esc_attr( $meta_key ); ?>[]" id="<?php echo esc_attr( $meta_key ); ?>" data-placeholder="<?php echo esc_attr__( 'Select locations', 'shopbuilder-pro' ); ?>..." class="rtsb-select2-input wc-enhanced-select" multiple>
				<?php
				foreach ( $continents as $continent_code => $continent ) {
					?>
					<optgroup label="<?php echo esc_attr( $continent['name'] ?? $continent_code ); ?>">
						<?php
						foreach ( $continent['countries'] as $country_code ) {
							if ( isset( $countries[ $country_code ] ) ) {
								?>
								<option value="<?php echo esc_attr( $country_code ); ?>" <?php selected( in_array( html_entity_decode( $country_code ), $locations, true ) ); ?>>
									<?php echo esc_html( $countries[ $country_code ] ); ?>
								</option>
								<?php
							}
						}
						?>
					</optgroup>
					<?php
				}

				foreach ( $all_states as $country_code => $states ) {
					if ( empty( $states ) ) {
						continue;
					}
					?>
					<optgroup label="<?php echo esc_attr( $countries[ $country_code ] ?? $country_code ); ?>">
						<?php
						foreach ( $states as $state_code => $state_name ) {
							$country_state_code = $country_code . '::' . $state_code;
							?>
							<option value="<?php echo esc_attr( $country_state_code ); ?>" <?php selected( in_array( html_entity_decode( $country_state_code ), $locations, true ) ); ?>>
								<?php echo esc_html( $state_name ); ?>
							</option>
							<?php
						}
						?>
					</optgroup>
					<?php
				}
				?>
			</select>
			<?php
			echo wp_kses_post( wc_help_tip( __( 'Select the location to apply this coupon. This coupon will be applied only if these locations are selected.', 'shopbuilder-pro' ) ) );
			?>
		</p>
		<?php

		return ob_get_clean();
	}
}
