<?php
/**
 * Main ProductTitle class.
 *
 * @package RadiusTheme\SB
 */

namespace RadiusTheme\SBPRO\Elementor\Widgets\Single;

use RadiusTheme\SB\Helpers\Fns;
use RadiusTheme\SB\Abstracts\ElementorWidgetBase;
use RadiusTheme\SBPRO\Elementor\Widgets\Controls\FlashSaleCountdownSettings;
use RadiusTheme\SBPRO\Modules\FlashSaleCountdown\CountdownFns;

// Do not allow directly accessing this file.
if ( ! defined( 'ABSPATH' ) ) {
	exit( 'This script cannot be accessed directly.' );
}

/**
 * Product Title class
 */
class FlashSaleCountdown extends ElementorWidgetBase {
	/**
	 * Construct function
	 *
	 * @param array $data default array.
	 * @param mixed $args default arg.
	 */
	public function __construct( $data = [], $args = null ) {
		$this->rtsb_name = esc_html__( 'Flash Sale Countdown', 'shopbuilder-pro' );
		$this->rtsb_base = 'rtsb-product-flash-sale-countdown';
		parent::__construct( $data, $args );
	}

	/**
	 * Widget Field
	 *
	 * @return array
	 */
	public function widget_fields() {
		$fields     = FlashSaleCountdownSettings::settings( $this );
		$new_fields = [
			'layout_style'             => [
				'type'      => 'rtsb-image-selector',
				'options'   => [
					'layout1' => [
						'title' => esc_html__( 'Layout 1', 'shopbuilder-pro' ),
						'url'   => esc_url( rtsbpro()->get_assets_uri( 'images/layout/flash-sale-layout-1.png' ) ),
					],
					'layout2' => [
						'title' => esc_html__( 'Layout 2', 'shopbuilder-pro' ),
						'url'   => esc_url( rtsbpro()->get_assets_uri( 'images/layout/flash-sale-layout-2.png' ) ),
					],
				],
				'default'   => 'layout1',
				'separator' => 'default',
			],
			'count_text_label'         => [
				'label'       => esc_html__( 'Count Label Text', 'shopbuilder-pro' ),
				'type'        => 'text',
				'label_block' => true,
			],
			'count_label_icon'         => [
				'label' => esc_html__( 'Counter Label Icon', 'shopbuilder-pro' ),
				'type'  => 'icons',
			],
			'counter_label_typography' => [
				'mode'     => 'group',
				'type'     => 'typography',
				'label'    => esc_html__( 'Label Typography', 'shopbuilder-pro' ),
				'selector' => $this->selectors['counter_label_typography'],
			],
		];
		$fields     = Fns::insert_controls( 'counter_section_start', $fields, $new_fields, true );
		$insert_new = [
			'element_alignment' => [
				'mode'      => 'responsive',
				'type'      => 'choose',
				'label'     => esc_html__( 'Element Alignment', 'shopbuilder-pro' ),
				'options'   => [
					'left'   => [
						'title' => esc_html__( 'Left', 'shopbuilder-pro' ),
						'icon'  => 'eicon-text-align-left',
					],
					'center' => [
						'title' => esc_html__( 'Center', 'shopbuilder-pro' ),
						'icon'  => 'eicon-text-align-center',
					],
					'right'  => [
						'title' => esc_html__( 'Right', 'shopbuilder-pro' ),
						'icon'  => 'eicon-text-align-right',
					],
				],
				'selectors' => [
					$this->selectors['element_alignment'] => 'display: flex; justify-content: {{VALUE}};',
				],
			],
		];
		$fields     = Fns::insert_controls( 'counter_gap', $fields, $insert_new );
		return $fields;
	}
	/**
	 * Set Widget Keyword.
	 *
	 * @return array
	 */
	public function get_keywords() {
		return [ 'Flash sale', 'Countdown', 'Name' ] + parent::get_keywords();
	}

	/**
	 * Elementor Edit mode need some extra js for isotop reinitialize
	 *
	 * @return void
	 */
	public function editor_script() {
		if ( $this->is_edit_mode() ) { ?>
			<script type="text/javascript">
				if (!'<?php echo esc_attr( Fns::is_optimization_enabled() ); ?>') {
					setTimeout(function() {
						window.rtsbCountdownApply();
					}, 1000);
				} else {
					if (typeof elementorFrontend !== 'undefined') {
						elementorFrontend.hooks.addAction(
							'frontend/element_ready/rtsb-product-flash-sale-countdown.default',
							() => {
								window.waitForRTSB((RTSB) => {
									RTSB.modules.get('countdownPro')?.refresh();
								});
							}
						);
					}
				}
			</script>
			<?php
		}
	}

	/**
	 * Render Function
	 *
	 * @return void
	 */
	protected function render() {
		global $product;
		$_product    = $product;
		$product     = Fns::get_product();
		$controllers = $this->get_settings_for_display();
		$campaign    = CountdownFns::get_campaign_for_current_product( $product );
		if ( $this->is_edit_mode() && ! Fns::is_module_active( 'flash_sale_countdown' ) ) {
			echo '<p>' . esc_html__( 'Editor Notice: The Flash Sale Countdown Module Has been Disabled. Enable For displaying the output.', 'shopbuilder-pro' ) . '</p>';
		}
		if ( ! empty( $campaign ) ) {
			do_action( 'rtsb/before/product/flash/sale/countdown/widget', $controllers, $this );
			$this->theme_support();
			$layout   = $controllers['layout_style'] ?? '';
			$template = 'flash-sale-countdown';
			if ( 'layout2' === $layout ) {
				$template = 'flash-sale-countdown-layout2';
			}
			$data        = apply_filters(
				'rtsb/product/flash/sale/countdown/widget/data',
				[
					'template'    => 'elementor/single-product/' . $template,
					'controllers' => $controllers,
				]
			);
			$date_time   = $campaign['end_date'] ?? '';
			$expire_time = strtotime( gmdate( 'Y-m-d H:m', strtotime( $date_time ) ) );
			$now         = strtotime( 'now' );
			if ( $date_time && $now < $expire_time ) {
				Fns::load_template( $data['template'], $data, false, '', rtsbpro()->get_plugin_template_path() );
			}
			do_action( 'rtsb/after/product/flash/sale/countdown/widget', $controllers, $this );
		} elseif ( $this->is_edit_mode() ) {
			echo '<p>' . esc_html__( 'Editor Notice: No active campaign found for this product.', 'shopbuilder-pro' ) . '</p>';
		}

		$this->theme_support( 'render_reset' );
		$this->editor_script();
		$product = $_product; // phpcs:ignore WordPress.WP.GlobalVariablesOverride.Prohibited
	}
}
