<?php
/**
 * Ajax Actions Class for Product Lookbook.
 *
 * @package RadiusTheme\SB
 */

namespace RadiusTheme\SBPRO\Controllers\Frontend\Ajax;

// Do not allow directly accessing this file.
if ( ! defined( 'ABSPATH' ) ) {
	exit( 'This script cannot be accessed directly.' );
}

use RadiusTheme\SB\Elementor\Render\GeneralAddons;
use RadiusTheme\SB\Helpers\Fns;
use RadiusTheme\SBPRO\Traits\SingletonTrait;

/**
 * Ajax Actions Class.
 */
class AjaxLookbook {

	/**
	 * Singleton.
	 */
	use SingletonTrait;

	/**
	 * Ajax data.
	 *
	 * @var array
	 */
	public $ajax_data;

	/**
	 * Render class.
	 *
	 * @var array
	 */
	public $render;

	/**
	 * Class Constructor.
	 *
	 * @return void
	 */
	private function __construct() {
		add_action( 'wp_ajax_rtsb_get_product_info', [ $this, 'response' ] );
		add_action( 'wp_ajax_nopriv_rtsb_get_product_info', [ $this, 'response' ] );
	}

	/**
	 * Ajax Response.
	 *
	 * @return void
	 */
	public function response() {
		if ( ! Fns::verify_nonce() ) {
			wp_send_json_error( esc_html__( 'Security check failed!', 'shopbuilder-pro' ) );
		}

		$product_id = $this->get_product_id();

		if ( ! $product_id ) {
			wp_send_json_error( esc_html__( 'Invalid product ID', 'shopbuilder-pro' ) );
		}

		$product = wc_get_product( $product_id );

		if ( ! $product ) {
			wp_send_json_error( esc_html__( 'Product not found', 'shopbuilder-pro' ) );
		}

		$content = $this->get_product_content( $product_id, $product );

		wp_send_json_success( $content );
	}

	/**
	 * Generate Product Content.
	 *
	 * @param int    $product_id Product ID.
	 * @param object $product Product object.
	 *
	 * @return string
	 */
	private function get_product_content( $product_id, $product ) {
		$render = GeneralAddons::instance();

		$instance      = $this->get_instance_id();
		$gallery_ids   = $this->get_gallery_ids( $product );
		$product_title = $product->get_name();
		$aria_label    = esc_attr(
			/* translators: Product Name */
			sprintf( __( 'Image link for Product: %s', 'shopbuilder-pro' ), $product_title )
		);

		$data = [
			'product'        => $product,
			'product_id'     => $product_id,
			'p_link'         => $product->get_permalink(),
			'product_title'  => $product_title,
			'img_html'       => $this->get_image_html( $product_id ),
			'hover_img_html' => $this->get_hover_image_html( $gallery_ids ),
			'container_attr' => $this->get_container_attributes( $product_id, $instance, $render, $gallery_ids ),
			'aria_label'     => $aria_label,
			'add_to_cart'    => $this->get_add_to_cart_button( $product_id, $product, $render ),
			'badge_html'     => $this->get_badge_html( $product ),
			'items'          => [ 'wishlist', 'quick_view', 'add_to_cart' ],
		];

		$this->fix_icons();

		return Fns::load_template( 'elementor/general/lookbook/product-popup', $data, true );
	}

	/**
	 * Get Product Image HTML.
	 *
	 * @param int $product_id Product ID.
	 *
	 * @return string
	 */
	private function get_image_html( $product_id ) {
		return Fns::get_product_image_html( 'product', $product_id, 'woocommerce_thumbnail' );
	}

	/**
	 * Get Hover Image HTML.
	 *
	 * @param array $gallery_ids Product gallery IDs.
	 *
	 * @return string
	 */
	private function get_hover_image_html( $gallery_ids = [] ) {
		return ! empty( $gallery_ids ) ? Fns::get_product_image_html(
			'product',
			$gallery_ids[0],
			'woocommerce_thumbnail',
			null,
			[],
			false,
			true
		) : '';
	}

	/**
	 * Get Container Attributes.
	 *
	 * @param int    $product_id Product ID.
	 * @param int    $instance Instance ID.
	 * @param object $render Render object.
	 * @param array  $gallery_ids Product gallery IDs.
	 *
	 * @return string
	 */
	private function get_container_attributes( $product_id, $instance, $render, $gallery_ids = [] ) {
		$attributes = [
			'id'    => 'rtsb-lookbook-popup-' . $instance,
			'class' => array_filter(
				[
					'rtsb-elementor-container',
					! empty( $gallery_ids ) ? 'gallery-hover-fade' : '',
				]
			),
		];

		$render->add_attribute( 'rtsb_lookbook_container_attr_' . $product_id, $attributes );

		return $render->get_attribute_string( 'rtsb_lookbook_container_attr_' . $product_id );
	}

	/**
	 * Get Add to Cart Button HTML.
	 *
	 * @param int    $product_id Product ID.
	 * @param object $product Product object.
	 * @param object $render Render object.
	 *
	 * @return string
	 */
	private function get_add_to_cart_button( $product_id, $product, $render ) {
		if ( ! $product->is_in_stock() ) {
			return '';
		}

		return $render->render_add_to_cart_button(
			[
				'link'      => $product->get_permalink(),
				'id'        => $product_id,
				'type'      => $product->get_type(),
				'text'      => esc_html__( 'Add to cart', 'shopbuilder-pro' ),
				'success'   => esc_html__( 'Added!', 'shopbuilder-pro' ),
				'icon_html' => '',
				'alignment' => 'center',
				'product'   => $product,
			]
		);
	}

	/**
	 * Get Badge HTML.
	 *
	 * @param object $product Product object.
	 *
	 * @return string
	 */
	private function get_badge_html( $product ) {
		return Fns::get_promo_badge(
			$product,
			'text',
			esc_html__( 'Sale', 'shopbuilder-pro' ),
			esc_html__( 'Out of Stock', 'shopbuilder-pro' ),
		);
	}

	/**
	 * Get product gallery ids.
	 *
	 * @param object $product Product object.
	 *
	 * @return array
	 */
	private function get_gallery_ids( $product ) {
		if ( ! ( function_exists( 'rtwpvsp' ) && $product->is_type( 'variable' ) && $product->get_attributes() ) ) {
			return Fns::get_cached_gallery_ids( $product );
		}

		return [];
	}

	/**
	 * Get Product ID from Request.
	 *
	 * @return int
	 */
	private function get_product_id() {
		// phpcs:ignore WordPress.Security.NonceVerification.Recommended
		return ! empty( $_REQUEST['product_id'] ) ? absint( $_REQUEST['product_id'] ) : 0;
	}

	/**
	 * Get instance ID from Request.
	 *
	 * @return int
	 */
	private function get_instance_id() {
		// phpcs:ignore WordPress.Security.NonceVerification.Recommended
		return ! empty( $_REQUEST['instance_id'] ) ? absint( $_REQUEST['instance_id'] ) : 0;
	}

	/**
	 * Icon fix.
	 *
	 * @return void
	 */
	private function fix_icons() {
		add_filter( 'rtsb/module/wishlist/icon_html', [ $this, 'fix_wishlist_icon' ] );
	}

	/**
	 * Wishlist icon fix
	 *
	 * @return string
	 */
	public function fix_wishlist_icon() {
		$wishlist_icon  = [
			'value'   => 'rtsb-icon rtsb-icon-heart-empty',
			'library' => 'rtsb-fonts',
		];
		$wishlist_added = [
			'value'   => 'rtsb-icon rtsb-icon-heart',
			'library' => 'rtsb-fonts',
		];

		$icon_html  = Fns::icons_manager( $wishlist_icon, 'icon-default' );
		$icon_html .= Fns::icons_manager( $wishlist_added, 'added-icon' );

		return $icon_html;
	}
}
