<?php
/**
 * Ajax Filters Class.
 *
 * @package RadiusTheme\SB
 */

namespace RadiusTheme\SBPRO\Controllers\Frontend\Ajax;

use WP_Query;
use RadiusTheme\SB\Helpers\Fns;
use RadiusTheme\SBPRO\Helpers\FnsPro;
use RadiusTheme\SBPRO\Traits\SingletonTrait;
use RadiusTheme\SB\Elementor\Render\Render;
use RadiusTheme\SB\Controllers\Hooks\ActionHooks;
use RadiusTheme\SB\Elementor\Helper\RenderHelpers;
use RadiusTheme\SBPRO\Controllers\Hooks\ElementorHooks;

// Do not allow directly accessing this file.
if ( ! defined( 'ABSPATH' ) ) {
	exit( 'This script cannot be accessed directly.' );
}

/**
 * Ajax Filters Class.
 */
class AjaxFilters {
	/**
	 * Singleton.
	 */
	use SingletonTrait;

	/**
	 * Filter Ajax data.
	 *
	 * @var array
	 */
	public $filter_ajax_data;

	/**
	 * Product Ajax data.
	 *
	 * @var array
	 */
	public $archive_ajax_data;

	/**
	 * Render data.
	 *
	 * @var array
	 */
	public $render;

	/**
	 * Class Constructor.
	 *
	 * @return void
	 */
	private function __construct() {
		add_action( 'wp_ajax_rtsb_load_archive_template', [ $this, 'response' ] );
		add_action( 'wp_ajax_nopriv_rtsb_load_archive_template', [ $this, 'response' ] );
	}

	/**
	 * Ajax Response.
	 *
	 * @return void
	 */
	public function response() { // phpcs:ignore Generic.Metrics.CyclomaticComplexity.MaxExceeded
		$msg            = null;
		$data           = null;
		$metaText       = null;
		$paginationData = null;
		$error          = false;

		if ( ! Fns::verify_nonce() ) {
			wp_send_json_error();
		}

		// phpcs:disable
		$this->filter_ajax_data  = ! empty( $_REQUEST['filterAjaxData'] ) ? json_decode( wp_unslash( $_REQUEST['filterAjaxData'] ), true ) : [];
		$this->archive_ajax_data = ! empty( $_REQUEST['archiveAjaxData'] ) ? json_decode( wp_unslash( $_REQUEST['archiveAjaxData'] ), true ) : [];
		$this->render            = new Render();
		// phpcs:enable

		$products_row = absint( get_option( 'woocommerce_catalog_rows', 4 ) );
		$products_col = absint( get_option( 'woocommerce_catalog_columns', 4 ) );

		$view_mode = ! empty( $this->archive_ajax_data['view_mode'] ) ? sanitize_text_field( wp_unslash( $this->archive_ajax_data['view_mode'] ) ) : 'grid';
		$view_mode = ! empty( $_REQUEST['viewMode'] ) ? sanitize_text_field( wp_unslash( $_REQUEST['viewMode'] ) ) : $view_mode; // phpcs:ignore WordPress.Security.NonceVerification.Recommended
		$orderby   = ! empty( $_REQUEST['orderby'] ) ? sanitize_text_field( wp_unslash( $_REQUEST['orderby'] ) ) : 'menu_order'; // phpcs:ignore WordPress.Security.NonceVerification.Recommended

		$terms        = ! empty( $_REQUEST['terms'] ) ? array_map( 'sanitize_text_field', wp_unslash( $_REQUEST['terms'] ) ) : []; // phpcs:ignore WordPress.Security.NonceVerification.Recommended
		$widget       = esc_html( $this->archive_ajax_data['widget'] );
		$template     = esc_html( $this->archive_ajax_data['template'] );
		$per_page     = ! empty( $this->archive_ajax_data['posts_per_page'] ) ? $this->archive_ajax_data['posts_per_page'] : $products_row * $products_col;
		$paged        = ! empty( $_REQUEST['paged'] ) ? absint( $_REQUEST['paged'] ) : 2; // phpcs:ignore WordPress.Security.NonceVerification.Recommended
		$queried_tax  = ! empty( $this->archive_ajax_data['queried_tax'] ) ? esc_html( $this->archive_ajax_data['queried_tax'] ) : null;
		$queried_term = ! empty( $this->archive_ajax_data['queried_term'] ) ? esc_html( $this->archive_ajax_data['queried_term'] ) : null;

		$args = [
			'post_type'      => [ 'product' ],
			'posts_per_page' => absint( $per_page ),
			'order'          => esc_html( $this->archive_ajax_data['rtsb_order'] ),
			'orderby'        => esc_html( $this->archive_ajax_data['rtsb_orderby'] ),
			'post_status'    => 'publish',
			'paged'          => $paged,
		];

		$args['tax_query'][] = [
			'taxonomy' => 'product_visibility',
			'terms'    => [ 'exclude-from-catalog' ],
			'field'    => 'name',
			'operator' => 'NOT IN',
		];

		if ( ! empty( $queried_tax ) && ! empty( $queried_term ) ) {
			$args['tax_query'][] = [
				'taxonomy' => $queried_tax,
				'field'    => 'term_id',
				'terms'    => $queried_term,
			];
		}

		if ( 'yes' === get_option( 'woocommerce_hide_out_of_stock_items' ) ) {
			$args['meta_query'][] = [
				'relation' => 'AND',
				[
					'key'     => '_stock_status',
					'value'   => 'outofstock',
					'compare' => '!=',
				],
			];
		}

		if ( ! empty( $terms ) ) {
			$args['tax_query']['relation'] = 'AND';

			foreach ( $terms as $tax_name => $term ) {
				if ( strpos( $tax_name, 'query_type_' ) === 0 ) {
					continue;
				}

				$operator = 'IN';

				if ( 'sale_filter' === $tax_name ) {
					$on_sale_products = wc_get_product_ids_on_sale();

					if ( 'onsale' === $term ) {
						$args['post__in'] = ! empty( $on_sale_products ) ? $on_sale_products : [ 0 ];
					}

					if ( 'regular' === $term ) {
						$args['post__not_in'] = $on_sale_products;
					}

					continue;
				}

				if ( 'price_filter' === $tax_name ) {
					$args['meta_query'][] = [
						'relation' => 'AND',
					];

					$args['meta_query'][] = [
						'key'     => '_price',
						'value'   => explode( ',', $term ),
						'type'    => 'numeric',
						'compare' => 'BETWEEN',
					];

					continue;
				}

				if ( 'rating_filter' === $tax_name ) {
					$args['meta_query'][] = [
						'relation' => 'AND',
					];

					$args['meta_query'][] = [
						'key'     => '_wc_average_rating',
						'value'   => explode( ',', $term ),
						'type'    => 'numeric',
						'compare' => 'IN',
					];

					continue;
				}

				if ( 'search' === $tax_name ) {
					// phpcs:ignore WordPress.Security.NonceVerification.Recommended, WordPress.Security.ValidatedSanitizedInput.InputNotSanitized
					$search_term = isset( $_REQUEST['terms']['search'] ) ? wp_unslash( $_REQUEST['terms']['search'] ) : '';
					$args['s']   = sanitize_text_field( urldecode( $search_term ) );

					continue;
				}

				if ( strpos( $tax_name, 'filter_' ) !== false ) {
					$base_name      = str_replace( 'filter_', '', $tax_name );
					$tax_name       = str_replace( 'filter_', 'pa_', $tax_name );
					$query_type_key = 'query_type_' . $base_name;
					$operator       = ! empty( $terms[ $query_type_key ] ) && strtoupper( $terms[ $query_type_key ] ) === 'OR' ? 'IN' : 'AND';
				}

				$args['tax_query'][] = [
					'taxonomy' => $tax_name,
					'field'    => 'slug',
					'terms'    => explode( ',', $term ),
					'operator' => $operator,
				];
			}
		}

		if ( 'menu_order' === $this->archive_ajax_data['rtsb_orderby'] ) {
			$args['orderby'] = 'menu_order title';
			$args['order']   = 'ASC';
		}

		if ( 'price' === $orderby || 'price' === $this->archive_ajax_data['rtsb_orderby'] ) {
			$args['orderby']  = 'meta_value_num';
			$args['order']    = 'ASC';
			$args['meta_key'] = '_price'; // phpcs:ignore WordPress.DB.SlowDBQuery.slow_db_query_meta_key
		}

		if ( 'price-desc' === $orderby || 'price-desc' === $this->archive_ajax_data['rtsb_orderby'] ) {
			$args['orderby']  = 'meta_value_num';
			$args['order']    = 'DESC';
			$args['meta_key'] = '_price'; // phpcs:ignore WordPress.DB.SlowDBQuery.slow_db_query_meta_key
		}

		if ( 'rating' === $orderby || 'rating' === $this->archive_ajax_data['rtsb_orderby'] ) {
			$args['meta_key'] = '_wc_average_rating'; // phpcs:ignore WordPress.DB.SlowDBQuery.slow_db_query_meta_key
			$args['orderby']  = [
				'meta_value_num' => 'DESC',
				'date'           => 'DESC',
			];
		}

		if ( 'popularity' === $orderby || 'popularity' === $this->archive_ajax_data['rtsb_orderby'] ) {
			$args['meta_key'] = 'total_sales'; // phpcs:ignore WordPress.DB.SlowDBQuery.slow_db_query_meta_key
			$args['orderby']  = [
				'meta_value_num' => 'DESC',
				'date'           => 'DESC',
			];
		}

		if ( 'date' === $orderby || 'date' === $this->archive_ajax_data['rtsb_orderby'] ) {
			$args['orderby'] = 'date ID';
			$args['order']   = 'DESC';
		}

		if ( 'menu_order' === $orderby ) {
			$args['orderby'] = 'menu_order title';
			$args['order']   = 'ASC';

			unset( $args['meta_key'] );
		}

		$args = apply_filters( 'rtsb/ajax_filter/args', $args, $this->archive_ajax_data );

		$product_query = new WP_Query( $args );
		$total         = $product_query->found_posts;
		$current       = ! empty( $paged ) ? $paged : 1;

		$metaText .= $this->getPaginationMetaText( $total, $per_page, $current );

		$filtered_price = FnsPro::get_custom_filtered_price( $product_query );

		// Init Hooks.
		$this->init_hooks();

		if ( $product_query->have_posts() ) {
			while ( $product_query->have_posts() ) {
				$product_query->the_post();

				if ( 'custom' === $widget ) {
					$custom_arg = $this->render->wp_arg_dataset( $this->archive_ajax_data, get_the_ID(), $this->archive_ajax_data['lazy_load'] );

					// Image gallery.
					$this->image_gallery_support();

					if ( 'grid' === $view_mode ) {
						$template = 'elementor/general/grid/';
						$layout   = $this->archive_ajax_data['grid_layout'];
					} elseif ( 'list' === $view_mode ) {
						$template = 'elementor/general/list/';
						$layout   = $this->archive_ajax_data['list_layout'];
					} else {
						$layout = $this->archive_ajax_data['layout'];
					}

					$custom_arg['title']   = apply_filters( 'rtsb/archive/custom/product/title', $custom_arg['title'] );
					$custom_arg['excerpt'] = apply_filters( 'rtsb/archive/custom/product/excerpt', $custom_arg['excerpt'] );

					// Load Template.
					$data .= Fns::load_template( $template . $layout, $custom_arg, true );
				} else {
					ob_start();
					wc_get_template_part( 'content', 'product' );
					$data .= ob_get_clean();
				}
			}

			if ( $product_query->max_num_pages > 1 ) {
				$paginationData = "
                     <div class='rtsb-load-more rtsb-pos-r'>
                        <button data-paged='2' data-max-page='{$product_query->max_num_pages}'><span>" . esc_html__( 'Load More', 'shopbuilder-pro' ) . "</span></button>
                        <div class='rtsb-loadmore-loading rtsb-ball-clip-rotate'><div></div></div>
                     </div>
             ";
			}
		} else {
			$data .= '<div class="rtsb-notice woocommerce-no-products-found"><div class="woocommerce-info">' . esc_html__( 'No products were found matching your selection.', 'shopbuilder-pro' ) . '</div></div>';
		}

		wp_reset_postdata();

		wp_send_json(
			[
				'msg'            => $msg,
				'metaText'       => $metaText,
				'data'           => $data,
				'paginationData' => $paginationData,
				'error'          => $error,
				'filtered_price' => $filtered_price,
			]
		);
	}

	/**
	 * Init hooks.
	 *
	 * @return void
	 */
	public function init_hooks() {
		$this->action_hooks()->filter_hooks()->remove_hooks();
	}

	/**
	 * Gallery image support.
	 *
	 * @return void
	 */
	public function image_gallery_support() {
		remove_action( 'rtsb/shop/product/image', [ ActionHooks::CLASS, 'render_image' ] );
		add_action( 'rtsb/shop/product/image', [ ElementorHooks::class, 'render_image' ], 10, 2 );
	}

	/**
	 * Set up action hooks.
	 *
	 * @return self
	 */
	private function action_hooks() {
		add_action( 'woocommerce_before_shop_loop_item', [ $this, 'image_wrapper_start' ], -1 );
		add_action( 'woocommerce_before_shop_loop_item_title', [ $this, 'wrapper_close' ], 12 );
		add_action( 'woocommerce_before_shop_loop_item_title', [ $this, 'product_content_wrapper_start' ], 13 );
		add_action( 'woocommerce_after_shop_loop_item', [ $this, 'wrapper_close' ], 99 );
		add_action( 'woocommerce_before_shop_loop_item_title', 'woocommerce_template_loop_product_link_open', 9 );
		add_action( 'woocommerce_before_shop_loop_item_title', 'woocommerce_template_loop_product_link_close', 11 );
		add_action( 'woocommerce_shop_loop_item_title', 'woocommerce_template_loop_product_link_open', 9 );
		add_action( 'woocommerce_shop_loop_item_title', 'woocommerce_template_loop_product_link_close', 11 );

		return $this;
	}

	/**
	 * Set up filter hooks.
	 *
	 * @return self
	 */
	private function filter_hooks() {
		add_filter( 'rtsb/module/wishlist/show_button', '__return_false' );
		add_filter( 'rtsb/module/quick_view/show_button', '__return_false' );
		add_filter( 'rtsb/module/compare/show_button', '__return_false' );
		add_filter( 'rtsb/quick_checkout/button/icon', '__return_false' );
		add_filter( 'woocommerce_loop_add_to_cart_link', [ $this, 'print_action_button' ] );

		if ( 'custom' === $this->archive_ajax_data['widget'] ) {
			$show_quick_view = in_array( 'quick_view', $this->archive_ajax_data['visibility'], true );
			$show_compare    = in_array( 'compare', $this->archive_ajax_data['visibility'], true );
			$show_wishlist   = in_array( 'wishlist', $this->archive_ajax_data['visibility'], true );
			$show_rating     = in_array( 'rating', $this->archive_ajax_data['visibility'], true );
			$show_q_checkout = in_array( 'quick_checkout', $this->archive_ajax_data['visibility'], true );
		} else {
			$show_quick_view = $this->archive_ajax_data['quick_view_button'];
			$show_compare    = $this->archive_ajax_data['comparison_button'];
			$show_wishlist   = $this->archive_ajax_data['wishlist_button'];
			$show_rating     = ! empty( $this->archive_ajax_data['show_rating'] );
			$show_q_checkout = false;
		}

		if ( empty( $this->archive_ajax_data['show_flash_sale'] ) ) {
			add_filter( 'woocommerce_sale_flash', '__return_false' );
		}

		if ( ! $show_rating ) {
			add_filter( 'woocommerce_product_get_rating_html', '__return_false' );
		}

		if ( $show_quick_view ) {
			add_filter( 'rtsb/module/quick_view/icon_html', [ $this, 'quick_view_icon' ] );
		}

		if ( $show_compare ) {
			add_filter( 'rtsb/module/compare/icon_html', [ $this, 'compare_icon' ] );
		}

		if ( $show_wishlist ) {
			add_filter( 'rtsb/module/wishlist/icon_html', [ $this, 'wishlist_icon' ] );
		}

		if ( $show_q_checkout ) {
			add_filter( 'rtsb/quick_checkout/button/icon', [ $this, 'quick_checkout_icon' ] );
			add_action( 'rtsb/quick_checkout/parent/class', [ $this, 'quick_checkout_parent_classes' ], 15 );
			add_action( 'rtsb/quick_checkout/button/text', [ $this, 'quick_checkout_btn_txt' ], 15 );
		}

		return $this;
	}

	/**
	 * Removing some hooks.
	 *
	 * @return void
	 */
	private function remove_hooks() {
		if ( empty( $this->archive_ajax_data['show_rating'] ) ) {
			remove_action( 'woocommerce_after_shop_loop_item_title', 'woocommerce_template_loop_rating', 5 );
		}

		if ( class_exists( 'WooProductVariationSwatches' ) ) {
			remove_action( 'woocommerce_before_shop_loop_item_title', 'woocommerce_template_loop_product_link_open', 5 );
			remove_action( 'woocommerce_before_shop_loop_item_title', 'woocommerce_template_loop_product_link_close', 20 );
			remove_action( 'woocommerce_shop_loop_item_title', 'woocommerce_template_loop_product_link_open', 8 );
			remove_action( 'woocommerce_shop_loop_item_title', 'woocommerce_template_loop_product_link_close', 12 );
		}

		remove_action( 'woocommerce_before_shop_loop_item', 'woocommerce_template_loop_product_link_open', 10 );
		remove_action( 'woocommerce_after_shop_loop_item', 'woocommerce_template_loop_product_link_close', 5 );
	}

	/**
	 * Ger total count text.
	 *
	 * @param int $total Total Posts.
	 * @param int $per_page Posts Per page.
	 * @param int $current Current page.
	 *
	 * @return string
	 */
	private function getPaginationMetaText( $total, $per_page, $current ) {
		$metaText = '';

		if ( 1 === $total ) {
			$metaText .= esc_html__( 'Showing the single result', 'shopbuilder-pro' );
		} elseif ( $total <= $per_page || -1 === $per_page ) {
			/* translators: %d: total results */
			$metaText .= sprintf( _n( 'Showing all %d result', 'Showing all %d results', $total, 'shopbuilder-pro' ), $total );
		} else {
			// $first     = ( $per_page * $current ) - $per_page + 1;
			$first     = 1;
			$last      = min( $total, $per_page * $current );
			$metaText .= sprintf(
				/* translators: %1$d: first result, %2$d: last result, %3$d: total results */
				_nx(
					'Showing the single result', // phpcs:ignore WordPress.WP.I18n.MissingSingularPlaceholder
					'Showing %1$d&ndash;<span class="wc-last-result-count">%2$d</span> of %3$d results',
					$total,
					'with first and last result',
					'shopbuilder-pro'
				),
				$first,
				$last,
				$total
			);
		}

		return $metaText;
	}

	/**
	 * Print the action button and return the modified link HTML.
	 *
	 * @param string $link_html The HTML of the link.
	 *
	 * @return string
	 */
	public function print_action_button( $link_html ) {
		ob_start();

		global $product;

		echo "<div class='action-button-wrapper'>";

		if ( Fns::is_module_active( 'wishlist' ) && ! empty( $this->archive_ajax_data['wishlist_button'] ) ) {
			do_action( 'rtsb/modules/wishlist/print_button', $product->get_id() );
		}

		if ( Fns::is_module_active( 'compare' ) && ! empty( $this->archive_ajax_data['comparison_button'] ) ) {
			do_action( 'rtsb/modules/compare/print_button', $product->get_id() );
		}

		if ( Fns::is_module_active( 'quick_view' ) && ! empty( $this->archive_ajax_data['quick_view_button'] ) ) {
			do_action( 'rtsb/modules/quick_view/print_button', $product->get_id() );
		}

		echo '</div>';

		return $link_html . ob_get_clean();
	}

	/**
	 * Open the image wrapper div.
	 *
	 * @return void
	 */
	public function image_wrapper_start() {
		if ( ! RenderHelpers::is_wrapper_needed() ) {
			return;
		}
		?>
		<div class="rtsb-image-wrapper">
		<?php
	}

	/**
	 * Open the product content wrapper div.
	 *
	 * @return void
	 */
	public function product_content_wrapper_start() {
		if ( ! RenderHelpers::is_wrapper_needed() ) {
			return;
		}
		?>
		<div class="rtsb-product-content">
		<?php
	}

	/**
	 * Wrapper close.
	 *
	 * @return void
	 */
	public function wrapper_close() {
		if ( ! RenderHelpers::is_wrapper_needed() ) {
			return;
		}
		?>
		</div>
		<?php
	}

	/**
	 * Quick View icon
	 *
	 * @param string $icon_html Icon Html.
	 *
	 * @return string
	 */
	public function quick_view_icon( $icon_html ) {
		if ( ! empty( $this->archive_ajax_data['quick_view_icon'] ) ) {
			$icon_html = Fns::icons_manager( $this->archive_ajax_data['quick_view_icon'], 'icon-default' );
		}

		return $icon_html;
	}

	/**
	 * Compare icon
	 *
	 * @param string $icon_html Icon Html.
	 *
	 * @return string
	 */
	public function compare_icon( $icon_html ) {
		if ( ! empty( $this->archive_ajax_data['comparison_icon'] ) ) {
			$icon_html  = Fns::icons_manager( $this->archive_ajax_data['comparison_icon'], 'icon-default' );
			$icon_html .= Fns::icons_manager( $this->archive_ajax_data['comparison_icon_added'], 'added-icon' );
		}

		return $icon_html;
	}

	/**
	 * Wishlist icon
	 *
	 * @param string $icon_html Icon Html.
	 *
	 * @return string
	 */
	public function wishlist_icon( $icon_html ) {
		if ( ! empty( $this->archive_ajax_data['wishlist_icon'] ) ) {
			$icon_html  = Fns::icons_manager( $this->archive_ajax_data['wishlist_icon'], 'icon-default' );
			$icon_html .= Fns::icons_manager( $this->archive_ajax_data['wishlist_icon_added'], 'added-icon' );
		}

		return $icon_html;
	}

	/**
	 * Quick Checkout icon
	 *
	 * @param string $icon_html Icon Html.
	 *
	 * @return string
	 */
	public function quick_checkout_icon( $icon_html ) {
		if ( ! empty( $this->archive_ajax_data['quick_checkout_icon'] ) ) {
			$icon_html = Fns::icons_manager( $this->archive_ajax_data['quick_checkout_icon'], 'icon-default' );
		}

		return $icon_html;
	}

	/**
	 * Quick Checkout Parent Classes.
	 *
	 * @param array $classes Parent Classes.
	 * @return array
	 */
	public function quick_checkout_parent_classes( $classes ) {
		 $has_text = $this->archive_ajax_data['quick_checkout_text'] ?? false;
		 $key      = array_search( 'button', $classes, true );

		if ( false !== $key ) {
			unset( $classes[ $key ] );
		}

		 $classes[] = $has_text ? 'has-text' : 'no-text';

		 return $classes;
	}

	/**
	 * Quick Checkout Button Text.
	 *
	 * @param string $button_text Button text.
	 * @return string
	 */
	public function quick_checkout_btn_txt( $button_text ) {
		$text = $this->archive_ajax_data['quick_checkout_text'] ?? false;

		if ( ! $text ) {
			$button_text = '';
		}

		return $button_text;
	}
}
