<?php
/**
 * Ajax Actions Class.
 *
 * @package RadiusTheme\SB
 */

namespace RadiusTheme\SBPRO\Controllers\Frontend\Ajax;

use WC_Product_Query;
use RadiusTheme\SB\Helpers\Fns;
use RadiusTheme\SB\Models\QueryArgs;
use RadiusTheme\SBPRO\Traits\SingletonTrait;
use RadiusTheme\SB\Elementor\Render\Render;
use RadiusTheme\SB\Controllers\Hooks\ActionHooks;
use RadiusTheme\SBPRO\Controllers\Hooks\ElementorHooks;

// Do not allow directly accessing this file.
if ( ! defined( 'ABSPATH' ) ) {
	exit( 'This script cannot be accessed directly.' );
}

/**
 * Ajax Actions Class.
 */
class AjaxActions {

	/**
	 * Singleton.
	 */
	use SingletonTrait;

	/**
	 * Ajax data.
	 *
	 * @var array
	 */
	public $ajax_data;

	/**
	 * Render class.
	 *
	 * @var array
	 */
	public $render;

	/**
	 * Class Constructor.
	 *
	 * @return void
	 */
	private function __construct() {
		add_action( 'wp_ajax_rtsb_ajax_actions', [ $this, 'response' ] );
		add_action( 'wp_ajax_nopriv_rtsb_ajax_actions', [ $this, 'response' ] );
	}

	/**
	 * Ajax Response.
	 *
	 * @return void
	 */
	public function response() {
		$msg   = null;
		$data  = null;
		$error = false;

		if ( ! Fns::verify_nonce() ) {
			wp_send_json_error();
		}

		$this->ajax_data = json_decode( wp_unslash( $_REQUEST['ajaxData'] ), true );
		$this->render    = Render::instance();
		$layout          = $this->ajax_data['layout'];
		$template        = $this->ajax_data['template'];
		$is_carousel     = preg_match( '/slider/', $layout );
		$is_special      = 'special' === $this->ajax_data['ajax_filter_type'];
		$custom_tabs     = $this->ajax_data['custom_tabs'];
		$queried_tax     = ! empty( $this->ajax_data['queried_tax'] ) ? esc_html( $this->ajax_data['queried_tax'] ) : null;
		$queried_term    = ! empty( $this->ajax_data['queried_term'] ) ? esc_html( $this->ajax_data['queried_term'] ) : null;
		// Query Args.
		$args = ( new QueryArgs() )->buildArgs( $this->ajax_data, 'product', $is_carousel );

		if ( ! empty( $queried_tax ) && ! empty( $queried_term ) ) {
			$args['tax_query'][] = [
				'taxonomy' => $queried_tax,
				'field'    => 'term_id',
				'terms'    => $queried_term,
			];
		}

		if ( is_front_page() ) {
			// phpcs:ignore WordPress.Security.NonceVerification.Recommended
			$paged = ! empty( $_REQUEST['page'] ) ? absint( $_REQUEST['page'] ) : 2;
		} else {
			// phpcs:ignore WordPress.Security.NonceVerification.Recommended
			$paged = ! empty( $_REQUEST['paged'] ) ? absint( $_REQUEST['paged'] ) : 2;
		}

		$limit          = ( ( empty( $this->ajax_data['limit'] ) || '-1' === $this->ajax_data['limit'] ) ? 10000000 : absint( $this->ajax_data['limit'] ) );
		$posts_per_page = ( ! empty( $this->ajax_data['posts_per_page'] ) ? absint( $this->ajax_data['posts_per_page'] ) : $limit );

		if ( $posts_per_page > $limit ) {
			$posts_per_page = $limit;
		}

		$args['limit'] = $posts_per_page;
		$offset        = $posts_per_page * ( (int) $paged - 1 );
		$args['page']  = $paged;

		if ( absint( $args['limit'] ) > $limit - $offset ) {
			$args['limit']  = $limit - $offset;
			$args['offset'] = $offset;
		}

		$temp_args = $args;

		$action_taxonomy = ! empty( $_REQUEST['taxonomy'] ) ? sanitize_text_field( wp_unslash( $_REQUEST['taxonomy'] ) ) : null; // phpcs:ignore WordPress.Security.NonceVerification.Recommended
		$action_term     = ! empty( $_REQUEST['term'] ) ? ( 'all' === $_REQUEST['term'] ? 'all' : absint( $_REQUEST['term'] ) ) : 0; // phpcs:ignore WordPress.Security.NonceVerification.Recommended

		if ( $action_taxonomy && $action_term && 'all' !== $action_term ) {
			if ( 'product_attr' === $action_taxonomy && $action_term > 0 ) {
				$args['product_attribute_id'] = [ $action_term ];
			} else {
				switch ( $action_taxonomy ) {
					case 'product_cat':
						$action_taxonomy = 'product_category_id';
						break;
					case 'product_tag':
						$action_taxonomy = 'product_tag_id';
						break;
					case 'product_brand':
						$action_taxonomy = 'product_brand_id';
						break;
					default:
						$action_taxonomy = 'product_cat_id';

				}
				$args[ $action_taxonomy ] = [ $action_term ];
			}
		} else {
			unset( $args );
			$args = $temp_args;
		}

		/**
		 * Before Product query hook.
		 *
		 * @hooked RadiusTheme\SB\Controllers\Hooks\ActionHooks::modify_wc_query_args 10
		 */
		do_action( 'rtsb/elements/render/before_query', $this->ajax_data, $args );

		if ( $is_special && ! empty( $custom_tabs ) ) {
			$action_term = ! empty( $_REQUEST['term'] ) ? ( 'all' === $_REQUEST['term'] ? 'all' : sanitize_text_field( wp_unslash( $_REQUEST['term'] ) ) ) : 'recent'; // phpcs:ignore WordPress.Security.NonceVerification.Recommended

			switch ( $action_term ) {
				case 'featured':
					$args['featured'] = true;
					break;

				case 'best-selling':
					$args['meta_key'] = 'total_sales'; // phpcs:ignore WordPress.DB.SlowDBQuery.slow_db_query_meta_key
					$args['orderby']  = 'meta_value_num';
					break;

				case 'sale':
					// phpcs:ignore WordPress.Security.NonceVerification.Recommended
					$post_in         = ! empty( $_REQUEST['post_in'] ) ? array_map( 'absint', $_REQUEST['post_in'] ) : [ 0 ];
					$args['include'] = array_merge( $post_in, wc_get_product_ids_on_sale() );
					$args['orderby'] = 'title';
					break;

				case 'top-rated':
					$args['meta_key'] = '_wc_average_rating'; // phpcs:ignore WordPress.DB.SlowDBQuery.slow_db_query_meta_key
					$args['orderby']  = 'meta_value_num';
					break;

				default:
					$args['orderby'] = 'date';
					break;
			}
		}

		// Query.
		$product_query = new WC_Product_Query( $args );
		$products      = ! empty( $this->ajax_data['pagination'] ) ? $product_query->get_products()->products : $product_query->get_products();

		if ( ! empty( $products ) ) {
			foreach ( $products as $product ) {
				$GLOBALS['product'] = $product;

				// Badge module support.
				do_action( 'rtsb_before_product_template_render' );

				// Loop arg.
				$arg = $this->render->arg_dataset( $this->ajax_data, $product, $this->ajax_data['lazy_load'] );

				// Fix icons.
				$this->fix_icons();

				// Image gallery.
				$this->image_gallery_support();

				// Load Template.
				$data .= Fns::load_template( $template . $layout, $arg, true );

				unset( $GLOBALS['product'] );
			}
		} else {
			$data .= '<div class="rtsb-notice woocommerce-no-products-found"><div class="woocommerce-info">' . esc_html__( 'No products were found matching your selection.', 'shopbuilder-pro' ) . '</div></div>';
		}

		wp_reset_postdata();

		/**
		 * After Product query hook.
		 *
		 * @hooked RadiusTheme\SB\Controllers\Hooks\ActionHooks::reset_wc_query_args 10
		 */
		do_action( 'rtsb/elements/render/after_query', $this->ajax_data, $args );

		wp_send_json(
			[
				'msg'   => $msg,
				'data'  => $data,
				'error' => $error,
			]
		);
	}

	/**
	 * Update icons.
	 *
	 * @return void
	 */
	public function fix_icons() {
		if ( in_array( 'quick_view', $this->ajax_data['visibility'], true ) ) {
			add_filter( 'rtsb/module/quick_view/icon_html', [ $this, 'quick_view_icon' ] );
		}

		if ( in_array( 'compare', $this->ajax_data['visibility'], true ) ) {
			add_filter( 'rtsb/module/compare/icon_html', [ $this, 'compare_icon' ] );
		}

		if ( in_array( 'wishlist', $this->ajax_data['visibility'], true ) ) {
			add_filter( 'rtsb/module/wishlist/icon_html', [ $this, 'wishlist_icon' ] );
		}

		if ( in_array( 'quick_checkout', $this->ajax_data['visibility'], true ) ) {
			add_filter( 'rtsb/quick_checkout/button/icon', [ $this, 'quick_checkout_icon' ] );
			add_action( 'rtsb/quick_checkout/parent/class', [ $this, 'quick_checkout_parent_classes' ], 15 );
			add_action( 'rtsb/quick_checkout/button/text', [ $this, 'quick_checkout_btn_txt' ], 15 );
		}
	}

	/**
	 * Gallery image support.
	 *
	 * @return void
	 */
	public function image_gallery_support() {
		remove_action( 'rtsb/shop/product/image', [ ActionHooks::CLASS, 'render_image' ] );
		add_action( 'rtsb/shop/product/image', [ ElementorHooks::class, 'render_image' ], 10, 2 );
	}

	/**
	 * Quick View icon
	 *
	 * @param string $icon_html Icon Html.
	 *
	 * @return string
	 */
	public function quick_view_icon( $icon_html ) {
		if ( ! empty( $this->ajax_data['quick_view_icon'] ) ) {
			$icon_html = Fns::icons_manager( $this->ajax_data['quick_view_icon'], 'icon-default' );
		}

		return $icon_html;
	}

	/**
	 * Compare icon
	 *
	 * @param string $icon_html Icon Html.
	 *
	 * @return string
	 */
	public function compare_icon( $icon_html ) {
		if ( ! empty( $this->ajax_data['comparison_icon'] ) ) {
			$icon_html  = Fns::icons_manager( $this->ajax_data['comparison_icon'], 'icon-default' );
			$icon_html .= Fns::icons_manager( $this->ajax_data['comparison_icon_added'], 'added-icon' );
		}

		return $icon_html;
	}

	/**
	 * Wishlist icon
	 *
	 * @param string $icon_html Icon Html.
	 *
	 * @return string
	 */
	public function wishlist_icon( $icon_html ) {
		if ( ! empty( $this->ajax_data['wishlist_icon'] ) ) {
			$icon_html  = Fns::icons_manager( $this->ajax_data['wishlist_icon'], 'icon-default' );
			$icon_html .= Fns::icons_manager( $this->ajax_data['wishlist_icon_added'], 'added-icon' );
		}

		return $icon_html;
	}

	/**
	 * Quick Checkout icon
	 *
	 * @param string $icon_html Icon Html.
	 *
	 * @return string
	 */
	public function quick_checkout_icon( $icon_html ) {
		if ( ! empty( $this->ajax_data['quick_checkout_icon'] ) ) {
			$icon_html = Fns::icons_manager( $this->ajax_data['quick_checkout_icon'], 'icon-default' );
		}

		return $icon_html;
	}

	/**
	 * Quick Checkout Parent Classes.
	 *
	 * @param array $classes Parent Classes.
	 * @return array
	 */
	public function quick_checkout_parent_classes( $classes ) {
		$has_text = $this->ajax_data['quick_checkout_text'] ?? false;
		$key      = array_search( 'button', $classes, true );

		if ( false !== $key ) {
			unset( $classes[ $key ] );
		}

		$classes[] = $has_text ? 'has-text' : 'no-text';

		return $classes;
	}

	/**
	 * Quick Checkout Button Text.
	 *
	 * @param string $button_text Button text.
	 * @return string
	 */
	public function quick_checkout_btn_txt( $button_text ) {
		$text = $this->ajax_data['quick_checkout_text'] ?? false;

		if ( ! $text ) {
			$button_text = '';
		}

		return $button_text;
	}
}
